/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Instant;
import java.util.Base64;
import java.util.Objects;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.PreEncodedHttpField;
import org.eclipse.jetty.http.QuotedCSV;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.resource.Resource;

public final class EtagUtils {
    public static final String ETAG_SEPARATOR = System.getProperty(EtagUtils.class.getName() + ".ETAG_SEPARATOR", "--");

    private EtagUtils() {
    }

    public static HttpField createWeakEtagField(Resource resource2) {
        return EtagUtils.createWeakEtagField(resource2, null);
    }

    public static HttpField createWeakEtagField(Resource resource2, String etagSuffix) {
        Path path = resource2.getPath();
        if (path == null) {
            return null;
        }
        String etagValue = EtagUtils.computeWeakEtag(path, etagSuffix);
        if (etagValue != null) {
            return new PreEncodedHttpField(HttpHeader.ETAG, etagValue);
        }
        return null;
    }

    public static String computeWeakEtag(Resource resource2) {
        return EtagUtils.computeWeakEtag(resource2, null);
    }

    public static String computeWeakEtag(Resource resource2, String etagSuffix) {
        if (resource2 == null || !resource2.exists() || resource2.isDirectory()) {
            return null;
        }
        Path path = resource2.getPath();
        if (path != null) {
            return EtagUtils.computeWeakEtag(path, etagSuffix);
        }
        String name = resource2.getName();
        Instant lastModified = resource2.lastModified();
        long size = resource2.length();
        return EtagUtils.computeWeakEtag(name, lastModified, size, etagSuffix);
    }

    public static String computeWeakEtag(Path path) {
        return EtagUtils.computeWeakEtag(path, null);
    }

    public static String computeWeakEtag(Path path, String suffix) {
        if (path == null) {
            return null;
        }
        String name = path.toAbsolutePath().toString();
        Instant lastModified = Instant.EPOCH;
        try {
            lastModified = Files.getLastModifiedTime(path, new LinkOption[0]).toInstant();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        long size = -1L;
        try {
            size = Files.size(path);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return EtagUtils.computeWeakEtag(name, lastModified, size, suffix);
    }

    private static String computeWeakEtag(String name, Instant lastModified, long size, String etagSuffix) {
        StringBuilder b = new StringBuilder(32);
        b.append("W/\"");
        Base64.Encoder encoder = Base64.getEncoder().withoutPadding();
        long lastModifiedMillis = lastModified.toEpochMilli();
        int lhash = Objects.hashCode(name);
        b.append(encoder.encodeToString(EtagUtils.longToBytes(lastModifiedMillis ^ (long)lhash)));
        b.append(encoder.encodeToString(EtagUtils.longToBytes(size ^ (long)lhash)));
        if (etagSuffix != null) {
            b.append(etagSuffix);
        }
        b.append('\"');
        return b.toString();
    }

    public static String rewriteWithSuffix(String etag, String newSuffix) {
        StringBuilder ret = new StringBuilder();
        boolean weak = etag.startsWith("W/");
        int start = 0;
        if (etag.startsWith("W/")) {
            weak = true;
            start = 2;
        }
        while (etag.charAt(start) == '\"') {
            ++start;
        }
        int end = etag.length();
        while (etag.charAt(end - 1) == '\"') {
            --end;
        }
        int suffixIdx = etag.lastIndexOf(45, end);
        if (suffixIdx - 1 >= 0 && suffixIdx - 1 >= start && etag.charAt(suffixIdx - 1) == '-') {
            end = suffixIdx - 1;
        } else if (suffixIdx >= 0 && suffixIdx >= start) {
            end = suffixIdx;
        }
        if (weak) {
            ret.append("W/");
        }
        ret.append('\"');
        ret.append(etag, start, end);
        ret.append(newSuffix);
        ret.append('\"');
        return ret.toString();
    }

    public static boolean isWeak(String etag) {
        return etag.startsWith("W/");
    }

    public static boolean isStrong(String etag) {
        return !EtagUtils.isWeak(etag);
    }

    public static boolean matches(String etag, String etagWithOptionalSuffix) {
        if (etag.equals(etagWithOptionalSuffix)) {
            return true;
        }
        if (StringUtil.isEmpty(ETAG_SEPARATOR)) {
            return false;
        }
        boolean etagQuoted = etag.endsWith("\"");
        boolean etagSuffixQuoted = etagWithOptionalSuffix.endsWith("\"");
        int separator = etagWithOptionalSuffix.lastIndexOf(ETAG_SEPARATOR);
        if (etagQuoted == etagSuffixQuoted) {
            return separator > 0 && etag.regionMatches(0, etagWithOptionalSuffix, 0, separator);
        }
        if (etagWithOptionalSuffix.startsWith("W/") || etag.startsWith("W/")) {
            return false;
        }
        etag = etagQuoted ? QuotedCSV.unquote(etag) : etag;
        etagWithOptionalSuffix = etagSuffixQuoted ? QuotedCSV.unquote(etagWithOptionalSuffix) : etagWithOptionalSuffix;
        separator = etagWithOptionalSuffix.lastIndexOf(ETAG_SEPARATOR);
        if (separator > 0) {
            return etag.regionMatches(0, etagWithOptionalSuffix, 0, separator);
        }
        return Objects.equals(etag, etagWithOptionalSuffix);
    }

    private static byte[] longToBytes(long value) {
        byte[] result = new byte[8];
        for (int i2 = 7; i2 >= 0; --i2) {
            result[i2] = (byte)(value & 0xFFL);
            value >>= 8;
        }
        return result;
    }
}

