//    Copyright (c) 2014 - 2015 payu@india.com
//
//    Permission is hereby granted, free of charge, to any person obtaining a copy
//    of this software and associated documentation files (the "Software"), to deal
//    in the Software without restriction, including without limitation the rights
//    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//    copies of the Software, and to permit persons to whom the Software is
//    furnished to do so, subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be included in
//    all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//    THE SOFTWARE.

package com.payu.india.PostParams;

import android.content.Context;
import android.text.TextUtils;

import com.payu.india.BuildConfig;
import com.payu.india.Model.PaymentParams;
import com.payu.india.Model.PostData;
import com.payu.india.Payu.Payu;
import com.payu.india.Payu.PayuConstants;
import com.payu.india.Payu.PayuErrors;
import com.payu.india.Payu.PayuUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by franklin on 27/07/15.
 * Making one common class which takes all the inputs required to make a payment.
 * Inorder to create an object of this kind user needs to provide two inputs.
 */
public class PaymentPostParams extends PayuUtils {

    private PaymentParams mPaymentParams;
    private String mPaymentMode;
    private Context applicationContext;

    /**
     * Not allowing the user to create empty object.
     */
    private PaymentPostParams() {
    }

    /**
     * PaymentPostParams will accept object of {@link PaymentParams} and a payment mode {@link com.payu.india.Payu.PayuConstants#NB}
     *
     * @param paymentParams
     */
    public PaymentPostParams(PaymentParams paymentParams, String paymentMode) throws Exception {
        this.mPaymentParams = paymentParams;
        this.mPaymentMode = paymentMode;
        if (null == Payu.getInstance()) {
            // throw exception;
            throw new Exception("Application context not found please set your application context by adding Payu.setInstance(this) from your activity ");
        } else {
            this.applicationContext = Payu.getInstance().getCallingAppContext();
        }
//        applicationContext = (Payu.getInstance()==null ? null : Payu.getInstance().getCallingAppContext());
    }

    public PostData getPaymentPostParams() {
        PostData postData = new PostData();
        StringBuffer post = new StringBuffer();


        // lets validate the pg using payment mode.
        if (!PayuConstants.PG_SET.contains(this.mPaymentMode) && !mPaymentMode.equalsIgnoreCase(PayuConstants.LAZYPAY)) { // invalid pg value.
            return getReturnData(PayuErrors.INVALID_PG);
        }

        // lets set the default Parameters
        // device_type is mandatory, With out this flag SDK or CB can not receive payuPostData (ie: onPayuSuccess or onPayuFailure js callback from payu server)
        post.append(concatParams(PayuConstants.DEVICE_TYPE, "1"));
        // Adding udid and imei
        post.append(concatParams(PayuConstants.PAYU_UDID, getUdid()));
        post.append(concatParams(PayuConstants.PAYU_IMEI, getImei()));

        // lets begin with the mandatory default params.
        // TODO apply the validation according to the pg, payment mode!
        for (int i = 0; i < PayuConstants.PAYMENT_PARAMS_ARRAY.length; i++) {
            switch (PayuConstants.PAYMENT_PARAMS_ARRAY[i]) {
//                PayuConstants.KEY, PayuConstants.TXNID, PayuConstants.AMOUNT, PayuConstants.PRODUCT_INFO, PayuConstants.FIRST_NAME, PayuConstants.EMAIL, PayuConstants.SURL, PayuConstants.FURL, PayuConstants.HASH
                case PayuConstants.KEY: // TODO add validation for key
                    if (mPaymentParams.getKey() == null || mPaymentParams.getKey().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_KEY_IS_MISSING);
                    post.append(concatParams(PayuConstants.KEY, mPaymentParams.getKey()));
                    break;
                case PayuConstants.TXNID: // TODO add validation for txnid
                    if (mPaymentParams.getTxnId() == null || mPaymentParams.getTxnId().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_TXNID_IS_MISSING);
                    post.append(concatParams(PayuConstants.TXNID, mPaymentParams.getTxnId()));
                    break;
                case PayuConstants.AMOUNT: // validation for amount
                    Double amount = 0.0;
                    try { // this will take care of null check also!
                        amount = mPaymentParams != null ? Double.parseDouble(mPaymentParams.getAmount()) : 0.0;
                    } catch (NumberFormatException e) {
                        return getReturnData(PayuErrors.NUMBER_FORMAT_EXCEPTION, PayuErrors.INVALID_AMOUNT);
                    } catch (NullPointerException e) {
                        return getReturnData(PayuErrors.INVALID_AMOUNT_EXCEPTION, PayuErrors.INVALID_AMOUNT);
                    }
                    post.append(concatParams(PayuConstants.AMOUNT, mPaymentParams.getAmount()));
                    break;
                case PayuConstants.PRODUCT_INFO: // TODO add validation for product info
                    if (mPaymentParams.getProductInfo() == null || mPaymentParams.getProductInfo().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_PRODUCT_INFO_IS_MISSING);
                    post.append(concatParams(PayuConstants.PRODUCT_INFO, mPaymentParams.getProductInfo()));
                    break;
                case PayuConstants.FIRST_NAME: // TODO add validation for first name
                    if (mPaymentParams.getFirstName() == null) // empty string is allowed
                        return getReturnData(PayuErrors.MANDATORY_PARAM_FIRST_NAME_IS_MISSING);
                    post.append(concatParams(PayuConstants.FIRST_NAME, mPaymentParams.getFirstName()));
                    break;
                case PayuConstants.EMAIL: // TODO add validation for email
                    if (mPaymentParams.getEmail() == null)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_EMAIL_IS_MISSING);
                    post.append(concatParams(PayuConstants.EMAIL, mPaymentParams.getEmail()));
                    break;
                case PayuConstants.SURL: // TODO add validation for SURL
                    if (mPaymentParams.getSurl() == null || mPaymentParams.getSurl().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_SURL_IS_MISSING);
                    // we gotta encode surl
                    try {
                        post.append(PayuConstants.SURL + "=").append(URLEncoder.encode(mPaymentParams.getSurl(), "UTF-8")).append("&");
                    } catch (UnsupportedEncodingException e) {
                        return getReturnData(PayuErrors.UN_SUPPORTED_ENCODING_EXCEPTION, PayuConstants.SURL + PayuErrors.INVALID_URL);
                    }
                    break;
                case PayuConstants.FURL: // TODO add validation for FURL
                    if (mPaymentParams.getFurl() == null || mPaymentParams.getFurl().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_FURL_IS_MISSING);
                    // we gotta encode furl
                    try {
                        post.append(PayuConstants.FURL + "=").append(URLEncoder.encode(mPaymentParams.getFurl(), "UTF-8")).append("&");
                    } catch (UnsupportedEncodingException e) {
                        return getReturnData(PayuErrors.UN_SUPPORTED_ENCODING_EXCEPTION, PayuConstants.FURL + PayuErrors.INVALID_URL);
                    }
                    break;
                case PayuConstants.HASH: // TODO add validation for Hash
                    if (mPaymentParams.getHash() == null || mPaymentParams.getHash().length() < 1)
                        return getReturnData(PayuErrors.MANDATORY_PARAM_HASH_IS_MISSING);
                    post.append(concatParams(PayuConstants.HASH, mPaymentParams.getHash()));
                    break;
                case PayuConstants.UDF1:
                    if (mPaymentParams.getUdf1() == null)
                        return getReturnData(PayuErrors.INVALID_UDF1);
                    post.append(concatParams(PayuConstants.UDF1, mPaymentParams.getUdf1()));
                    break;
                case PayuConstants.UDF2: // TODO add validation for UDF2
                    if (mPaymentParams.getUdf2() == null)
                        return getReturnData(PayuErrors.INVALID_UDF2);
                    post.append(concatParams(PayuConstants.UDF2, mPaymentParams.getUdf2()));
                    break;
                case PayuConstants.UDF3: // TODO add validation for UDF3
                    if (mPaymentParams.getUdf3() == null)
                        return getReturnData(PayuErrors.INVALID_UDF3);
                    post.append(concatParams(PayuConstants.UDF3, mPaymentParams.getUdf3()));
                    break;
                case PayuConstants.UDF4: // TODO add validation for UDF4
                    if (mPaymentParams.getUdf4() == null)
                        return getReturnData(PayuErrors.INVALID_UDF4);
                    post.append(concatParams(PayuConstants.UDF4, mPaymentParams.getUdf4()));
                    break;
                case PayuConstants.UDF5: // TODO add validation for UDF5
                    if (mPaymentParams.getUdf5() == null)
                        return getReturnData(PayuErrors.INVALID_UDF5);
                    post.append(concatParams(PayuConstants.UDF5, mPaymentParams.getUdf5()));
                    break;
            }
        }

        if (mPaymentParams.getPhone() != null) { // TODO add phone number validation
            post.append(concatParams(PayuConstants.PHONE, mPaymentParams.getPhone()));
        }

        // optional fields.
        post.append(mPaymentParams.getOfferKey() != null ? concatParams(PayuConstants.OFFER_KEY, mPaymentParams.getOfferKey()) : "");
        post.append(mPaymentParams.getLastName() != null ? concatParams(PayuConstants.LASTNAME, mPaymentParams.getLastName()) : "");
        post.append(mPaymentParams.getAddress1() != null ? concatParams(PayuConstants.ADDRESS1, mPaymentParams.getAddress1()) : "");
        post.append(mPaymentParams.getAddress2() != null ? concatParams(PayuConstants.ADDRESS2, mPaymentParams.getAddress2()) : "");
        post.append(mPaymentParams.getCity() != null ? concatParams(PayuConstants.CITY, mPaymentParams.getCity()) : "");
        post.append(mPaymentParams.getState() != null ? concatParams(PayuConstants.STATE, mPaymentParams.getState()) : "");
        post.append(mPaymentParams.getCountry() != null ? concatParams(PayuConstants.COUNTRY, mPaymentParams.getCountry()) : "");
        post.append(mPaymentParams.getZipCode() != null ? concatParams(PayuConstants.ZIPCODE, mPaymentParams.getZipCode()) : "");
        post.append(mPaymentParams.getCodUrl() != null ? concatParams(PayuConstants.CODURL, mPaymentParams.getCodUrl()) : "");
        post.append(mPaymentParams.getDropCategory() != null ? concatParams(PayuConstants.DROP_CATEGORY, mPaymentParams.getDropCategory()) : "");
        post.append(mPaymentParams.getEnforcePayMethod() != null ? concatParams(PayuConstants.ENFORCE_PAYMETHOD, mPaymentParams.getEnforcePayMethod()) : "");
        post.append(mPaymentParams.getCustomNote() != null ? concatParams(PayuConstants.CUSTOM_NOTE, mPaymentParams.getCustomNote()) : "");
        post.append(mPaymentParams.getNoteCategory() != null ? concatParams(PayuConstants.NOTE_CATEGORY, mPaymentParams.getNoteCategory()) : "");
        post.append(mPaymentParams.getShippingFirstName() != null ? concatParams(PayuConstants.SHIPPING_FIRSTNAME, mPaymentParams.getShippingFirstName()) : "");
        post.append(mPaymentParams.getShippingLastName() != null ? concatParams(PayuConstants.SHIPPING_LASTNAME, mPaymentParams.getShippingLastName()) : "");
        post.append(mPaymentParams.getShippingAddress1() != null ? concatParams(PayuConstants.SHIPPING_ADDRESS1, mPaymentParams.getShippingAddress1()) : "");
        post.append(mPaymentParams.getShippingAddress2() != null ? concatParams(PayuConstants.SHIPPING_ADDRESS2, mPaymentParams.getShippingAddress2()) : "");
        post.append(mPaymentParams.getShippingCity() != null ? concatParams(PayuConstants.SHIPPING_CITY, mPaymentParams.getShippingCity()) : "");
        post.append(mPaymentParams.getShippingState() != null ? concatParams(PayuConstants.SHIPPING_STATE, mPaymentParams.getShippingState()) : "");
        post.append(mPaymentParams.getShippingCounty() != null ? concatParams(PayuConstants.SHIPPING_CONTRY, mPaymentParams.getShippingCounty()) : "");
        post.append(mPaymentParams.getShippingZipCode() != null ? concatParams(PayuConstants.SHIPPING_ZIPCODE, mPaymentParams.getShippingZipCode()) : "");
        post.append(mPaymentParams.getShippingPhone() != null ? concatParams(PayuConstants.SHIPPING_PHONE, mPaymentParams.getShippingPhone()) : "");

        // if it is not emi, and subvention amount is passed, just send it server.
        if (!mPaymentMode.contentEquals(PayuConstants.EMI)) {
            post.append(mPaymentParams.getSubventionAmount() != null ? concatParams(PayuConstants.SUBVENTION_AMOUNT, mPaymentParams.getSubventionAmount()) : "");
        }


        // Lets  make the analytics call here.
        analizingTransaction();

        // lets setup the user inputs.
        switch (mPaymentMode) {
            case PayuConstants.CC:  //credit/debit/stored card/one click card
                post.append(concatParams(PayuConstants.PG, PayuConstants.CC));
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.CC));

                if (null != this.mPaymentParams.getCardNumber() && validateCardNumber(this.mPaymentParams.getCardNumber())) { // card payment
                    // okay its a valid card number
                    post.append(concatParams(PayuConstants.CC_NUM, this.mPaymentParams.getCardNumber()));
                    // if card number is not smae then validate cvv and expiry.
                    if (!getIssuer(this.mPaymentParams.getCardNumber()).contentEquals(PayuConstants.SMAE)) {
                        if (validateCvv(this.mPaymentParams.getCardNumber(), this.mPaymentParams.getCvv())) {
                            post.append(concatParams(PayuConstants.C_CVV, this.mPaymentParams.getCvv()));
                        } else {
                            return getReturnData(PayuErrors.INVALID_CVV_EXCEPTION, PayuErrors.INVALID_CVV);
                        }
                        try {
                            if (validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                post.append(concatParams(PayuConstants.CC_EXP_YR, this.mPaymentParams.getExpiryYear()));
                                post.append(concatParams(PayuConstants.CC_EXP_MON, this.mPaymentParams.getExpiryMonth()));
                            } else {
                                return getReturnData(PayuErrors.CARD_EXPIRED_EXCEPTION, PayuErrors.CARD_EXPIRED);
                            }
                        } catch (NumberFormatException e) {
                            return getReturnData(PayuErrors.NUMBER_FORMAT_EXCEPTION, PayuErrors.CARD_EXPIRED); // todo wrong expiry format
                        } catch (Exception e) {
                            return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuErrors.CARD_EXPIRED);
                        }
                    } else { // some smae might have cvv, make sure that we have have added them in our post params.
                        if (validateCvv(this.mPaymentParams.getCardNumber(), this.mPaymentParams.getCvv())) {
                            post.append(concatParams(PayuConstants.C_CVV, this.mPaymentParams.getCvv()));
                        }
                        try {
                            if (validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                post.append(concatParams(PayuConstants.CC_EXP_YR, this.mPaymentParams.getExpiryYear()));
                                post.append(concatParams(PayuConstants.CC_EXP_MON, this.mPaymentParams.getExpiryMonth()));
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }

                    // if name on card is not given use default name on card as "PayuUser"
                    String nameOnCard = null != this.mPaymentParams.getNameOnCard() && this.mPaymentParams.getNameOnCard().trim().length() > 0 ? this.mPaymentParams.getNameOnCard() : "PayuUser";
                    // if card name is not given use name on card instead
                    String cardName = null != this.mPaymentParams.getCardName() ? this.mPaymentParams.getCardName() : nameOnCard;
                    post.append(concatParams(PayuConstants.CC_NAME, nameOnCard));
                    if (this.mPaymentParams.getStoreCard() == 1) {
                        if (this.mPaymentParams.getUserCredentials() != null) {
                            post.append(concatParams(PayuConstants.CARD_NAME, cardName));
                            post.append(this.mPaymentParams.getUserCredentials() != null ? concatParams(PayuConstants.USER_CREDENTIALS, this.mPaymentParams.getUserCredentials()) : "");
                            post.append(this.mPaymentParams.getStoreCard() == 1 ? concatParams(PayuConstants.STORED_CARD, "" + this.mPaymentParams.getStoreCard()) : "");

                            post.append(this.mPaymentParams.getEnableOneClickPayment() == 1 ? concatParams(PayuConstants.ONE_CLICK_CHECKOUT, "" + this.mPaymentParams.getEnableOneClickPayment()) : "");

                        } else {
                            return getReturnData(PayuErrors.USER_CREDENTIALS_NOT_FOUND);
                        }
                    }

                    // TODO add validation for store_card and user_credentials
                    // thats it we can return post Data
                    return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
                } else if (null != this.mPaymentParams.getCardToken()) {
                    // its stored card payment! we gotta verify user credentials
                    if (this.mPaymentParams.getUserCredentials() != null) {
                        post.append(concatParams(PayuConstants.USER_CREDENTIALS, this.mPaymentParams.getUserCredentials()));
                        post.append(concatParams(PayuConstants.STORE_CARD_TOKEN, this.mPaymentParams.getCardToken()));
                        if (this.mPaymentParams.getCardBin() != null) {
                            // here we have the card bin we can validate cvv, expiry
                            if (!getIssuer(this.mPaymentParams.getCardBin()).contentEquals(PayuConstants.SMAE)) {
                                if (this.mPaymentParams.getCvv() == null && null == this.mPaymentParams.getCardCvvMerchant()) {
                                    return getReturnData(PayuErrors.INVALID_CVV);
                                }
                                if (!validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                    return getReturnData(PayuErrors.CARD_EXPIRED);
                                }
                            }
                        }
                        if (this.mPaymentParams.getCardCvvMerchant() == null)
                            post.append(this.mPaymentParams.getCvv() != null ? concatParams(PayuConstants.C_CVV, this.mPaymentParams.getCvv()) : concatParams(PayuConstants.C_CVV, "123")); // its not necessary that all the stored cards should have a cvv && we dont have card number so no validation.
                        else
                            post.append(concatParams(PayuConstants.CARD_MERCHANT_PARAM, this.mPaymentParams.getCardCvvMerchant()));

                        post.append(this.mPaymentParams.getExpiryMonth() != null ? concatParams(PayuConstants.CC_EXP_MON, this.mPaymentParams.getExpiryMonth()) : concatParams(PayuConstants.CC_EXP_MON, "12"));
                        post.append(this.mPaymentParams.getExpiryYear() != null ? concatParams(PayuConstants.CC_EXP_YR, this.mPaymentParams.getExpiryYear()) : concatParams(PayuConstants.CC_EXP_MON, "2080"));

                        post.append(this.mPaymentParams.getNameOnCard() == null ? concatParams(PayuConstants.CC_NAME, "PayuUser") : concatParams(PayuConstants.CC_NAME, mPaymentParams.getNameOnCard()));

                        post.append(this.mPaymentParams.getEnableOneClickPayment() == 1 ? concatParams(PayuConstants.ONE_CLICK_CHECKOUT, "" + this.mPaymentParams.getEnableOneClickPayment()) : "");

                        // okey we have data
                        return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
                    } else {
                        return getReturnData(PayuErrors.USER_CREDENTIALS_NOT_FOUND_EXCEPTION, PayuErrors.USER_CREDENTIALS_MISSING);
                    }
                } else {
                    return getReturnData(PayuErrors.INVALID_CARD_NUMBER_EXCEPTION, PayuErrors.INVALID_CARD_NUMBER);
                }
            case PayuConstants.NB: // netbanking
                if (this.mPaymentParams.getBankCode() != null && this.mPaymentParams.getBankCode().length() > 1) { // assuming we have a valid bank code now.
                    post.append(concatParams(PayuConstants.PG, PayuConstants.NB));
                    post.append(concatParams(PayuConstants.BANK_CODE, this.mPaymentParams.getBankCode()));
                } else {
                    return getReturnData(PayuErrors.INVALID_BANKCODE_EXCEPTION, PayuErrors.INVALID_BANK_CODE);
                }
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.EMI: // emi
                if (this.mPaymentParams.getBankCode() != null && this.mPaymentParams.getBankCode().length() > 1) { // TODO: add proper validation for bankcode.
                    post.append(concatParams(PayuConstants.PG, PayuConstants.EMI));
                    post.append(concatParams(PayuConstants.BANK_CODE, this.mPaymentParams.getBankCode()));

                    // Subvention amount check
                    if (null != mPaymentParams.getSubventionAmount()) {
                        // number validation
                        Double subventionAmount = 0.0;
                        try { // this will take care of null, empty check also!
                            subventionAmount = Double.parseDouble(mPaymentParams.getSubventionAmount());
                        } catch (NumberFormatException e) {
                            return getReturnData(PayuErrors.NUMBER_FORMAT_EXCEPTION, PayuErrors.INVALID_SUBVENTION_AMOUNT_TYPE_1);
                        } catch (NullPointerException e) {
                            return getReturnData(PayuErrors.INVALID_SUBVENTION_AMOUNT_EXCEPTION, PayuErrors.INVALID_SUBVENTION_AMOUNT_TYPE_1);
                        }

                        // subvention amount validation formula:  subvention_amount <= amount - offer_amount
                        // sdk does not deal with offer amount
                        // subvention amount should not be negative and should not be greater than amount.
                        if (subventionAmount >= 0 && subventionAmount <= Double.parseDouble(mPaymentParams.getAmount())) {
                            post.append(concatParams(PayuConstants.SUBVENTION_AMOUNT, mPaymentParams.getSubventionAmount()));
                        } else {
                            return getReturnData(PayuErrors.INVALID_SUBVENTION_AMOUNT_EXCEPTION, PayuErrors.INVALID_SUBVENTION_AMOUNT_TYPE_2);
                        }
                    }

                    // lets validate card number
                    if (validateCardNumber("" + this.mPaymentParams.getCardNumber())) {
                        // okay its a valid card number
                        post.append(concatParams(PayuConstants.CC_NUM, "" + this.mPaymentParams.getCardNumber()));
                        // if card number is not smae then validate cvv and expiry.
                        if (!getIssuer("" + this.mPaymentParams.getCardNumber()).contentEquals(PayuConstants.SMAE)) {
                            if (validateCvv("" + this.mPaymentParams.getCardNumber(), "" + this.mPaymentParams.getCvv())) {
                                post.append(concatParams(PayuConstants.C_CVV, "" + this.mPaymentParams.getCvv()));
                            } else {
                                return getReturnData(PayuErrors.INVALID_CVV_EXCEPTION, PayuErrors.INVALID_CVV);
                            }
                            try {
                                if (validateExpiry(Integer.parseInt(this.mPaymentParams.getExpiryMonth()), Integer.parseInt(this.mPaymentParams.getExpiryYear()))) {
                                    post.append(concatParams(PayuConstants.CC_EXP_YR, "" + this.mPaymentParams.getExpiryYear()));
                                    post.append(concatParams(PayuConstants.CC_EXP_MON, "" + this.mPaymentParams.getExpiryMonth()));
                                } else {
                                    return getReturnData(PayuErrors.CARD_EXPIRED_EXCEPTION, PayuErrors.CARD_EXPIRED);
                                }
                            } catch (NumberFormatException e) {
                                return getReturnData(PayuErrors.NUMBER_FORMAT_EXCEPTION, PayuErrors.CARD_EXPIRED); // TODO add proper message cast exception
                            }
                        }

                        post.append(this.mPaymentParams.getNameOnCard() == null ? concatParams(PayuConstants.CC_NAME, "PayuUser") : concatParams(PayuConstants.CC_NAME, mPaymentParams.getNameOnCard()));
                        if (this.mPaymentParams.getStoreCard() == 1) {
                            if (this.mPaymentParams.getUserCredentials() != null) {
                                post.append(this.mPaymentParams.getCardName() == null ? concatParams(PayuConstants.CARD_NAME, "PayuUser") : concatParams(PayuConstants.NAME_ON_CARD, mPaymentParams.getCardName()));
                                post.append(this.mPaymentParams.getUserCredentials() != null ? concatParams(PayuConstants.USER_CREDENTIALS, this.mPaymentParams.getUserCredentials()) : "");
                                post.append(this.mPaymentParams.getStoreCard() == 1 ? concatParams(PayuConstants.STORED_CARD, "" + this.mPaymentParams.getStoreCard()) : "");
                            } else {
                                return getReturnData(PayuErrors.USER_CREDENTIALS_NOT_FOUND);
                            }
                        }
                        // TODO add validation for store_card and user_credentials
                        // thats it we can return post Data
                        return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
                    } else {
                        return getReturnData(PayuErrors.INVALID_CARD_NUMBER_EXCEPTION, PayuErrors.INVALID_CARD_NUMBER);
                    }
                } else {
                    return getReturnData(PayuErrors.INVALID_EMI_DETAILS);
                }
            case PayuConstants.CASH: // cash
                post.append(concatParams(PayuConstants.PG, PayuConstants.CASH)); // cash card
                // lets validate payment bank code
                if (this.mPaymentParams != null && this.mPaymentParams.getBankCode() != null && this.mPaymentParams.getBankCode().length() > 1) { // assuming we have a valid bank code now.
                    post.append(concatParams(PayuConstants.BANK_CODE, this.mPaymentParams.getBankCode()));
                } else {
                    return getReturnData(PayuErrors.INVALID_BANKCODE_EXCEPTION, PayuErrors.INVALID_BANK_CODE);
                }
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());
            case PayuConstants.PAYU_MONEY: // payu money.
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.PAYUW.toLowerCase()));
                post.append(concatParams(PayuConstants.PG, PayuConstants.WALLET));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());

            case PayuConstants.LAZYPAY: // payu money.
                post.append(concatParams(PayuConstants.BANK_CODE, ""));
                post.append(concatParams(PayuConstants.PG, ""));
                try {
                    if (mPaymentParams.getNotifyURL() != null && !mPaymentParams.getNotifyURL().equalsIgnoreCase("")) {
                        post.append(concatParams(PayuConstants.NOTIFY_URL, URLEncoder.encode(mPaymentParams.getNotifyURL(), "UTF-8")));
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
                post.append(concatParams(PayuConstants.ENFORCE_PAYMETHOD, PayuConstants.LAZYPAY.toUpperCase()));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));

            case PayuConstants.SAMSUNG_PAY:
                if (this.mPaymentParams.getUserCredentials() != null) {
                    post.append(concatParams(PayuConstants.USER_CREDENTIALS, this.mPaymentParams.getUserCredentials()));
                } else {
                    return getReturnData(PayuErrors.USER_CREDENTIALS_NOT_FOUND);
                }
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.SAMSUNG_PAY.toLowerCase()));
                post.append(concatParams(PayuConstants.PG, PayuConstants.SAMSUNG_PAY));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());
            case PayuConstants.UPI:
                // Virtual address Check (vpa check)
                // 1)Vpa length should be less than or equal to 50
                // 2)It can be alphanumeric and can contain a dot(.).
                // 3)It should contain a @

                // NULL Check for Virtual address

                if (mPaymentParams.getVpa() == null) {
                    return getReturnData(PayuErrors.INVALID_VPA, PayuErrors.VPA_NULL);
                } else if (mPaymentParams.getVpa().trim().length() == 0) { // Vpa should not be empty
                    return getReturnData(PayuErrors.INVALID_VPA, PayuErrors.VPA_EMPTY);
                } else if (mPaymentParams.getVpa().trim().length() > PayuConstants.MAX_VPA_SIZE) {
                    return getReturnData(PayuErrors.INVALID_VPA, PayuErrors.VPA_GREATER); // Vpa size should not be greater than 50
                } else {
                    String userVirtualAddress = mPaymentParams.getVpa().trim();
                    // Pattern pattern = Pattern.compile("^([A-Za-z0-9\\.])+\\@[A-Za-z0-9]+$"); //Vpa name can be alphanumeric and can contain a dot(.) preceded by @. After '@' only alphanumeric is allowed.
                    Pattern pattern = Pattern.compile(".+@.+"); // VPA Check: non empty + @ + non empty
                    Matcher matcher = pattern.matcher(userVirtualAddress);
                    if (!matcher.matches()) {
                        return getReturnData(PayuErrors.INVALID_VPA, PayuErrors.INVALID_VPA_MESSAGE);
                    }
                }
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.UPI.toLowerCase()));
                post.append(concatParams(PayuConstants.PG, PayuConstants.UPI));
                post.append(concatParams(PayuConstants.VPA, mPaymentParams.getVpa().trim()));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
            case PayuConstants.TEZ:
                post.append(concatParams(PayuConstants.BANK_CODE, PayuConstants.TEZ));
                post.append(concatParams(PayuConstants.PG, PayuConstants.UPI));
                if (!TextUtils.isEmpty(mPaymentParams.getVpa()))
                    post.append(concatParams(PayuConstants.VPA, mPaymentParams.getVpa().trim()));
                return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, trimAmpersand(post.toString()));
        }

        return getReturnData(PayuErrors.NO_ERROR, PayuConstants.SUCCESS, post.toString());
    }


    public void analizingTransaction() {
        PayuUtils payuUtils = new PayuUtils();
        HashMap<String, String> hashMap = new HashMap<>();
        hashMap.put("transactionID", mPaymentParams.getTxnId());
        hashMap.put("keyAnalytics", mPaymentParams.getKey());
        hashMap.put("paymentMode", mPaymentMode);
        hashMap.put("sdkVersion", BuildConfig.VERSION_NAME);
        PayuUtils.setVariableCB("com.payu.custombrowser.Bank", hashMap, "Version");
        payuUtils.deviceAnalytics(applicationContext, mPaymentParams.getKey(), mPaymentParams.getTxnId());

    }
}
