//    Copyright (c) 2014 - 2015 payu@india.com
//
//    Permission is hereby granted, free of charge, to any person obtaining a copy
//    of this software and associated documentation files (the "Software"), to deal
//    in the Software without restriction, including without limitation the rights
//    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//    copies of the Software, and to permit persons to whom the Software is
//    furnished to do so, subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be included in
//    all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//    THE SOFTWARE.

package com.payu.india.Payu;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.telephony.CellInfo;
import android.telephony.CellInfoCdma;
import android.telephony.CellInfoGsm;
import android.telephony.CellInfoLte;
import android.telephony.CellInfoWcdma;
import android.telephony.CellSignalStrengthCdma;
import android.telephony.CellSignalStrengthGsm;
import android.telephony.CellSignalStrengthLte;
import android.telephony.CellSignalStrengthWcdma;
import android.telephony.TelephonyManager;
import android.util.DisplayMetrics;

import com.payu.india.BuildConfig;
import com.payu.india.Model.PayuConfig;
import com.payu.india.Model.PostData;
import com.payu.india.Model.StoredCard;
import com.payu.india.Model.TransactionResponse;
import com.payu.india.Tasks.PayuUploadDeviceAnalytics;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * Created by franklin on 6/18/15.
 * Payu util class provides protected, public functions to sdk and App.
 * Use the {@link PayuUtils#SBI_MAES_BIN } set to find whether the number is state bank maestro number or not
 * Use the {@link PayuUtils#validateCardNumber(String)} method to validate user card number, it includes length luhn validation.
 * Use the {@link PayuUtils#luhn(String)} method to validate card number, use this function only if you validate length at your end.
 * Use the {@link PayuUtils#getIssuer(String)} method to get the issuer, it will return the issuer from {VISA, AMEX, MAES, MAST, SMAE, LASER, DINR, JCB}
 * use the {@link PayuUtils#validateCvv(String, String)} method to validate cvv, AMEX has 4 digit, SMAE has no cvv, other has 3 dight cvv.
 * use the {@link PayuUtils#validateExpiry(int, int)} method to validate whether the card is expired or not!.
 * use the {@link PayuUtils#concatParams(String, String)} method to concat both params1 and param2 using equal to (=) and add a ampersand at the end!
 * adding default arguments{@link PayuUtils#getReturnData(String)} {@link PayuUtils#getReturnData(int, String)}
 * use the {@link PayuUtils#getReturnData(int, String, String)} to send the data back to app as {@link PostData}
 * use the {@link PayuUtils#trimAmpersand(String)} remove leading ampersand if any!.
 */
public class PayuUtils {


public  static String cbVersion;
    /**
     * Known Sate bank maestro bins
     * helps us to validate cvv expmon, expyr,
     * SMAE cards do not have CVV, Expiry month, Expiry year.
     */
    public static Set<String> SBI_MAES_BIN;

    //analytics key for device and event analytics
    private static String keyAnalyticsUtil;

    static {
        SBI_MAES_BIN = new HashSet<>();
        SBI_MAES_BIN.add("504435");
        SBI_MAES_BIN.add("504645");
        SBI_MAES_BIN.add("504775");
        SBI_MAES_BIN.add("504809");
        SBI_MAES_BIN.add("504993");
        SBI_MAES_BIN.add("600206");
        SBI_MAES_BIN.add("603845");
        SBI_MAES_BIN.add("622018");
        SBI_MAES_BIN.add("504774");
    }

    /**
     * includes length validation also
     * use {@link PayuUtils#luhn(String)} to validate.
     *
     * @param cardNumber any card number
     * @return true if valid else false
     */
    public Boolean validateCardNumber(String cardNumber) {
        if (cardNumber.length() < 12) {
            return false;
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.RUPAY) && cardNumber.length() == 16){
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.VISA) && cardNumber.length() == 16) {
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.MAST) && cardNumber.length() == 16) {
            return luhn(cardNumber);
        } else if ((getIssuer(cardNumber).contentEquals(PayuConstants.MAES) || getIssuer(cardNumber).contentEquals(PayuConstants.SMAE)) && cardNumber.length() >= 12 && cardNumber.length() <= 19) {
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.DINR) && cardNumber.length() == 14) {
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.AMEX) && cardNumber.length() == 15) {
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.JCB) && cardNumber.length() == 16) {
            return luhn(cardNumber);
        }
        return false;
    }

    /**
     * Universal luhn validation for Credit, Debit cards.
     *
     * @param cardNumber any card number
     * @return true if valid else false
     */
    public Boolean luhn(String cardNumber) {
        int sum = 0;
        boolean alternate = false;
        for (int i = cardNumber.length() - 1; i >= 0; i--) {
            int n = Integer.parseInt(cardNumber.substring(i, i + 1));
            if (alternate) {
                n *= 2;
                if (n > 9) {
                    n = (n % 10) + 1;
                }
            }
            sum += n;
            alternate = !alternate;
        }
        if (sum % 10 == 0) {
            return true;
        }
        return false;
    }


    /**
     * Gives the Issuer of the card number.
     * using simple regex we can actually figure out the card belong to which issuer amoung VISA, AMEX, MAES, MAST, SMAE, LASER, DINR, JCB
     *
     * @param mCardNumber first 6 digit of card number is more than enough to figure out. (Card Bin)
     * @return Issuer of the card
     */
    public String getIssuer(String mCardNumber) {
        if (mCardNumber.startsWith("4")) {
            return PayuConstants.VISA;
        }else if (mCardNumber.matches("^508[5-9][0-9][0-9]|60698[5-9]|60699[0-9]|607[0-8][0-9][0-9]|6079[0-7][0-9]|60798[0-4]|(?!608000)608[0-4][0-9][0-9]|608500|6521[5-9][0-9]|652[2-9][0-9][0-9]|6530[0-9][0-9]|6531[0-4][0-9]")){
            return PayuConstants.RUPAY;
        } else if (mCardNumber.matches("^((6304)|(6706)|(6771)|(6709))[\\d]+")) {
            return PayuConstants.LASER;
        } else if (mCardNumber.matches("6(?:011|5[0-9]{2})[0-9]{12}[\\d]+")) {
            return PayuConstants.LASER;
        } else if (mCardNumber.matches("(5[06-8]|6\\d)\\d{14}(\\d{2,3})?[\\d]+") || mCardNumber.matches("(5[06-8]|6\\d)[\\d]+") || mCardNumber.matches("((504([435|645|774|775|809|993]))|(60([0206]|[3845]))|(622[018])\\d)[\\d]+")) {
            if (mCardNumber.length() >= 6) { // wel we have 6 digit bin
                if (SBI_MAES_BIN.contains(mCardNumber.substring(0, 6))) {
                    return PayuConstants.SMAE;
                }
            }
            return PayuConstants.MAES;
        } else if (mCardNumber.matches("^5[1-5][\\d]+")) {
            return PayuConstants.MAST;
        } else if (mCardNumber.matches("^3[47][\\d]+")) {
            return PayuConstants.AMEX;
        } else if (mCardNumber.startsWith("36") || mCardNumber.matches("^30[0-5][\\d]+") || mCardNumber.matches("2(014|149)[\\d]+")) {
            return PayuConstants.DINR;
        } else if (mCardNumber.matches("^35(2[89]|[3-8][0-9])[\\d]+")) {
            return PayuConstants.JCB;
        }
        return "";
    }

    /**
     * helps to validate cvv
     * we need card bin to figure out issuer,
     * from issuer we validate cvv
     * Oh! AMEX has 4 digit,
     * Crap! SMAE has no cvv,
     * okay, other has 3 dight cvv.
     *
     * @param cardNumber Card bin
     * @param cvv        cvv
     * @return true if valid cvv else false
     */
    public boolean validateCvv(String cardNumber, String cvv) {
        String issuer = getIssuer(cardNumber);
        if (issuer.contentEquals(PayuConstants.SMAE)) {
            return true;
        }else if (issuer.contentEquals("")){
            return false;
        }
        else if (issuer.contentEquals(PayuConstants.AMEX) & cvv.length() == 4) {
            return true;
        } else if (!issuer.contentEquals(PayuConstants.AMEX) && cvv.length() == 3) {
            return true;
        }
        return false;
    }

    /**
     * helps to validate whether the card is expired or not!.
     *
     * @param expiryMonth expmon
     * @param expiryYear  expyr
     * @return true if valid else false
     */
    public boolean validateExpiry(int expiryMonth, int expiryYear) {
        Calendar calendar = Calendar.getInstance();
        if (expiryMonth < 1 || expiryMonth > 12 || String.valueOf(expiryYear).length() != 4) { // expiry month validation
            return false;
        } else if (calendar.get(Calendar.YEAR) > expiryYear || (calendar.get(Calendar.YEAR) == expiryYear && calendar.get(Calendar.MONTH) + 1 > expiryMonth)) { // expiry date validation.
            return false;
        }
        return true;
    }

    /**
     * just to make our life easier, lets define a function
     * concat two strings with equal sign and add an ampersand at the end
     *
     * @param key   example txnid
     * @param value example payu12345
     * @return concatenated String
     */

    protected String concatParams(String key, String value) {
        return key + "=" + value + "&";
    }

    /**
     * Defalut param of {@link PayuUtils#getReturnData(int, String, String)}
     * Use this if your first param is {@link PayuErrors#MISSING_PARAMETER_EXCEPTION} and second param is {@link PayuConstants#ERROR}
     *
     * @param result result it the result of your PostData
     * @return PostData object.
     */
    protected PostData getReturnData(String result) {
        return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuConstants.ERROR, result);
    }

    /**
     * Defalut param of {@link PayuUtils#getReturnData(int, String, String)}
     * Use this param if there is something wrong while validating params {@link PayuConstants#ERROR}
     *
     * @param code   error code to be returned with PostData
     * @param result error message (result) to be returned with PostData
     * @return PostData object.
     */
    protected PostData getReturnData(int code, String result) {
        return getReturnData(code, PayuConstants.ERROR, result);
    }

    /**
     * Use this is method to return the PostData to the user {@link PostData#code} {@link PostData#status} {@link PostData#result}
     *
     * @param code    error code to be returned with PostData
     * @param status, status message it can either be {@link PayuConstants#SUCCESS} or {@link PayuConstants#ERROR}
     * @param result  error message (result) to be returned with PostData
     * @return PostData object.
     */
    protected PostData getReturnData(int code, String status, String result) {
        PostData postData = new PostData();
        postData.setCode(code);
        postData.setStatus(status);
        postData.setResult(result);
        return postData;
    }

    /**
     * This method is used to trim the leading Ampersand sign
     *
     * @param data , should be concatenated data with equal sign and ampersand.
     * @return example txnid=payu12345
     */
    protected String trimAmpersand(String data) {
        return data.charAt(data.length() - 1) == '&' ? data.substring(0, data.length() - 1) : data;
    }

    static String getNetworkStatus(Activity activity) {
        try {
            if (null != activity && !activity.isFinishing()) {
                ConnectivityManager cm = (ConnectivityManager) activity.getSystemService(Context.CONNECTIVITY_SERVICE);
                NetworkInfo info = cm.getActiveNetworkInfo();
                if (info == null || !info.isConnected())
                    return "Not connected"; //not connected
                if (info.getType() == ConnectivityManager.TYPE_WIFI)
                    return "WIFI";
                if (info.getType() == ConnectivityManager.TYPE_MOBILE) {
                    int networkType = info.getSubtype();
                    switch (networkType) {
                        case TelephonyManager.NETWORK_TYPE_GPRS:
                            return "GPRS";
                        case TelephonyManager.NETWORK_TYPE_EDGE:
                            return "EDGE";
                        case TelephonyManager.NETWORK_TYPE_CDMA:
                            return "CDMA";
                        case TelephonyManager.NETWORK_TYPE_1xRTT:
                        case TelephonyManager.NETWORK_TYPE_IDEN: //api<8 : replace by 11
                            return "2G";
                        case TelephonyManager.NETWORK_TYPE_UMTS:
                        case TelephonyManager.NETWORK_TYPE_EVDO_0:
                        case TelephonyManager.NETWORK_TYPE_EVDO_A:
                        case TelephonyManager.NETWORK_TYPE_HSDPA:
                        case TelephonyManager.NETWORK_TYPE_HSUPA:
                        case TelephonyManager.NETWORK_TYPE_HSPA:
                            return "HSPA";
                        case TelephonyManager.NETWORK_TYPE_EVDO_B: //api<9 : replace by 14
                        case TelephonyManager.NETWORK_TYPE_EHRPD:  //api<11 : replace by 12
                        case TelephonyManager.NETWORK_TYPE_HSPAP:  //api<13 : replace by 15
                            return "3G";
                        case TelephonyManager.NETWORK_TYPE_LTE:    //api<11 : replace by 13
                            return "4G";
                        default:
                            return "?";
                    }
                }
            }
        }catch (Exception e){
            return "?";
        }
        return "?";
    }

    /**
     *
     * provide the density of device in DPI
     *
     * @param activity base activity context
     * @return density in dpi of the device
     */
    private String getDeviceDensity(Activity activity){
        DisplayMetrics metrics = new DisplayMetrics();
        activity.getWindowManager().getDefaultDisplay().getMetrics(metrics);
        return metrics.densityDpi+"";
    }

    /**
     *
     * Provide the network strength of connected network
     *
     * @param activity base activity context
     * @return return the network strength
     */
    private int getNetworkStrength(Activity activity){

        try {
            ConnectivityManager connectivityManager = (ConnectivityManager) activity.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo[] netInfo = connectivityManager.getAllNetworkInfo();
            TelephonyManager telephonyManager = (TelephonyManager) activity.getSystemService(Context.TELEPHONY_SERVICE);

            int strength = 0;

            for (NetworkInfo networkInfo : netInfo) {
                // we are interested only in mobile.
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2 && networkInfo.getTypeName().equalsIgnoreCase("MOBILE") && networkInfo.isConnected()) { // okay now its mobile network
                    // lets find the strength.
                    for (CellInfo info : telephonyManager.getAllCellInfo()) {
                        if (info.isRegistered()) { // connected
                            if (info instanceof CellInfoGsm) {
                                CellSignalStrengthGsm gsm = ((CellInfoGsm) info).getCellSignalStrength();
                                strength = gsm.getDbm();
                            } else if (info instanceof CellInfoCdma) {
                                CellSignalStrengthCdma cdma = ((CellInfoCdma) info).getCellSignalStrength();
                                strength = cdma.getDbm();
                            } else if (info instanceof CellInfoLte) {
                                CellSignalStrengthLte lte = ((CellInfoLte) info).getCellSignalStrength();
                                strength = lte.getDbm();
                            } else if (info instanceof CellInfoWcdma) {
                                CellSignalStrengthWcdma wcdma = ((CellInfoWcdma) info).getCellSignalStrength(); // jelly bean mr2
                                strength = wcdma.getDbm();
                            }
                        }
                    }
                }
            }
            return strength;
        }catch (Exception e){
            return 0;
        }

    }

    /**
     * convert params to json object, and call device analytics
     *
     * @param context base activity instance
     * @param merchantKey merchant key
     * @param transactionId transaction id
     */

    public void deviceAnalytics(Context context, String merchantKey, String transactionId ) {
        Activity activity = (Activity) context;
        JSONObject deviceDetails = new JSONObject();
        try {
            deviceDetails.put(PayuConstants.DEVICE_RESOLUTION, getDeviceDensity(activity) + "");
            deviceDetails.put(PayuConstants.DEVICE_MANUFACTURE, Build.MANUFACTURER);
            deviceDetails.put(PayuConstants.DEVICE_MODEL, Build.MODEL);

         if(keyAnalyticsUtil!=null && !keyAnalyticsUtil.trim().equals(""))
                deviceDetails.put(PayuConstants.MERCHANT_KEY, keyAnalyticsUtil);
            else
            deviceDetails.put(PayuConstants.MERCHANT_KEY, merchantKey);
            deviceDetails.put(PayuConstants.TRANSACTION_ID, transactionId);
            deviceDetails.put(PayuConstants.SDK_VERSION_NAME, BuildConfig.VERSION_NAME + "");
            deviceDetails.put(PayuConstants.CB_VERSION_NAME, cbVersion);
            deviceDetails.put(PayuConstants.DEVICE_OS_VERSION, Build.VERSION.SDK_INT + "");
            deviceDetails.put(PayuConstants.NETWORK_INFO, PayuUtils.getNetworkStatus(activity) + "");
            deviceDetails.put(PayuConstants.NETWORK_STRENGTH, getNetworkStrength(activity) + "");

        } catch (Exception e) {
            e.printStackTrace();
        }

        new PayuUploadDeviceAnalytics(activity, "sdk_local_cache_device").log(deviceDetails.toString());
    }

    /**
     * Store key value pair in shared preference
     *
     * @param context activity instance
     * @param key key for shared preference
     * @param value value of the corresponding key
     */

    public void storeInSharedPreferences(Context context, String key, String value){
        SharedPreferences sharedPreferences = context.getSharedPreferences(PayuConstants.PAYU_PREFERENCE, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.putString(key, value);
        editor.apply();
    }

    /**
     * Get the value for the provided key from shared preference
     *
     * @param context base activity context
     * @param key shared preference key
     * @return value of the provided key
     */
    public String getFromSharedPreferences(Context context, String key){
        return getFromSharedPreferences(context, key, PayuConstants.DEFAULT);
    }

    /**
     *
     * @param context base activity context
     * @param key shared preference key
     * @param defaultValue default value
     * @return  value for the provided key store in shared preference
     * default value if no key exist
     */
    public String getFromSharedPreferences(Context context, String key, String defaultValue){
        SharedPreferences sharedPreferences = context.getSharedPreferences(PayuConstants.PAYU_PREFERENCE, Context.MODE_PRIVATE);
        return sharedPreferences.getString(key, defaultValue);
    }

    /**
     * Delete the key from shared preference
     * @param context base activity context
     * @param key shared preference key
     */
    public void removeFromSharedPreferences(Context context, String key){
        SharedPreferences sharedPreferences = context.getSharedPreferences(PayuConstants.PAYU_PREFERENCE, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.remove(key);
        editor.apply();
    }


    /**
     * Segregate onclick payment cards and stored cards.
     * should be used when {@link PayuConstants#STORE_ONE_CLICK_HASH_MOBILE}
     * @param context app context
     * @param cardList list of cards get from payu server
     * Hash map {@link PayuConstants#ONE_CLICK_CHECKOUT} - one click cards
     * Hash map {@link PayuConstants#STORED_CARD} - stored cards
     * @return Hash map or cards
     */
    public HashMap<String, ArrayList<StoredCard>> getStoredCard(Context context, ArrayList<StoredCard>cardList){
        ArrayList<StoredCard> oneClickCards = new ArrayList<>();
        ArrayList<StoredCard> storedCards = new ArrayList<>();
        HashMap<String, ArrayList<StoredCard>> cardMap = new HashMap<>();
        SharedPreferences sharedPreferences = context.getSharedPreferences(PayuConstants.PAYU_PREFERENCE, Context.MODE_PRIVATE);
        if(null != cardList) {
            for (StoredCard card : cardList) {
                if ((card.getEnableOneClickPayment() == 1 && !sharedPreferences.getString(card.getCardToken(), PayuConstants.DEFAULT).contains(PayuConstants.DEFAULT)) || getIssuer(card.getCardBin()).contentEquals(PayuConstants.SMAE)) { // smae considered as one click card
                    oneClickCards.add(card);
                } else {
                    storedCards.add(card);
                }
            }
        }
        cardMap.put(PayuConstants.ONE_CLICK_CHECKOUT, oneClickCards);
        cardMap.put(PayuConstants.STORED_CARD, storedCards);
        return cardMap;
    }

    /**
     * Segregate onclick payment cards and stored cards
     * Should be used when {@link PayuConstants#STORE_ONE_CLICK_HASH_SERVER}
     * @param cardList list of stored cards received from payu server
     * @param oneClickCardTokens map of card token and merchant hashes.
     * Hash map {@link PayuConstants#ONE_CLICK_CHECKOUT} - one click cards
     * Hash map {@link PayuConstants#STORED_CARD} - stored cards
     * @return Hash map or cards
     */
    public HashMap<String, ArrayList<StoredCard>> getStoredCard(ArrayList<StoredCard>cardList, HashMap<String, String>oneClickCardTokens){
        ArrayList<StoredCard> oneClickCards = new ArrayList<>();
        ArrayList<StoredCard> storedCards = new ArrayList<>();
        HashMap<String, ArrayList<StoredCard>> cardMap = new HashMap<>();
        if(null != cardList) {
            for (StoredCard card : cardList) {
                if ((null != oneClickCardTokens && card.getEnableOneClickPayment() == 1 && oneClickCardTokens.containsKey(card.getCardToken()))){
                    oneClickCards.add(card);
                } else {
                    storedCards.add(card);
                }
            }
        }
        cardMap.put(PayuConstants.ONE_CLICK_CHECKOUT, oneClickCards);
        cardMap.put(PayuConstants.STORED_CARD, storedCards);
        return cardMap;
    }

    /**
     * set the analytics key for the analytics of events and device
     *
     * @param payuConfig payment params
     */
    public static  void setAnalyticsKeyCB(PayuConfig payuConfig){
            String analyticsKey = getAnalyticsKeyFromConfig(payuConfig);
            setAnalyticsKeyCB(analyticsKey);
    }

    /**
     * set the analytics key for the analytics of events and device
     *
     * @param analyticsKey analytics key
     */

    public static void setAnalyticsKeyCB(String analyticsKey){
        try {
            if (analyticsKey != null && !analyticsKey.trim().equals("")) {
                keyAnalyticsUtil=analyticsKey;
                Class aClass = Class.forName("com.payu.custombrowser.Bank");
                Field field = aClass.getDeclaredField("keyAnalytics");
                field.setAccessible(true);
                field.set(null, analyticsKey);
                field.setAccessible(false);

            }
        }catch(Exception e)
        {
            e.printStackTrace();
        }
    }

    public static void setVariableCB(String pakageName, HashMap<String,String> cbAnalyticsKeys,String Version){
        try {
                Class aClass = Class.forName(pakageName);
                for(String key :cbAnalyticsKeys.keySet()) {
                    try {
                        Field field = aClass.getDeclaredField(key);
                        field.setAccessible(true);
                        field.set(null, cbAnalyticsKeys.get(key));
                        field.setAccessible(false);
                    }catch(Exception e){

                    }
                }
            Field field = aClass.getDeclaredField(Version);
            field.setAccessible(true);
            cbVersion = (String) field.get(null);
        }catch(Exception e) {
            e.printStackTrace();
        }
    }


    /**
     * filter analytics key from payment params
     *
     * @param payuConfig payment params bean
     * @return analytics key
     */

    private  static String getAnalyticsKeyFromConfig(PayuConfig payuConfig){
        try {
            String[] list = payuConfig.getData().split("&");
            for (String item : list) {
                String[] items = item.split("=");
                if (items.length >= 2) {
                    String id = items[0];
                    switch (id) {
                        case "key":
                            return items[1];
                    }
                }
            }
        }catch(Exception e)
        {
            e.printStackTrace();
        }
        return null;
    }

    private String getVariableReflection(String className, String fieldName){
        try {
            Class mClass = Class.forName(className);
            Field mField = mClass.getDeclaredField(fieldName);
            return (String) mField.get(null);
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (NoSuchFieldException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
        return null;
    }

    public TransactionResponse getTransactionResponse(String transactionRespObj) {
        try {
            JSONObject mJSONObject = new JSONObject(transactionRespObj);
            TransactionResponse transactionResponse = new TransactionResponse();
            Iterator<String> keysIterator = mJSONObject.keys();

            while (keysIterator.hasNext()) {
                switch (keysIterator.next()) {
                    case PayuConstants.ID:
                        if (mJSONObject.has(PayuConstants.ID)) {
                            transactionResponse.setId(mJSONObject.get(PayuConstants.ID).toString());
                        }
                        break;

                    case PayuConstants.MODE:
                        if (mJSONObject.has(PayuConstants.MODE)) {
                            transactionResponse.setMode(mJSONObject.get(PayuConstants.MODE).toString());
                        }
                        break;

                    case PayuConstants.STATUS:
                        if (mJSONObject.has(PayuConstants.STATUS)) {
                            transactionResponse.setStatus(mJSONObject.get(PayuConstants.STATUS).toString());
                        }
                        break;

                    case PayuConstants.UNMAPPED_STATUS:
                        if (mJSONObject.has(PayuConstants.UNMAPPED_STATUS)) {
                            transactionResponse.setUnMappedStatus(mJSONObject.get(PayuConstants.UNMAPPED_STATUS).toString());
                        }
                        break;

                    case PayuConstants.KEY:
                        if (mJSONObject.has(PayuConstants.KEY)) {
                            transactionResponse.setKey(mJSONObject.get(PayuConstants.KEY).toString());
                        }
                        break;

                    case PayuConstants.TXNID:
                        if (mJSONObject.has(PayuConstants.TXNID)) {
                            transactionResponse.setTxnid(mJSONObject.get(PayuConstants.TXNID).toString());
                        }
                        break;

                    case PayuConstants.TRANSACTION_FEE:
                        if (mJSONObject.has(PayuConstants.TRANSACTION_FEE)) {
                            transactionResponse.setTransactionFee(mJSONObject.get(PayuConstants.TRANSACTION_FEE).toString());
                        }
                        break;

                    case PayuConstants.CARDCATEGORY:
                        if (mJSONObject.has(PayuConstants.CARDCATEGORY)) {
                            transactionResponse.setCardCategory(mJSONObject.get(PayuConstants.CARDCATEGORY).toString());
                        }
                        break;

                    case PayuConstants.DISCOUNT:
                        if (mJSONObject.has(PayuConstants.DISCOUNT)) {
                            transactionResponse.setDiscount(mJSONObject.get(PayuConstants.DISCOUNT).toString());
                        }
                        break;

                    case PayuConstants.ADDITIONAL_CHARGES:
                        if (mJSONObject.has(PayuConstants.ADDITIONAL_CHARGES)) {
                            transactionResponse.setAdditionalCharges(mJSONObject.get(PayuConstants.ADDITIONAL_CHARGES).toString());
                        }
                        break;

                    case PayuConstants.ADDED_ON:
                        if (mJSONObject.has(PayuConstants.ADDED_ON)) {
                            transactionResponse.setAddedOn(mJSONObject.get(PayuConstants.ADDED_ON).toString());
                        }
                        break;

                    case PayuConstants.PRODUCT_INFO:
                        if (mJSONObject.has(PayuConstants.PRODUCT_INFO)) {
                            transactionResponse.setProductInfo(mJSONObject.get(PayuConstants.PRODUCT_INFO).toString());
                        }
                        break;

                    case PayuConstants.FIRST_NAME:
                        if (mJSONObject.has(PayuConstants.FIRST_NAME)) {
                            transactionResponse.setFirstName(mJSONObject.get(PayuConstants.FIRST_NAME).toString());
                        }
                        break;

                    case PayuConstants.EMAIL:
                        if (mJSONObject.has(PayuConstants.EMAIL)) {
                            transactionResponse.setEmail(mJSONObject.get(PayuConstants.EMAIL).toString());
                        }
                        break;

                    case PayuConstants.UDF1:
                        if (mJSONObject.has(PayuConstants.UDF1)) {
                            transactionResponse.setUdf1(mJSONObject.get(PayuConstants.UDF1).toString());
                        }
                        break;

                    case PayuConstants.UDF2:
                        if (mJSONObject.has(PayuConstants.UDF2)) {
                            transactionResponse.setUdf2(mJSONObject.get(PayuConstants.UDF2).toString());
                        }
                        break;

                    case PayuConstants.UDF3:
                        if (mJSONObject.has(PayuConstants.UDF3)) {
                            transactionResponse.setUdf3(mJSONObject.get(PayuConstants.UDF3).toString());
                        }
                        break;

                    case PayuConstants.UDF4:
                        if (mJSONObject.has(PayuConstants.UDF4)) {
                            transactionResponse.setUdf4(mJSONObject.get(PayuConstants.UDF4).toString());
                        }
                        break;

                    case PayuConstants.UDF5:
                        if (mJSONObject.has(PayuConstants.UDF5)) {
                            transactionResponse.setUdf5(mJSONObject.get(PayuConstants.UDF5).toString());
                        }
                        break;

                    case PayuConstants.HASH:
                        if (mJSONObject.has(PayuConstants.HASH)) {
                            transactionResponse.setHash(mJSONObject.get(PayuConstants.HASH).toString());
                        }
                        break;

                    case PayuConstants.FIELD1:
                        if (mJSONObject.has(PayuConstants.FIELD1)) {
                            transactionResponse.setField1(mJSONObject.get(PayuConstants.FIELD1).toString());
                        }
                        break;

                    case PayuConstants.FIELD2:
                        if (mJSONObject.has(PayuConstants.FIELD2)) {
                            transactionResponse.setField2(mJSONObject.get(PayuConstants.FIELD2).toString());
                        }
                        break;

                    case PayuConstants.FIELD3:
                        if (mJSONObject.has(PayuConstants.FIELD3)) {
                            transactionResponse.setField3(mJSONObject.get(PayuConstants.FIELD3).toString());
                        }
                        break;

                    case PayuConstants.FIELD4:
                        if (mJSONObject.has(PayuConstants.FIELD4)) {
                            transactionResponse.setField4(mJSONObject.get(PayuConstants.FIELD4).toString());
                        }
                        break;

                    case PayuConstants.FIELD9:
                        if (mJSONObject.has(PayuConstants.FIELD9)) {
                            transactionResponse.setField9(mJSONObject.get(PayuConstants.FIELD9).toString());
                        }
                        break;

                    case PayuConstants.PAYMENT_SOURCE:
                        if (mJSONObject.has(PayuConstants.PAYMENT_SOURCE)) {
                            transactionResponse.setPaymentSource(mJSONObject.get(PayuConstants.PAYMENT_SOURCE).toString());
                        }
                        break;

                    case PayuConstants.PG_TYPE:
                        if (mJSONObject.has(PayuConstants.PG_TYPE)) {
                            transactionResponse.setPgType(mJSONObject.get(PayuConstants.PG_TYPE).toString());
                        }
                        break;

                    case PayuConstants.BANK_REF_NUM:
                        if (mJSONObject.has(PayuConstants.BANK_REF_NUM)) {
                            transactionResponse.setBankRefNo(mJSONObject.get(PayuConstants.BANK_REF_NUM).toString());
                        }
                        break;

                    case PayuConstants.IBIBO_CODE:
                        if (mJSONObject.has(PayuConstants.IBIBO_CODE)) {
                            transactionResponse.setIbiboCode(mJSONObject.get(PayuConstants.IBIBO_CODE).toString());
                        }
                        break;

                    case PayuConstants.ERROR_CODE:
                        if (mJSONObject.has(PayuConstants.ERROR_CODE)) {
                            transactionResponse.setErrorCode(mJSONObject.get(PayuConstants.ERROR_CODE).toString());
                        }
                        break;

                    case PayuConstants.ERROR_MESSAGE2:
                        if (mJSONObject.has(PayuConstants.ERROR_MESSAGE2)) {
                            transactionResponse.setErrorMessage(mJSONObject.get(PayuConstants.ERROR_MESSAGE2).toString());
                        }
                        break;

                    case PayuConstants.CARD_TOKEN:
                        if (mJSONObject.has(PayuConstants.CARD_TOKEN)) {
                            transactionResponse.setCardToken(mJSONObject.get(PayuConstants.CARD_TOKEN).toString());
                        }
                        break;

                    case PayuConstants.NAME_ON_CARD:
                        if (mJSONObject.has(PayuConstants.NAME_ON_CARD)) {
                            transactionResponse.setNameOnCard(mJSONObject.get(PayuConstants.NAME_ON_CARD).toString());
                        }
                        break;

                    case PayuConstants.CARD_NO:
                        if (mJSONObject.has(PayuConstants.CARD_NO)) {
                            transactionResponse.setCardNumber(mJSONObject.get(PayuConstants.CARD_NO).toString());
                        }
                        break;

                    case PayuConstants.ISSUINGBANK:
                        if (mJSONObject.has(PayuConstants.ISSUINGBANK)) {
                            transactionResponse.setIssuingBank(mJSONObject.get(PayuConstants.ISSUINGBANK).toString());
                        }
                        break;

                    case PayuConstants.CARD_TYPE:
                        if (mJSONObject.has(PayuConstants.CARD_TYPE)) {
                            transactionResponse.setCardType(mJSONObject.get(PayuConstants.CARD_TYPE).toString());
                        }
                        break;

                    case PayuConstants.IS_SEAMLESS:
                        if (mJSONObject.has(PayuConstants.IS_SEAMLESS)) {
                            transactionResponse.setIsSeamless(mJSONObject.get(PayuConstants.IS_SEAMLESS).toString());
                        }
                        break;

                    case PayuConstants.SURL:
                        if (mJSONObject.has(PayuConstants.SURL)) {
                            transactionResponse.setSurl(mJSONObject.get(PayuConstants.SURL).toString());
                        }
                        break;

                    case PayuConstants.FURL:
                        if (mJSONObject.has(PayuConstants.FURL)) {
                            transactionResponse.setFurl(mJSONObject.get(PayuConstants.FURL).toString());
                        }
                        break;


                    case PayuConstants.MERCHANT_HASH:
                        if (mJSONObject.has(PayuConstants.MERCHANT_HASH)) {
                            transactionResponse.setMerchantHash(mJSONObject.get(PayuConstants.MERCHANT_HASH).toString());
                        }
                        break;

                }
            }

            return transactionResponse;
        } catch (JSONException e) {
            e.printStackTrace();
            return null;
        }
    }
}

