// The MIT License (MIT)
//
// Copyright (c) 2014-2015 PayU India
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package com.payu.custombrowser.util;

import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.graphics.drawable.Drawable;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkInfo;
import android.net.TrafficStats;
import android.net.Uri;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.support.v7.app.NotificationCompat;
import android.telephony.CellInfo;
import android.telephony.CellInfoCdma;
import android.telephony.CellInfoGsm;
import android.telephony.CellInfoLte;
import android.telephony.CellInfoWcdma;
import android.telephony.CellSignalStrengthCdma;
import android.telephony.CellSignalStrengthGsm;
import android.telephony.CellSignalStrengthLte;
import android.telephony.CellSignalStrengthWcdma;
import android.telephony.TelephonyManager;
import android.util.Base64;
import android.util.DisplayMetrics;
import android.view.View;
import android.view.animation.AlphaAnimation;
import android.webkit.CookieManager;
import android.webkit.CookieSyncManager;
import android.webkit.WebView;
import com.payu.custombrowser.R;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;

/**
 * Created by minie on 29/6/15.
 * Contains cb's utility functions.
 */
public class CBUtil {

    public final static String CB_PREFERENCE = "com.payu.custombrowser.payucustombrowser";
    private static SharedPreferences sharedPreferences;

    /**
     * Return system current time of system
     *
     * @return system current time in string
     */
    public static String getSystemCurrentTime() {
        try {
            Date currentDate = new Date(System.currentTimeMillis());
            DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            String date = df.format(currentDate);
            return date;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }
    /**
     * Provide JSONObject for analytics
     *
     * @param context        application context
     * @param key            analytics event ket
     * @param value          event value
     * @param bank           name of the bank
     * @param sdkMerchantKey merchant key
     * @param trnxID         transaction  ID
     * @return
     */
    public String getLogMessage(Context context, String key, String value, String bank, String sdkMerchantKey, String trnxID, String pageType) {
        try {
            JSONObject eventAnalytics = new JSONObject();
            eventAnalytics.put(CBAnalyticsConstant.PAYU_ID, getCookie(CBConstant.PAYUID, context));
            eventAnalytics.put(CBAnalyticsConstant.TRANSACTION_ID, trnxID);
            eventAnalytics.put(CBAnalyticsConstant.MERCHANT_KEY, sdkMerchantKey);
            eventAnalytics.put(CBAnalyticsConstant.PAGE_TYPE, pageType);
            eventAnalytics.put(CBAnalyticsConstant.KEY, key);
            eventAnalytics.put(CBAnalyticsConstant.VALUE, URLEncoder.encode(value, "UTF-8"));
            eventAnalytics.put(CBAnalyticsConstant.BANK_NAME, (bank == null ? "" : bank));
            eventAnalytics.put(CBAnalyticsConstant.PAKAGE_NAME, context.getPackageName());
            eventAnalytics.put(CBAnalyticsConstant.TIMESTAMP, getSystemCurrentTime());
            return eventAnalytics.toString();
        } catch (Exception e) {
            e.printStackTrace();
            return "{}";
        }
    }

    /**
     * should be used to decode initialize.js and bank specific js.
     *
     * @param fileInputStream reference of file input stream
     * @return decoded string.
     */

    public static String decodeContents(FileInputStream fileInputStream) {
        StringBuilder decoded = new StringBuilder();
        try {
            int c;
            int i = 0;
            while ((c = fileInputStream.read()) != -1) {
                if (i % 2 == 0) {
                    decoded.append((char) (c - ((i % 5) + 1)));
                } else {
                    decoded.append((char) (c + ((i % 5) + 1)));
                }
                i++;
            }
            fileInputStream.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return decoded.toString();
    }

    public static void setAlpha(float alpha, View view) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.HONEYCOMB) {
            final AlphaAnimation animation = new AlphaAnimation(alpha, alpha);
            animation.setDuration(10);
            animation.setFillAfter(true);
            view.startAnimation(animation);
        } else {
            view.setAlpha(alpha);
        }
    }

    /**
     * Update the last Url
     *
     * @param lastUrl last url in webview before transaction terminate
     * @return s:last stated url||f last finished url
     */
    public static String updateLastUrl(String lastUrl) {
        try {
            if (!(lastUrl.contains("||"))) {
                if (lastUrl.length() > 128)
                    return lastUrl.substring(0, 127);
                else
                    return lastUrl;
            } else {
                StringTokenizer st = new StringTokenizer(lastUrl, "||");
                String firstURl = st.nextToken();
                String secondUrl = st.nextToken();
                if (firstURl.length() > 128)
                    firstURl = firstURl.substring(0, 125);
                if (secondUrl.length() > 128)
                    secondUrl = secondUrl.substring(0, 125);
                return firstURl + "||" + secondUrl;
            }
        } catch (Exception e) {
            e.printStackTrace();
            return "";

        }
    }

    /**
     * set the analytics key for the analytics of events and device
     *
     * @param value analytics key
     */
    public static void setVariableReflection(String className, String value, String varName) {
        try {

            if (value != null && !value.trim().equals("")) {
                Class aClass = Class.forName(className);
                Field field = aClass.getDeclaredField(varName);
                field.setAccessible(true);
                field.set(null, value);
                field.setAccessible(false);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Filter the OTP from sms
     *
     * @param mBankJS bank JS
     * @param msgBody message received
     * @return filter OTP
     */
    public static String filterSMS(JSONObject mBankJS, String msgBody, Context context) {
        String mPassword = null;
        try {
            Matcher match;
            if (msgBody != null) {
                match = Pattern.compile(mBankJS.getString(context.getString(R.string.cb_detect_otp)), Pattern.CASE_INSENSITIVE).matcher(msgBody);
                if (match.find()) {
                    // we have otp sms
                    match = Pattern.compile(mBankJS.getString(context.getString(R.string.cb_find_otp)), Pattern.CASE_INSENSITIVE).matcher(msgBody);
                    if (match.find()) {
                        mPassword = match.group(1).replaceAll("[^0-9]", "");
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return mPassword;
    }

    /***
     * Return HttpURLConnection object
     *
     * @param strURL   postURL
     * @param postData data to be posted
     * @return connection object
     */
    public  HttpURLConnection getHttpsConn(String strURL, String postData) {
        try {
            return getHttpsConn(strURL, postData, -1);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }



    /***
     * Return HttpURLConnection object
     *
     * @param strURL   postURL
     * @param postData data to be posted
     * @param timeout timeout
     * @param cookiesList list of cookie
     * @return connection object
     */
    public  HttpURLConnection getHttpsConn(String strURL, String postData,int timeout,String cookiesList) {
        try {
            URL url = new URL(strURL);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod("POST");
            if(timeout!=-1) {
                conn.setConnectTimeout(timeout);
            }
            conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            if(postData!=null) {
                conn.setRequestProperty("Content-Length", String.valueOf(postData.length()));
            }
            if(cookiesList!=null){
                conn.setRequestProperty("Cookie",cookiesList);
            }
            conn.setDoOutput(true);
            if(postData!=null) {
                byte[] postParamsByte = postData.getBytes();
                conn.getOutputStream().write(postParamsByte);
            }
            return conn;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }
    /***
     * Return HttpURLConnection object
     *
     * @param strURL   postURL
     * @param postData data to be posted
     * @param timeout timeout
     * @return connection object
     */
    public  HttpURLConnection getHttpsConn(String strURL, String postData,int timeout) {
    return getHttpsConn(strURL,  postData, timeout,null);
    }
    /**
     * Return the connection object
     *
     * @param strURL url
     * @return HttpURLConnection object
     */
    public static HttpURLConnection getHttpsConn(String strURL) {
        try {
            URL url = new URL(strURL);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod("GET");
            conn.setRequestProperty("Accept-Charset", "UTF-8");
            return conn;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Read InputStream and return string buffer
     *
     * @param responseInputStream input stream
     * @return StringBuffer
     */
    public static StringBuffer getStringBufferFromInputStream(InputStream responseInputStream) {
        try {
            StringBuffer responseStringBuffer = new StringBuffer();
            byte[] byteContainer = new byte[1024];
            for (int i; (i = responseInputStream.read(byteContainer)) != -1; ) {
                responseStringBuffer.append(new String(byteContainer, 0, i));
            }
            return responseStringBuffer;
        } catch (Exception e) {
            e.printStackTrace();

        }
        return null;
    }

    /**
     * Function to find whether network is available or not.
     *
     * @param context application context
     * @return true if network available else false
     */
    public static boolean isNetworkAvailable(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
        return (networkInfo != null && networkInfo.isAvailable() && networkInfo.isConnected());
    }

    /**
     * Method to update retry urls
     *
     * @param data    URL List
     * @param context Application context
     * @return
     */
    public static List<String> updateRetryData(String data, Context context) {

        // Update data into SP
        setRetryData(data, context);
        // Then, update whitelistedRetryURl list
        return processAndAddWhiteListedUrls(data);

    }

    /**
     * Save retry urls in shared preference
     *
     * @param data
     * @param context
     */
    private static void setRetryData(String data, Context context) {
        if (data == null) {
            // Data is null, clear whitelisted urls
            SharedPreferenceUtil.addStringToSharedPreference(context, CBConstant.SP_RETRY_FILE_NAME, CBConstant.SP_RETRY_WHITELISTED_URLS, "");
        } else {
            // Update whitelisted urls in SP
            SharedPreferenceUtil.addStringToSharedPreference(context, CBConstant.SP_RETRY_FILE_NAME, CBConstant.SP_RETRY_WHITELISTED_URLS, data);
        }

        L.v("#### PAYU", "DATA UPDATED IN SHARED PREFERENCES");
    }

    /**
     * Clear Cookie(with n expiry date)
     */
    public void clearCookie(){
        CookieManager cookieManager = CookieManager.getInstance();

        if (Build.VERSION.SDK_INT >= 21) { // for icici white page fix.
            cookieManager.removeSessionCookies(null);
        } else {
           // if (Build.VERSION.SDK_INT >= 19)
                cookieManager.removeSessionCookie();
        }
    }

    /**
     * Return a list of whitelisted urls
     *
     * @param data
     * @return
     */
    public static List<String> processAndAddWhiteListedUrls(String data) {
        if (data != null && !data.equalsIgnoreCase("")) {
            // Explode at |
            String[] urls = data.split("\\|");
            for (String url : urls) {
                L.v("#### PAYU", "Split Url: " + url);
            }
            // For each of the urls in this exploded array, add to whitelisted url list
            if (urls != null && urls.length > 0) {
                List<String> whiteListedUrls = Arrays.asList(urls);
                return whiteListedUrls;
            }
            L.v("#### PAYU", "Whitelisted URLs from JS: " + data);
        }

        return new ArrayList<String>();
    }

    /**
     * @param key     shared preference key
     * @param context application context
     * @return
     */

    public boolean getBooleanSharedPreference(String key, Context context) {
        sharedPreferences = context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE);
        return sharedPreferences.getBoolean(key, false);
    }

    /**
     * @param key     shared preference key
     * @param context application context
     * @return
     */

    public boolean getBooleanSharedPreferenceDefaultTrue(String key, Context context) {
        sharedPreferences = context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE);
        return sharedPreferences.getBoolean(key, true);
    }

    /**
     * @param key     shared preference key
     * @param value   value of shared preference
     * @param context application context
     */
    public void setBooleanSharedPreference(String key, boolean value, Context context) {
        SharedPreferences.Editor editor = context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE).edit();
        editor.putBoolean(key, value);
        editor.apply();
    }

    /**
     * Calculate Device Density
     *
     * @param activity base activity
     * @return device density
     */
    public String getDeviceDensity(Activity activity) {
        DisplayMetrics metrics = new DisplayMetrics();
        activity.getWindowManager().getDefaultDisplay().getMetrics(metrics);
        return metrics.densityDpi + "";
    }

    /**
     * Calculate download speed
     */
    private void getDownloadSpeed() {
        String[] testing;
        testing = new String[2];
        long BeforeTime = System.currentTimeMillis();
        long TotalTxBeforeTest = TrafficStats.getTotalTxBytes();
        long TotalRxBeforeTest = TrafficStats.getTotalRxBytes();

        long TotalTxAfterTest = TrafficStats.getTotalTxBytes();
        long TotalRxAfterTest = TrafficStats.getTotalRxBytes();
        long AfterTime = System.currentTimeMillis();

        double TimeDifference = AfterTime - BeforeTime;

        double rxDiff = TotalRxAfterTest - TotalRxBeforeTest;
        double txDiff = TotalTxAfterTest - TotalTxBeforeTest;

        if ((rxDiff != 0) && (txDiff != 0)) {
            double rxBPS = (rxDiff / (TimeDifference / 1000)); // total rx bytes per second.
            double txBPS = (txDiff / (TimeDifference / 1000)); // total tx bytes per second.
            testing[0] = String.valueOf(rxBPS) + "bps. Total rx = " + rxDiff;
            testing[1] = String.valueOf(txBPS) + "bps. Total tx = " + txDiff;
        } else {
            testing[0] = "No uploaded or downloaded bytes.";
        }
    }

    /**
     * Calculate network status
     *
     * @param context
     * @return
     */

    public String getNetworkStatus(Context context) {
        try {
            if (null != context) {
                ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
                NetworkInfo info = cm.getActiveNetworkInfo();
                if (info == null || !info.isConnected())
                    return "Not connected"; //not connected
                if (info.getType() == ConnectivityManager.TYPE_WIFI)
                    return "WIFI";
                if (info.getType() == ConnectivityManager.TYPE_MOBILE) {
                    int networkType = info.getSubtype();
                    switch (networkType) {
                        case TelephonyManager.NETWORK_TYPE_GPRS:
                            return "GPRS";
                        case TelephonyManager.NETWORK_TYPE_EDGE:
                            return "EDGE";
                        case TelephonyManager.NETWORK_TYPE_CDMA:
                            return "CDMA";
                        case TelephonyManager.NETWORK_TYPE_1xRTT:
                        case TelephonyManager.NETWORK_TYPE_IDEN: //api<8 : replace by 11
                            return "2G";
                        case TelephonyManager.NETWORK_TYPE_UMTS:
                        case TelephonyManager.NETWORK_TYPE_EVDO_0:
                        case TelephonyManager.NETWORK_TYPE_EVDO_A:
                        case TelephonyManager.NETWORK_TYPE_HSDPA:
                        case TelephonyManager.NETWORK_TYPE_HSUPA:
                        case TelephonyManager.NETWORK_TYPE_HSPA:
                            return "HSPA";
                        case TelephonyManager.NETWORK_TYPE_EVDO_B: //api<9 : replace by 14
                        case TelephonyManager.NETWORK_TYPE_EHRPD:  //api<11 : replace by 12
                        case TelephonyManager.NETWORK_TYPE_HSPAP:  //api<13 : replace by 15
                            return "3G";
                        case TelephonyManager.NETWORK_TYPE_LTE:    //api<11 : replace by 13
                            return "4G";
                        default:
                            return "?";
                    }
                }
            }
        } catch (Exception e) {
            return "?";
        }
        return "?";
    }

    public NetworkInfo getNetWorkInfo(Context mContext) {
        ConnectivityManager connectivityManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo network = null;
        if (Build.VERSION.SDK_INT >= 21) {
            Network[] networks = connectivityManager.getAllNetworks();
            for (Network mNetwork : networks) {
                NetworkInfo networkInfo = connectivityManager.getNetworkInfo(mNetwork);
                if (networkInfo.getState().equals(NetworkInfo.State.CONNECTED)) {
                    network = networkInfo;
                }
            }
        } else {
            //noinspection deprecation
            NetworkInfo[] info = connectivityManager.getAllNetworkInfo();
            if (info != null) {
                for (NetworkInfo anInfo : info) {
                    if (anInfo.getState() == NetworkInfo.State.CONNECTED) {
                        network = anInfo;
                    }
                }
            }

        }
        return network;
    }

    /**
     * Checking for all possible internet providers
     **/

    public int getNetworkStrength(Context mContext) {
        NetworkInfo network = getNetWorkInfo(mContext);
        //  Toast.makeText(mContext, mContext.getString(R.string.please_connect_to_internet), Toast.LENGTH_SHORT).show();
        if (network != null) {
            if (network.getTypeName().equalsIgnoreCase("MOBILE")) {
                return getMobileStrength(mContext, network);
            } else if (network.getTypeName().equalsIgnoreCase("wifi") && hasPermission(mContext, "android.permission.ACCESS_WIFI_STATE")) {
                final WifiManager wifiManager = (WifiManager) mContext.getSystemService(Context.WIFI_SERVICE);
                try {
                    final WifiInfo connectionInfo = wifiManager.getConnectionInfo();
                    if (connectionInfo != null) {
                        // connectionInfo.getRssi() - According to IEEE 802.11 documentation: Lesser negative values denotes higher signal strength.The range is between -100 to 0 dBm, closer to 0 is higher strength and vice-versa.
                        return WifiManager.calculateSignalLevel(connectionInfo.getRssi(), 5); //returning a number between 0 and 4 (i.e. numLevel-1) : the number of bars you see in toolbar.
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            return 0;
        }
        return 0;
    }

    /**
     * Determines if the context calling has the required permission
     *
     * @param context    - the IPC context
     * @param permission - The permissions to check
     * @return true if the IPC has the granted permission
     */
    private boolean hasPermission(Context context, String permission) {
        int res = context.checkCallingOrSelfPermission(permission);
        return res == PackageManager.PERMISSION_GRANTED;

    }

    private int getMobileStrength(Context context, NetworkInfo networkInfo) {
        try {
            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            int strength = 0;
            // we are interested only in mobile.
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) { // okay now its mobile network
                // lets find the strength.
                for (CellInfo info : telephonyManager.getAllCellInfo()) {
                    if (info.isRegistered()) { // connected
                        if (info instanceof CellInfoGsm) {
                            CellSignalStrengthGsm gsm = ((CellInfoGsm) info).getCellSignalStrength();
                            strength = gsm.getDbm();
                        } else if (info instanceof CellInfoCdma) {
                            CellSignalStrengthCdma cdma = ((CellInfoCdma) info).getCellSignalStrength();
                            strength = cdma.getDbm();
                        } else if (info instanceof CellInfoLte) {
                            CellSignalStrengthLte lte = ((CellInfoLte) info).getCellSignalStrength();
                            strength = lte.getDbm();
                        } else if (info instanceof CellInfoWcdma) {
                            CellSignalStrengthWcdma wcdma = ((CellInfoWcdma) info).getCellSignalStrength(); // jelly bean mr2
                            strength = wcdma.getDbm();
                        }
                    }
                }
            }

            return strength;
        } catch (Exception e) {
            return 0;
        }

    }

    /**
     * Update the last Url in shared preference
     *
     * @param context base activity context
     * @param key     shared preference key
     * @param url     lastUrl stored
     */
    public void setStringSharedPreferenceLastURL(Context context, String key, String url) {
        String str = getStringSharedPreference(context, key);
        if (str.equalsIgnoreCase("")) {
            str = url;
        } else if (!str.contains("||")) {
            str = str + "||" + url;
        } else {
            StringTokenizer st = new StringTokenizer(str, "||");
            st.nextToken();
            str = st.nextToken() + "||" + url;
        }
        storeInSharedPreferences(context, key, str);
    }

    /**
     * @param context base activity context
     * @param key     shared preferance key
     * @return value for the provided key
     */
    public String getStringSharedPreference(Context context, String key) {
        SharedPreferences sharedPreferences = (context.getSharedPreferences(CB_PREFERENCE, Activity.MODE_PRIVATE));
        return sharedPreferences.getString(key, "");
    }


    /**
     * @param context base activity context
     * @param key     shared preferance key
     * @param value for the provided key
     */
    public void setStringSharedPreference(Context context, String key,String value) {
        SharedPreferences.Editor sharedPreferencesEditor = (context.getSharedPreferences(CB_PREFERENCE, Activity.MODE_PRIVATE)).edit();
        sharedPreferencesEditor.putString(key, value);
        sharedPreferencesEditor.commit();
    }


    /**
     * delete the given key from shared preference
     *
     * @param context base activity context
     * @param key     shared preference key
     */
    public void deleteSharedPrefKey(Context context, String key) {
        try {
            SharedPreferences.Editor sharedPreferencesEditor = (context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE)).edit();
            sharedPreferencesEditor.remove(key);
            sharedPreferencesEditor.apply();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * store the value for given key in shared Preference
     *
     * @param context base activity context
     * @param key     shared preference key
     * @param value   value for the given key
     */
    public void storeInSharedPreferences(Context context, String key, String value) {
        SharedPreferences sharedPreferences = context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.putString(key, value);
        editor.apply();
    }

    /**
     * @param context of base activity
     * @param key     key of share preference
     */
    public void removeFromSharedPreferences(Context context, String key) {
        SharedPreferences sharedPreferences = context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.remove(key);
        editor.apply();
    }

    /**
     * **
     * Return the drawble for the given resourse ID
     *
     * @param context context of base activity
     * @param resID   resourse ID of the drawable
     * @return bank icon drawable
     */
    public Drawable getDrawableCB(Context context, int resID) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            return context.getResources().getDrawable(resID);
        } else {
            return context.getResources().getDrawable(resID, context.getTheme());
        }
    }

    /**
     * Cancel timer
     *
     * @param timer timer object
     */
    public void cancelTimer(Timer timer) {
        if (timer != null) {
            timer.cancel();
            timer.purge();
        }
    }

    /**
     * Read input stream from given file
     * base activity context
     *
     * @param fileName    file name
     * @param contextMode define access to file
     * @return data read from file
     */
    public String readFileInputStream(Context mContext, String fileName, int contextMode) {//how to handle first parameter as it can be both context or activity
        int c;
        String temp = "";
        try {
            File file = new File(mContext.getFilesDir(), fileName);
            if (!file.exists()) { // create the file if not created yet.
                mContext.openFileOutput(fileName, contextMode);
            }
            FileInputStream fileInputStream = mContext.openFileInput(fileName);
            while ((c = fileInputStream.read()) != -1) {
                temp = temp + Character.toString((char) c);
            }
            fileInputStream.close();
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return temp;


    }

    /**
     * Write the input stream to file
     *
     * @param inputStream InputStream
     * @param context     base activity
     * @param fileName    file to which data to be written
     * @param contextMode define access to file
     */
    public void writeFileOutputStream(InputStream inputStream, Context context, String fileName, int contextMode) {

        try {
            GZIPInputStream responseInputStream = new GZIPInputStream(inputStream);
            byte[] buf = new byte[1024];
            int len;
            FileOutputStream outputStream = context.openFileOutput(fileName, contextMode);
            while ((len = responseInputStream.read(buf)) > 0) {
                outputStream.write(buf, 0, len);
            }
            responseInputStream.close();
            outputStream.close();
        } catch (IOException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    /**
     * reset payuID in cookie
     */
    public void resetPayuID(){
        // Reset PAYUID Cookie
        clearCookie();

        // set PAYUID empty(when OLD payuID is picked)
        /*CookieManager cookieManager = CookieManager.getInstance();
        cookieManager.setCookie(CBConstant.PAYU_DOMAIN, CBConstant.PAYUID+"="); // setting to empty string*/
    }

    /**
     * Return list of cookie of the given domain
     * @param appContext application context
     * @param domainName name of domain
     * @return cookie string
     */
    public String getCookieList(Context appContext,String domainName){
        String cookieString = "";

        try {
            CookieManager cookieManager = CookieManager.getInstance();
            if (Build.VERSION.SDK_INT < 21) {
                CookieSyncManager.createInstance(appContext);
                CookieSyncManager.getInstance().sync();
            }
            String cookies = cookieManager.getCookie(domainName); // return string of all cookie for the given domain
            if (cookies != null) {
                String[] temp = cookies.split(";");
                for (String ar1 : temp) {
                        String[] cookiesNameValue = ar1.split("=");
                        cookieString+=(cookiesNameValue[0]+"="+cookiesNameValue[1]+";");
                }
            }
            if(cookieString.length()>0){
                cookieString = cookieString.substring(0,cookieString.length()-1);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return cookieString;
    }

    /**
     * Return Value of given cookie name
     *
     * @param cookieName name of cookie
     * @return
     */
    public String getCookie(String cookieName, Context context) {

        String cookieValue = "";
        try {

            String siteName = CBConstant.PAYU_DOMAIN; // domain name for cookie
            CookieManager cookieManager = CookieManager.getInstance();
            if (Build.VERSION.SDK_INT < 21) {
                CookieSyncManager.createInstance(context);
                CookieSyncManager.getInstance().sync();
            }
            String cookies = cookieManager.getCookie(siteName); // return string of all cookie for the given domain
            if (cookies != null) {
                String[] temp = cookies.split(";");
                for (String ar1 : temp) {
                    if (ar1.contains(cookieName)) {
                        String[] temp1 = ar1.split("=");
                        cookieValue = temp1[1];
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return cookieValue;
    }

    // Todo: Remove this method use  public HashMap getDataFromPostData(String postData) instead
    @Deprecated
    public String getDataFromPostData(String postData, String key) {
        String[] list = postData.split("&");
        for (String item : list) {
            String[] items = item.split("=");
            if (items.length >= 2) {
                String id = items[0];
                if (id.equalsIgnoreCase(key)) {
                    return items[1];
                }
            }
        }
        return "";
    }

    /**
     * Using Hash map to get all the post data key and value.
     * Later we can use this methods to parase non payu's post data.
     *
     * @param postData
     * @return
     */

    public HashMap<String, String> getDataFromPostData(String postData) {
        HashMap<String, String> postParamsMap = new HashMap<>();
        if (null != postData) {
            StringTokenizer tokens = new StringTokenizer(postData, "&");
            while (tokens.hasMoreTokens()) {
                String[] keyValue = tokens.nextToken().split("=");
                if (null != keyValue && keyValue.length > 0 && null != keyValue[0]) {
                    postParamsMap.put(keyValue[0], keyValue.length > 1 ? keyValue[1] : "");
                }
            }
        }
        return postParamsMap;
    }

  /*  NotificationCompat.Builder mBuilder = new NotificationCompat.Builder(this);
    String snoozeNotificationTxnVerifiedBigText = verifyApiStatus == 1 ? customBrowserConfig.getSurePayNotificationTransactionVerifiedHeader() + "\n\n" + customBrowserConfig.getSurePayNotificationTransactionVerifiedBody() : customBrowserConfig.getSurePayNotificationTransactionNotVerifiedHeader() + "\n\n" + customBrowserConfig.getSurePayNotificationTransactionNotVerifiedBody();
    mBuilder.setContentTitle(verifyApiStatus == 1 ? customBrowserConfig.getSurePayNotificationTransactionVerifiedTitle() : customBrowserConfig.getSurePayNotificationTransactionNotVerifiedTitle())
            .setContentText(verifyApiStatus == 1 ? customBrowserConfig.getSurePayNotificationTransactionVerifiedHeader() : customBrowserConfig.getSurePayNotificationTransactionNotVerifiedHeader())
            .setSmallIcon(customBrowserConfig.getSurePayNotificationIcon())
            .setAutoCancel(true)
    .setPriority(NotificationCompat.PRIORITY_HIGH)
    .setDefaults(Notification.DEFAULT_VIBRATE)
    .setStyle(new NotificationCompat.BigTextStyle().bigText(snoozeNotificationTxnVerifiedBigText));

*/
    public void showNotification(Context context, Intent intent, String title, String txt, int smallIcon, boolean autoCancel, android.support.v4.app.NotificationCompat.Style style,int color, int ID) {
        NotificationCompat.Builder mBuilder = new NotificationCompat.Builder(context);
        mBuilder.setContentTitle(title)
                .setContentText(txt)
                .setSmallIcon(smallIcon)
                .setPriority(NotificationCompat.PRIORITY_HIGH)
                .setDefaults(Notification.DEFAULT_VIBRATE);

        if(autoCancel){
            mBuilder.setAutoCancel(autoCancel);
        }

        if(style!=null){
            mBuilder.setStyle(style);
        }

        if(color!=-1){
            mBuilder.setColor(color);
        }
        PendingIntent notifyPendingIntent = PendingIntent.getActivity(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);
        mBuilder.setContentIntent(notifyPendingIntent);
        NotificationManager mNotificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        mNotificationManager.notify(ID, mBuilder.build());


    }

    /**
     * TODO We need to init whitelisted urls from shared preference
     * public void initMRSettingsFromSharedPreference(Context context) {
     // Get setting MR Enabled from SP
     boolean isMREnabled = SharedPreferenceUtil.getBooleanFromSharedPreference(context, MagicRetryFragment.SP_MR_FILE_NAME, MagicRetryFragment.SP_IS_MR_ENABLED, !disableMagicRetry);
     // Set MR disable/enable
     disableMagicRetry(!isMREnabled);
     // Get list of whitelisted urls from SP
     String whiteListedUrls = SharedPreferenceUtil.getStringFromSharedPreference(context, MagicRetryFragment.SP_MR_FILE_NAME, MagicRetryFragment.SP_WHITELISTED_URLS, "");
     MagicRetryFragment.processAndAddWhiteListedUrls(whiteListedUrls);
     }**/


    /**
     * Helper method to parse snooze config and keep it in shard pref.
     * This method should be called from {@link com.payu.custombrowser.Bank#setSnoozeConfig(String)}
     * key - url
     * value - percentage||timeout
     * sample value of snoozeConfig {"default":[{"url":"default_payment_url","time_out":"5","progress_percent":"20"}],"0MQaQP":[{"url":"url21||url22","time_out":"5","progress_percent":"50"},{"url":"url23","time_out":"5","progress_percent":"20"}]}
     *
     * @param context      Activity or application context
     * @param snoozeConfig actual json string given by initialize js.
     */
    public SnoozeConfigMap storeSnoozeConfigInSharedPref(Context context, String snoozeConfig) {
        SnoozeConfigMap snoozeConfigMap = new SnoozeConfigMap();
        try {
            JSONObject snoozeConfigObject = new JSONObject(snoozeConfig);

            // Deleting the existing config.
            SharedPreferenceUtil.removeAllFromSharedPref(context, CBConstant.SNOOZE_SHARED_PREF);
            // storing the default values first.
            snoozeConfigMap = storeSnoozeConfigInSharedPref(context, snoozeConfigObject.getJSONArray("default"), snoozeConfigMap);
            // Removing default from snoozeConfig.
            snoozeConfigObject.remove("default");
            Iterator<String> snoozeConfigIterator = snoozeConfigObject.keys();
            if (snoozeConfigIterator.hasNext()) {
                snoozeConfigMap = storeSnoozeConfigInSharedPref(context, snoozeConfigObject.getJSONArray(snoozeConfigIterator.next()), snoozeConfigMap);
            }

        } catch (JSONException e) {
            e.printStackTrace();
        }
        return snoozeConfigMap;
    }

    private SnoozeConfigMap storeSnoozeConfigInSharedPref(Context context, JSONArray configArray, SnoozeConfigMap snoozeConfigMap) {
        String urlCollections = "";
        String progressPercent = "";
        String timeOut = "";
        StringTokenizer snoozeTokenizer;
        String url = "";
        int surePayEnableMode = CBConstant.ENABLE_WARN_AND_FAIL_MODE;
        try {
            for (int i = 0, snoozeDefaultArrayLength = configArray.length(); i < snoozeDefaultArrayLength; i++) {
                JSONObject currentSnoozeConfigObject = configArray.getJSONObject(i);
                // more than one url can have same config like: "url":"url21||url22"
                urlCollections = currentSnoozeConfigObject.get(CBConstant.URL).toString();
                progressPercent = currentSnoozeConfigObject.get(CBConstant.PROGRESS_PERCENT).toString();
                timeOut = currentSnoozeConfigObject.get(CBConstant.TIME_OUT).toString();
                if(currentSnoozeConfigObject.has(CBConstant.DISABLE_SP_FOR)) {
                    JSONObject surePayEnableModeObject = currentSnoozeConfigObject.getJSONObject(CBConstant.DISABLE_SP_FOR);
                    surePayEnableMode =  getSurePayEnableMode(surePayEnableModeObject);
                }
                // Using tokenizer because split requires to escape pipe: using pipe because url might have , . ; # etc.
                snoozeTokenizer = new StringTokenizer(urlCollections, CBConstant.CB_DELIMITER);
                while (snoozeTokenizer.hasMoreTokens()) {
                    url = snoozeTokenizer.nextToken();
                    // key : url
                    // value : progressPercentage||timeout||surepayEnableMode
                    SharedPreferenceUtil.addStringToSharedPreference(context, CBConstant.SNOOZE_SHARED_PREF, url.contentEquals("*") ? CBConstant.DEFAULT_PAYMENT_URLS : url.trim(), progressPercent.trim() + CBConstant.CB_DELIMITER + timeOut.trim() + CBConstant.CB_DELIMITER + surePayEnableMode);
                    snoozeConfigMap.put(url.contentEquals("*") ? CBConstant.DEFAULT_PAYMENT_URLS : url.trim(), progressPercent.trim() + CBConstant.CB_DELIMITER + timeOut.trim() + CBConstant.CB_DELIMITER + surePayEnableMode);
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return snoozeConfigMap;

    }

    /**
     * Method to convert the shared pref's hash map value to SnoozeConfigMap value.
     * Using this method because {@link SharedPreferenceUtil#getSharedPrefMap(Context, String)} is a generic method which returns Map.
     *
     * @param snoozeMap map of shardperf
     * @return snoozeConfigMap
     */
    public SnoozeConfigMap convertToSnoozeConfigMap(Map<String, ?> snoozeMap) {
        SnoozeConfigMap snoozeConfigMap = new SnoozeConfigMap();
        for (Map.Entry<String, ?> entry : snoozeMap.entrySet()) {
            snoozeConfigMap.put(entry.getKey(), entry.getValue());
        }
        return snoozeConfigMap;
    }

    /**
     * Method to add all the Error codes. - We can have these config from JS in future.
     * Contains list of error codes where we should show sure pay dialog in fail mode,
     * more info at https://developer.android.com/reference/android/webkit/WebViewClient.html#ERROR_AUTHENTICATION
     * @return set of error codes
     */
    public Set<String> getSurePayErrorCodes(){
        Set<String> showSurePayErrorCodesSet  = new HashSet<>();
        // showSurePayErrorCodesSet.add("-6"); // ERROR_CONNECT
        showSurePayErrorCodesSet.add("-7"); // ERROR_IO
        showSurePayErrorCodesSet.add("-8"); // ERROR_TIMEOUT
        showSurePayErrorCodesSet.add("-15"); // ERROR_TOO_MANY_REQUESTS
        return showSurePayErrorCodesSet;
    }


    /**
     * Method to get sure pay disable config.
     * 0 - don't disable sure pay.
     * 1 - only warn mode disabled.
     * 2 - only fail mode disabled
     * 3 - both warn and fail are disabled.
     * @param jsonObject
     * @return {'warn' : 'true', 'fail' : 'false'}
     */
    private int getSurePayEnableMode(JSONObject jsonObject){
        try {
            if (jsonObject.has(CBConstant.WARN) && jsonObject.getBoolean(CBConstant.WARN)
                    && jsonObject.has(CBConstant.FAIL) && jsonObject.getBoolean(CBConstant.FAIL)) {
                return CBConstant.DISABLE_WARN_AND_FAIL_MODE;
            } else if (jsonObject.has(CBConstant.FAIL) && jsonObject.getBoolean(CBConstant.FAIL)) {
                return CBConstant.DISABLE_FAIL_MODE;
            } else if (jsonObject.has(CBConstant.WARN) && jsonObject.getBoolean(CBConstant.WARN)) {
                return CBConstant.DISABLE_WARN_MODE;
            }
        }catch (Exception e){ // oops something went wrong, dont disable sure pay
            e.printStackTrace();
            return CBConstant.ENABLE_WARN_AND_FAIL_MODE;
        }
        // By default dont disable sure pay
        return CBConstant.ENABLE_WARN_AND_FAIL_MODE;
    }


    /**
     * Method returns sure pay disable status, by parsing the snooze config map.
     * @param snoozeConfigMap given snooze config map from js or shared pref
     * @param webViewUrl current url.
     * @return sure pay disable status
     */
    public int getSurePayDisableStatus(SnoozeConfigMap snoozeConfigMap, String webViewUrl){
        if(null != snoozeConfigMap && webViewUrl != null){
            int[] snoozeConfig;

            // loop through snooze config map and check given url is found
            for(Object key : snoozeConfigMap.keySet()){
                if(webViewUrl.startsWith(key.toString())){ // given url is in snoozeConfig,
                    snoozeConfig = snoozeConfigMap.getPercentageAndTimeout(key.toString());
                    return snoozeConfig[2];
                }
            }

            // We have not found the given url, lets return the default url value from the js.
            // This takes care of * as well!
            snoozeConfig = snoozeConfigMap.getPercentageAndTimeout(CBConstant.DEFAULT_PAYMENT_URLS);
            return snoozeConfig[2];
        }

        // we don't have snooze config map value lets return default snooze mode.
        return CBConstant.ENABLE_WARN_AND_FAIL_MODE;
    }

    /**
     * Function to get value of a key from json string.
     * @author: Minie Sahni.
     * @param data - input string .
     * @param key - input key
     * @return - value of the key.
     * @throws JSONException
     */
    public static String getValueOfJSONKey(String data, String key) throws JSONException {
        final JSONObject jsonObject = new JSONObject(data);
        if (jsonObject.has(key)) {
            return jsonObject.get(key).toString();
        } else {
            throw new JSONException("Key not found");
        }
    }

    /**
     * Method to launch play store app
     * @param context
     */
    public static void launchPlayStore(Context context){
        launchPlayStore(context, null, null);
    }

    /**
     * Method to launch play store with a specific app name.
     * @param context
     * @param url
     * @param webViewVersion
     */
    public static void launchPlayStore(Context context, String url, String webViewVersion){

        String packageName = getPackageNameFromPlayStoreLink(url);

        if(null == packageName){
            packageName = "";
        } else {
            packageName = "details?id=" + packageName;
        }

        setWebViewVersionInSP(context, webViewVersion);

        try{
            context.startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("market://" + packageName)));
        } catch (ActivityNotFoundException e){
            context.startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("https://play.google.com/store/apps/" + packageName)));
        }
    }

    /**
     * Method to check given url is a play store url
     * @param url
     * @return
     */
    public static boolean isPlayStoreUrl(String url) {
        return url.startsWith(CBConstant.PLAY_STORE_URL) || url.startsWith(CBConstant.PLAY_STORE_MARKET_URI);
    }


    /**
     * Method to get the package name from play store url
     * Using regex (?<=[?&]id=)[^&]+ to do so
     * the extra bracket in the regex is for making it as a group. ((?<=[?&]id=)[^&]+)
     * @param url
     * @return
     */
    public static String getPackageNameFromPlayStoreLink(String url){
        Matcher m = Pattern.compile("((?<=[?&]id=)[^&]+)").matcher(url);
        return m.find() ? m.group(1) : null;
    }

    /**
     * method to find the given webview version
     * @param view
     * @return
     */
    public static String getWebViewVersion(WebView view) {
        String userAgent =  view.getSettings().getUserAgentString();
        Matcher m = Pattern.compile("(Chrome\\/(.*?)\\s)").matcher(userAgent);
        return m.find() ? m.group(2) : null;
    }

    /**
     * method to set the webview version in SP
     * @param context
     * @param webViewVersion
     */
    public static void setWebViewVersionInSP(Context context, String webViewVersion){
        if(null != webViewVersion){
            SharedPreferenceUtil.addStringToSharedPreference(context, CB_PREFERENCE, CBConstant.WEBVIEW_VERSION, webViewVersion);
        }
    }

    /**
     * Method to get the previously stored webview/chrome version from sp
     * @param context
     * @return
     */
    public static String getWebViewVersionFromSP(Context context){
        return SharedPreferenceUtil.getStringFromSharedPreference(context, CB_PREFERENCE, CBConstant.WEBVIEW_VERSION, "");
    }

    /**
     * Method to get the Base64 Decoded String value
     *
     * @param data - Base64 encoded String
     * @return Base64 decoded String
     */
    public static String getBase64DecodedString(String data) {
        return new String(Base64.decode(data, Base64.DEFAULT));
    }
}
