// The MIT License (MIT)
//
// Copyright (c) 2014-2015 PayU India
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package com.payu.custombrowser.analytics;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import com.payu.custombrowser.Bank;
import com.payu.custombrowser.util.CBConstant;
import com.payu.custombrowser.util.CBUtil;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.FileOutputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Created by amitkumar on 16/01/15.
 * Helps payu to keep track of user interactions with bank page and more.
 * For later fix: make it as singleton.
 */
public class CBAnalytics {
    private static final String TEST_URL = "https://mobiletest.payu.in/merchant/MobileAnalytics";
    private static final String PRODUCTION_URL = "https://info.payu.in/merchant/MobileAnalytics";
    private String fileName;
    private static final long TIMER_DELAY = 5000;
    private static CBAnalytics INSTANCE;
    private final Context mcontext;
    public final static String ANALYTICS_URL = Bank.DEBUG ? TEST_URL : PRODUCTION_URL;
    private volatile  boolean mainFileLocked = false;
    private Timer mTimer;
    private CBUtil cbUtil;
    // Lock for buffer file(shared preference)
    private volatile boolean mBufferLock;
    // analytics buffer file - shared preference key
    private String ANALYTICS_BUFFER_KEY = "analytics_buffer_key";


    /**
     * Constructor
     *
     * @param context  instance of base activity
     * @param filename name of file for analytics event
     */
    private CBAnalytics(final Context context, String filename) {
        mcontext = context;
        fileName = filename;
        cbUtil = new CBUtil();
        final Thread.UncaughtExceptionHandler defaultUEH = Thread.getDefaultUncaughtExceptionHandler();
        Thread.UncaughtExceptionHandler _unCaughtExceptionHandler = new Thread.UncaughtExceptionHandler() {
            @Override
            public void uncaughtException(Thread thread, Throwable ex) {
                while (mainFileLocked) ;
                setLock();
                try {
                    FileOutputStream fileOutputStream = mcontext.openFileOutput(fileName, Context.MODE_PRIVATE);
                    int c = cbUtil.getStringSharedPreference(mcontext, ANALYTICS_BUFFER_KEY).length();
                    if (c > 0) {
                        JSONArray jsonArray = new JSONArray();
                        JSONArray bufferJsonArray = new JSONArray(cbUtil.getStringSharedPreference(mcontext, ANALYTICS_BUFFER_KEY).toString());
                        for (int i = 0; i < bufferJsonArray.length(); i++) {
                            JSONObject jsonObject = bufferJsonArray.getJSONObject(i);
                            jsonArray.put(jsonArray.length(), jsonObject);
                        }
                        fileOutputStream.write(jsonArray.toString().getBytes());
                        cbUtil.deleteSharedPrefKey(context, ANALYTICS_BUFFER_KEY);
                    }
                    fileOutputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                } catch (Exception e) {
                    e.printStackTrace();
                }
                releaseLock();
                defaultUEH.uncaughtException(thread, ex);
            }
        };
        Thread.setDefaultUncaughtExceptionHandler(_unCaughtExceptionHandler);
    }

  /*  *//**
     * keep only one object
     *
     * @param context  instance of base activity
     * @param fileName of file for aanalytics events
     * @return CBAnalytics instance.
     */
    public static CBAnalytics getInstance(Context context, String fileName) {
        if (INSTANCE == null) {
            synchronized (CBAnalytics.class) {
                if (INSTANCE == null) {
                    INSTANCE = new CBAnalytics(context, fileName);

                }
            }
        }
        return INSTANCE;
    }

    /**
     * Adding log information to a local file.
     *
     * @param msg json string
     */
    public void log(final String msg) {

        if(isOnline()) {
            // Reset the timer and push the data to server
            resetTimer();
        }
        // check if main file is lock
        if (mainFileLocked) {
            new Thread(new Runnable() {
                @Override
                public void run() {
                    try {
                        // wait till buffer file is in use
                        while (mBufferLock) ;
                        //read events from buffer file
                        String str = cbUtil.getStringSharedPreference(mcontext, ANALYTICS_BUFFER_KEY);
                        JSONArray jsonArray;
                        if (str != null && !str.equalsIgnoreCase("")) {
                            jsonArray = new JSONArray(str);
                        } else {
                            jsonArray = new JSONArray();
                        }

                        JSONObject newAnalyticsEvent = new JSONObject(msg);

                        // add new analytics event to buffer file
                        jsonArray.put(newAnalyticsEvent);
                        cbUtil.setStringSharedPreference(mcontext, ANALYTICS_BUFFER_KEY, jsonArray.toString());

                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }).start();
        } else {
            // read main file add new analytics event to it
            new Thread(new Runnable() {
                @Override
                public void run() {
                    while(mainFileLocked);
                    setLock();
                    try {
                        JSONObject newobject = new JSONObject(msg);
                        JSONArray jsonArray;
                        String temp = cbUtil.readFileInputStream(mcontext, fileName, Context.MODE_PRIVATE);
                        if (temp == null || temp.equalsIgnoreCase(""))
                            jsonArray = new JSONArray();
                        else
                            jsonArray = new JSONArray(temp);

                        FileOutputStream fileOutputStream = mcontext.openFileOutput(fileName, Context.MODE_PRIVATE);
                        jsonArray.put(jsonArray.length(), newobject);
                        fileOutputStream.write((jsonArray.toString()).getBytes());
                        fileOutputStream.close();

                    } catch (IOException e) {
                        e.printStackTrace();
                    } catch (JSONException e) {
                        e.printStackTrace();
                    } catch (Exception e) {
                        e.printStackTrace();
                    }finally {
                        releaseLock();
                    }
                }
                }).start();
        }

    }

    /**
     * Open file and read the content.
     * Send the contents to payu server
     * command should be sdkWs and var1 should be the actual data.
     */
    private void resetTimer() {

        if (mTimer != null) {
            mTimer.cancel();
        }
        mTimer = new Timer();
        mTimer.schedule(new TimerTask() {
            @Override
            public void run() {
                while (mainFileLocked);
                setLock();

                if (isOnline()) {
                    String temp = "";
                    try {
                        temp = cbUtil.readFileInputStream(mcontext, fileName, Context.MODE_PRIVATE);
                    } finally {
                        // temp is a json array.
                        // mBuffer is a json array again.
                        try {
                            // file data converted to json array
                            JSONArray tempJsonArray;
                            if (temp != null && !temp.equalsIgnoreCase("")) {
                                tempJsonArray = new JSONArray(temp);
                            } else {
                                tempJsonArray = new JSONArray();
                            }

                            // Copy buffer file data in file
                            if (cbUtil.getStringSharedPreference(mcontext, ANALYTICS_BUFFER_KEY).length() > 1) {
                                // lock buffer file
                                mBufferLock = true;
                                JSONArray analyticsBufferArray = new JSONArray(cbUtil.getStringSharedPreference(mcontext, ANALYTICS_BUFFER_KEY));
                                tempJsonArray = copyBufferToFile(tempJsonArray, analyticsBufferArray);
                            }

                            // upload main file events to server
                            if (tempJsonArray.length() > 0) {
                                String postData = "command=EventAnalytics&data=" + tempJsonArray.toString();
                                HttpURLConnection conn = cbUtil.getHttpsConn(ANALYTICS_URL, postData, CBConstant.HTTP_TIMEOUT);

                                if (conn != null) {
                                    int responseCode = conn.getResponseCode();
                                    if (responseCode == HttpURLConnection.HTTP_OK) {

                                        if (conn.getInputStream() != null) {
                                            // responseStringBuffer - response from server
                                            StringBuffer responseStringBuffer = CBUtil.getStringBufferFromInputStream(conn.getInputStream());
                                            if (responseStringBuffer != null) {
                                                JSONObject jsonObject = new JSONObject(responseStringBuffer.toString());
                                                // if we are able to post the data to server delete file
                                                if (jsonObject.has("status")) {
                                                    mcontext.deleteFile(fileName);
                                                }
                                            }
                                        }

                                    }
                                }
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }

                // release lock on main file
                releaseLock();

                // reset timer of buffer has some events to log
                if (cbUtil.getStringSharedPreference(mcontext, ANALYTICS_BUFFER_KEY).length() > 1) {
                    resetTimer();
                }



            }
        }, TIMER_DELAY);
    }

    /**
     * set lock on main file
     */
    private synchronized void setLock() {
        while(mainFileLocked);
        mainFileLocked = true;
    }

    /**
     * release lock on main file
     */
    private void releaseLock() {
        mainFileLocked = false;
    }


    /**
     * Check for internet connection
     *
     * @return true if internet connection if found as false
     */
    private boolean isOnline() {
        ConnectivityManager cm =
                (ConnectivityManager) mcontext.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo netInfo = cm.getActiveNetworkInfo();
        return netInfo != null && netInfo.isConnectedOrConnecting();
    }


    /**
     * @return analytics timer object
     */
    public Timer getmTimer() {
        return mTimer;
    }


    /**
     * Copy buffer analytics file to main analytics file
     *
     * @param fileJsonArray   main analytics file events
     * @param bufferJsonArray buffer analytics file events
     * @return fileJsonArray + bufferJsonArray(if buffer file data is successfully) else return fileJsonArray
     */
    private JSONArray copyBufferToFile(JSONArray fileJsonArray, JSONArray bufferJsonArray) {
        FileOutputStream fileOutputStream = null;
        try {

            // combining both main file json array and buffer array
            JSONArray mergeJsonArray = new JSONArray(fileJsonArray.toString());
            for (int i = 0; i < bufferJsonArray.length(); i++) {
                JSONObject jsonObject = bufferJsonArray.getJSONObject(i);
                mergeJsonArray.put(jsonObject);
            }
            fileOutputStream = mcontext.openFileOutput(fileName, Context.MODE_PRIVATE);
            fileOutputStream.write((mergeJsonArray.toString()).getBytes());
            cbUtil.deleteSharedPrefKey(mcontext,ANALYTICS_BUFFER_KEY);
            return mergeJsonArray;
        } catch (Exception e) {
            e.printStackTrace();
            // return main file array if not able to merge main and buffer file
            return fileJsonArray;
        } finally {
            if (fileOutputStream != null) {
                try {
                    fileOutputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            //release lock on buffer file
            mBufferLock = false;
        }
    }
}
