// The MIT License (MIT)
//
// Copyright (c) 2014-2015 PayU India
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package com.payu.custombrowser.util;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.graphics.drawable.Drawable;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkInfo;
import android.net.TrafficStats;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.telephony.CellInfo;
import android.telephony.CellInfoCdma;
import android.telephony.CellInfoGsm;
import android.telephony.CellInfoLte;
import android.telephony.CellInfoWcdma;
import android.telephony.CellSignalStrengthCdma;
import android.telephony.CellSignalStrengthGsm;
import android.telephony.CellSignalStrengthLte;
import android.telephony.CellSignalStrengthWcdma;
import android.telephony.TelephonyManager;
import android.util.DisplayMetrics;
import android.view.View;
import android.view.animation.AlphaAnimation;


import com.payu.custombrowser.BuildConfig;
import com.payu.custombrowser.R;

import org.json.JSONObject;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;

/**
 * Created by minie on 29/6/15.
 * Contains cb's utility functions.
 */
public class CBUtil {

    private final static String CB_PREFERENCE = "com.payu.custombrowser.payucustombrowser";
    private static SharedPreferences sharedPreferences;

    public static String getLogMessage(Context context, String key,String value,String bank,String sdkMerchantKey,String trnxID) {
        try {
            JSONObject eventAnalytics = new JSONObject();
            eventAnalytics.put(CBAnalyticsConstant.TRANSACTION_ID ,trnxID);
            eventAnalytics.put(CBAnalyticsConstant.CB_VERSION_NAME,BuildConfig.VERSION_NAME);
            eventAnalytics.put(CBAnalyticsConstant.PAKAGE_NAME,context.getPackageName());
            eventAnalytics.put(CBAnalyticsConstant.BANK_NAME,(bank==null?"":bank));
            eventAnalytics.put(CBAnalyticsConstant.KEY,key);
            eventAnalytics.put(CBAnalyticsConstant.VALUE, value);
            eventAnalytics.put(CBAnalyticsConstant.MERCHANT_KEY, sdkMerchantKey);
            return eventAnalytics.toString();
        }catch(Exception e){
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Decode the downloaded javascript files.
     * should be used to decode initialize.js and bank specific js.
     *
     * @param fileInputStream reference of fileinputstream
     * @return decoded string.
     */

    public  static String decodeContents(FileInputStream fileInputStream) {
        StringBuilder decoded = new StringBuilder();
        try {
            int c;
            int i = 0;
            while ((c = fileInputStream.read()) != -1) {
                if (i % 2 == 0) {
                    decoded.append((char) (c - ((i % 5) + 1)));
                } else {
                    decoded.append((char) (c + ((i % 5) + 1)));
                }
                i++;
            }
            fileInputStream.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return decoded.toString();
    }

    public static  boolean getBooleanSharedPreference(String key,Activity mActivity) {
        sharedPreferences = mActivity.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE);
        return sharedPreferences.getBoolean(key, false);
    }

    public static void setBooleanSharedPreference(String key, boolean value,Context context ) {
        SharedPreferences.Editor editor = context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE).edit();
        editor.putBoolean(key, value);
        editor.apply();
    }

    /**
     * Function to find whether network is available or not.
     * @param context of base activity
     * @return true if network available else false
     */
    public static boolean isNetworkAvailable(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            Network[] networks = connectivityManager.getAllNetworks();
            for (Network mNetwork : networks) {
                NetworkInfo networkInfo = connectivityManager.getNetworkInfo(mNetwork);
                if (networkInfo.getState().equals(NetworkInfo.State.CONNECTED)) {
                    return true;
                }
            }
        }else {
            //noinspection deprecation
            NetworkInfo[] info = connectivityManager.getAllNetworkInfo();
            if (info != null) {
                for (NetworkInfo anInfo : info) {
                    if (anInfo.getState() == NetworkInfo.State.CONNECTED) {
                        return true;
                    }
                }
            }

        }
        return false;
    }

    public static String getDeviceDensity(Activity activity){
        DisplayMetrics metrics = new DisplayMetrics();
        activity.getWindowManager().getDefaultDisplay().getMetrics(metrics);
        return metrics.densityDpi+"";
    }

    /**
     * @param activity context of base activity
     */
    private static void getDownloadSpeed(Activity activity){
        String[] testing;
        testing = new String[2];
        long BeforeTime = System.currentTimeMillis();
        long TotalTxBeforeTest = TrafficStats.getTotalTxBytes();
        long TotalRxBeforeTest = TrafficStats.getTotalRxBytes();

        long TotalTxAfterTest = TrafficStats.getTotalTxBytes();
        long TotalRxAfterTest = TrafficStats.getTotalRxBytes();
        long AfterTime = System.currentTimeMillis();

        double TimeDifference = AfterTime - BeforeTime;

        double rxDiff = TotalRxAfterTest - TotalRxBeforeTest;
        double txDiff = TotalTxAfterTest - TotalTxBeforeTest;

        if((rxDiff != 0) && (txDiff != 0)) {
            double rxBPS = (rxDiff / (TimeDifference/1000)); // total rx bytes per second.
            double txBPS = (txDiff / (TimeDifference/1000)); // total tx bytes per second.
            testing[0] = String.valueOf(rxBPS) + "bps. Total rx = " + rxDiff;
            testing[1] = String.valueOf(txBPS) + "bps. Total tx = " + txDiff;
        }
        else {
            testing[0] = "No uploaded or downloaded bytes.";
        }
    }
    public static  void setAlpha(float alpha,View view){
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.HONEYCOMB) {
            final AlphaAnimation animation = new AlphaAnimation(alpha, alpha);
            animation.setDuration(10);
            animation.setFillAfter(true);
            view.startAnimation(animation);
        } else {
            view.setAlpha(alpha);
        }
    }

    public static String getNetworkStatus(Activity activity) {
        try {
            if (null != activity && !activity.isFinishing()) {
                ConnectivityManager cm = (ConnectivityManager) activity.getSystemService(Context.CONNECTIVITY_SERVICE);
                NetworkInfo info = cm.getActiveNetworkInfo();
                if (info == null || !info.isConnected())
                    return "Not connected"; //not connected
                if (info.getType() == ConnectivityManager.TYPE_WIFI)
                    return "WIFI";
                if (info.getType() == ConnectivityManager.TYPE_MOBILE) {
                    int networkType = info.getSubtype();
                    switch (networkType) {
                        case TelephonyManager.NETWORK_TYPE_GPRS:
                            return "GPRS";
                        case TelephonyManager.NETWORK_TYPE_EDGE:
                            return "EDGE";
                        case TelephonyManager.NETWORK_TYPE_CDMA:
                            return "CDMA";
                        case TelephonyManager.NETWORK_TYPE_1xRTT:
                        case TelephonyManager.NETWORK_TYPE_IDEN: //api<8 : replace by 11
                            return "2G";
                        case TelephonyManager.NETWORK_TYPE_UMTS:
                        case TelephonyManager.NETWORK_TYPE_EVDO_0:
                        case TelephonyManager.NETWORK_TYPE_EVDO_A:
                        case TelephonyManager.NETWORK_TYPE_HSDPA:
                        case TelephonyManager.NETWORK_TYPE_HSUPA:
                        case TelephonyManager.NETWORK_TYPE_HSPA:
                            return "HSPA";
                        case TelephonyManager.NETWORK_TYPE_EVDO_B: //api<9 : replace by 14
                        case TelephonyManager.NETWORK_TYPE_EHRPD:  //api<11 : replace by 12
                        case TelephonyManager.NETWORK_TYPE_HSPAP:  //api<13 : replace by 15
                            return "3G";
                        case TelephonyManager.NETWORK_TYPE_LTE:    //api<11 : replace by 13
                            return "4G";
                        default:
                            return "?";
                    }
                }
            }
        }catch (Exception e) {
            return "?";
        }
        return "?";
    }


    public static NetworkInfo getNetWorkInfo(Context mContext){
        ConnectivityManager connectivityManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo network = null;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            Network[] networks = connectivityManager.getAllNetworks();
            for (Network mNetwork : networks) {
                NetworkInfo networkInfo = connectivityManager.getNetworkInfo(mNetwork);
                if (networkInfo.getState().equals(NetworkInfo.State.CONNECTED)) {
                    network=networkInfo;
                }
            }
        }else {
            //noinspection deprecation
            NetworkInfo[] info = connectivityManager.getAllNetworkInfo();
            if (info != null) {
                for (NetworkInfo anInfo : info) {
                    if (anInfo.getState() == NetworkInfo.State.CONNECTED) {
                        network=anInfo;
                    }
                }
            }

        }
        return network;
    }
    /**
     * Checking for all possible internet providers
     * **/
    public static int getNetworkStrength(Context mContext) {
        NetworkInfo network = getNetWorkInfo(mContext);
      //  Toast.makeText(mContext, mContext.getString(R.string.please_connect_to_internet), Toast.LENGTH_SHORT).show();
        if(network!=null) {
            if (network.getTypeName().equalsIgnoreCase("MOBILE")) {
                return getMobileStrength(mContext, network);
            } else if (network.getTypeName().equalsIgnoreCase("wifi") && hasPermission(mContext, "android.permission.ACCESS_WIFI_STATE")) {
                final WifiManager wifiManager = (WifiManager) mContext.getSystemService(Context.WIFI_SERVICE);
                try {
                    final WifiInfo connectionInfo = wifiManager.getConnectionInfo();
                    if (connectionInfo != null) {
                        // connectionInfo.getRssi() - According to IEEE 802.11 documentation: Lesser negative values denotes higher signal strength.The range is between -100 to 0 dBm, closer to 0 is higher strength and vice-versa.
                        return WifiManager.calculateSignalLevel(connectionInfo.getRssi(), 5); //returning a number between 0 and 4 (i.e. numLevel-1) : the number of bars you see in toolbar.
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            return 0;
        }
        return 0;
    }

    /** Determines if the context calling has the required permission
     * @param context - the IPC context
     * @param permission - The permissions to check
     * @return true if the IPC has the granted permission
     */
    private static boolean hasPermission(Context context, String permission) {
        int res = context.checkCallingOrSelfPermission(permission);
        return res == PackageManager.PERMISSION_GRANTED;

    }


    private static  int getMobileStrength(Context  context, NetworkInfo networkInfo){
        try {
            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            int strength = 0;
                // we are interested only in mobile.
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) { // okay now its mobile network
                    // lets find the strength.
                    for (CellInfo info : telephonyManager.getAllCellInfo()) {
                        if (info.isRegistered()) { // connected
                            if (info instanceof CellInfoGsm) {
                                CellSignalStrengthGsm gsm = ((CellInfoGsm) info).getCellSignalStrength();
                                strength = gsm.getDbm();
                            } else if (info instanceof CellInfoCdma) {
                                CellSignalStrengthCdma cdma = ((CellInfoCdma) info).getCellSignalStrength();
                                strength = cdma.getDbm();
                            } else if (info instanceof CellInfoLte) {
                                CellSignalStrengthLte lte = ((CellInfoLte) info).getCellSignalStrength();
                                strength = lte.getDbm();
                            } else if (info instanceof CellInfoWcdma) {
                                CellSignalStrengthWcdma wcdma = ((CellInfoWcdma) info).getCellSignalStrength(); // jelly bean mr2
                                strength = wcdma.getDbm();
                            }
                        }
                    }
                }

            return strength;
        }catch (Exception e){
            return 0;
        }

    }

    /**
     * Update the last Url in shared preference
     *
     * @param context base activity context
     * @param key shared preference key
     * @param url lastUrl stored
     */
    public static void setStringSharedPreference(Context context, String key,String url){
        String str = getStringSharedPreference(context, key);
        if(str.equalsIgnoreCase("")){
            str=url;
        }else if(!str.contains("||")){
            str=str+"||"+url;
        }else{
            StringTokenizer st = new StringTokenizer(str,"||");
            st.nextToken();
            str=st.nextToken()+"||"+url;
        }
        storeInSharedPreferences(context, key, str);
    }

    /**
     *
     * @param context base activity context
     * @param key shared preferance key
     * @return value for the provided key
     */
    public static  String getStringSharedPreference(Context context, String key){
        SharedPreferences sharedPreferences= (context.getSharedPreferences(CB_PREFERENCE, android.app.Activity.MODE_PRIVATE));
        return sharedPreferences.getString(key,"");
    }

    /**
     * delete the given key from shared preference
     *
     * @param context base activity context
     * @param key shared preference key
     */
    public static void deleteSharedPrefKey(Context context, String key){
        try {
            SharedPreferences.Editor sharedPreferencesEditor = (context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE)).edit();
            sharedPreferencesEditor.remove(key);
            sharedPreferencesEditor.apply();
        }catch(Exception e)
        {
            e.printStackTrace();
        }
    }


    /**
     * Update the last Url
     * @param lastUrl last url in webview before transaction terminate
     * @return s:last stated url||f last finished url
     */
    public static String updateLastUrl(String lastUrl)
    {
        try {
            if (!(lastUrl.contains("||"))) {
                if (lastUrl.length() > 128)
                    return lastUrl.substring(0, 127);
                else
                    return lastUrl;
            } else {
                StringTokenizer st = new StringTokenizer(lastUrl, "||");
                String firstURl = st.nextToken();
                String secondUrl = st.nextToken();
                if (firstURl.length() > 128)
                    firstURl = firstURl.substring(0, 125);
                if (secondUrl.length() > 128)
                    secondUrl = secondUrl.substring(0, 125);
                return firstURl + "||" + secondUrl;
            }
        }catch(Exception e)
        {
            e.printStackTrace();
            return "";

        }
    }

    /**
     * store the value for given key in shared Preference
     * @param context base activity context
     * @param key shared preference key
     * @param value value for the given key
     */
    public static void storeInSharedPreferences(Context context, String key, String value){
        SharedPreferences sharedPreferences = context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.putString(key, value);
        editor.apply();
    }


    /**
     * @param context  of base activity
     * @param key key of share preference
     */
    public static void removeFromSharedPreferences(Context context, String key){
        SharedPreferences sharedPreferences = context.getSharedPreferences(CB_PREFERENCE, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.remove(key);
        editor.apply();
    }

    /**
     ***
     * Return the drawble for the given resourse ID
     *
     * @param context context of base activity
     * @param resID resourse ID of the drawable
     * @return bank icon drawable
     */
    public static Drawable getDrawableCB(Context context,int resID) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP ) {
            return context.getResources().getDrawable(resID);
        } else {
            return context.getResources().getDrawable(resID, context.getTheme());
        }
    }

    /**
     * set the analytics key for the analytics of events and device
     *
     * @param value analytics key
     */
    public static void setVariableReflection(String className, String value, String varName) {
        try {

            if (value != null && !value.trim().equals("")) {
                Class aClass = Class.forName(className);
                Field field = aClass.getDeclaredField(varName);
                field.setAccessible(true);
                field.set(null, value);
                field.setAccessible(false);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    /**
     * Filter the OTP from sms
     * @param mBankJS bank JS
     * @param msgBody message received
     * @param context base activity context
     * @return filter OTP
     */
    public static String filterSMS(JSONObject mBankJS, String msgBody, Context context){
        String mPassword=null;
        try {
            Matcher match;
            if (msgBody != null) {
                match = Pattern.compile(mBankJS.getString(context.getString(R.string.cb_detect_otp)), Pattern.CASE_INSENSITIVE).matcher(msgBody);
                if (match.find()) {
                    // we have otp sms
                    match = Pattern.compile(mBankJS.getString(context.getString(R.string.cb_find_otp)), Pattern.CASE_INSENSITIVE).matcher(msgBody);
                    if (match.find()) {
                        mPassword = match.group(1).replaceAll("[^0-9]", "");
                    }
                }
            }
        }catch(Exception e)
        {
            e.printStackTrace();
        }
        return mPassword;
    }

    /**
     * Cancel timer
     * @param timer timer object
     */
    public static void cancelTimer(Timer timer){
        if (timer != null) {
            timer.cancel();
            timer.purge();
        }
    }

    /**
     * Read input stream from given file
     * @param mContext base activity context
     * @param fileName file name
     * @param contextMode define access to file
     * @return data read from file
     */
    public static String readFileInputStream(Context mContext, String fileName, int contextMode){//how to handle first parameter as it can be both context or activity
        int c;
        String temp = "";
        try {
            File file = new File(mContext.getFilesDir(), fileName);
            if (!file.exists()) { // create the file if not created yet.
                mContext.openFileOutput(fileName, contextMode);
            }
            FileInputStream fileInputStream = mContext.openFileInput(fileName);
            while ((c = fileInputStream.read()) != -1) {
                temp = temp + Character.toString((char) c);
            }
            fileInputStream.close();
        }catch (FileNotFoundException e){
            e.printStackTrace();
        }catch (IOException e){
            e.printStackTrace();
        }catch (Exception e){
            e.printStackTrace();
        }
        return temp;


    }

    /**
     * Write the input stream to file
     * @param inputStream InputStream
     * @param activity base activity
     * @param fileName file to which data to be written
     * @param contextMode define access to file
     */
    public static void writeFileOutputStream(InputStream inputStream, Activity activity, String fileName, int contextMode){

        try {
            GZIPInputStream responseInputStream = new GZIPInputStream(inputStream);
            byte[] buf = new byte[1024];
            int len;
            FileOutputStream outputStream = activity.openFileOutput(fileName, contextMode);
            while ((len = responseInputStream.read(buf)) > 0) {
                outputStream.write(buf, 0, len);
            }
            responseInputStream.close();
            outputStream.close();
        }catch (IOException e){
            e.printStackTrace();
        }catch (Exception e){
            e.printStackTrace();
        }

    }

    /***
     * Return HttpURLConnection object
     * @param strURL postURL
     * @param postData data to be posted
     * @return connection object
     */
    public  static HttpURLConnection getHttpsConn(String strURL , String postData){
        try {
            URL url = new URL(strURL);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod("POST");
            conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            conn.setRequestProperty("Content-Length", String.valueOf(postData.length()));
            conn.setDoOutput(true);
            byte[] postParamsByte = postData.getBytes();
            conn.getOutputStream().write(postParamsByte);
            return conn;
        }catch(Exception e){
            e.printStackTrace();
            return null;
        }
    }


    /**
     * Return the connection object
     * @param strURL url
     * @return HttpURLConnection object
     */
    public  static HttpURLConnection getHttpsConn(String strURL){
        try {
            URL url=new URL(strURL);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod("GET");
            conn.setRequestProperty("Accept-Charset", "UTF-8");
            return conn;
        }catch(Exception e){
            e.printStackTrace();
            return null;
        }
    }


    /**
     * Read InputStream and return string buffer
     * @param responseInputStream input stream
     * @return StringBuffer
     */
    public static StringBuffer getStringBufferFromInputStream(InputStream responseInputStream){
        try {
            StringBuffer responseStringBuffer = new StringBuffer();
            byte[] byteContainer = new byte[1024];
            for (int i; (i = responseInputStream.read(byteContainer)) != -1; ) {
                responseStringBuffer.append(new String(byteContainer, 0, i));
            }
            return responseStringBuffer;
        }catch(Exception e){
            e.printStackTrace();

        }
        return null;
    }

}
