// The MIT License (MIT)
//
// Copyright (c) 2014-2015 PayU India
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package com.payu.custombrowser.analytics;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.AsyncTask;

import com.payu.custombrowser.Bank;
import com.payu.custombrowser.util.CBUtil;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Created by amitkumar on 16/01/15.
 *
 * Helps payu to keep track of user interactions with bank page and more.
 * For later fix: make it as singleton.
 */
public class CBAnalytics {
    private static final String TEST_URL = "https://mobiletest.payu.in/merchant/postservice.php?form=2";
    private static final String PRODUCTION_URL = "https://info.payu.in/merchant/postservice.php?form=2";
    private String fileName;
    private static final long TIMER_DELAY = 5000;
    private static CBAnalytics INSTANCE;
    private final Context mcontext;
    private final String ANALYTICS_URL = Bank.DEBUG ? TEST_URL : PRODUCTION_URL;
    private boolean mIsLocked = false;
    private ArrayList<String> mBuffer;
    private Timer mTimer;

    /**
     * Constructor
     * @param context insatnce of base activity
     */
    private CBAnalytics(final Context context,String filename) {
        mcontext= context;
        fileName=filename;
        mBuffer=new ArrayList<>();

        final Thread.UncaughtExceptionHandler defaultUEH = Thread.getDefaultUncaughtExceptionHandler();
        Thread.UncaughtExceptionHandler _unCaughtExceptionHandler = new Thread.UncaughtExceptionHandler() {
            @Override
            public void uncaughtException(Thread thread, Throwable ex) {
                while (mIsLocked);
                setLock();
                try {
                    FileOutputStream fileOutputStream = mcontext.openFileOutput(fileName, Context.MODE_PRIVATE);
                    int c = mBuffer.size();
                    if(c>0) {
                        JSONArray jsonArray=new JSONArray();
                        for (int i = 0; i < c; i++) {
                            JSONObject jsonObject=new JSONObject(mBuffer.get(i));
                            jsonArray.put(jsonArray.length(),jsonObject);
                        }
                        fileOutputStream.write(jsonArray.toString().getBytes());
                        mBuffer=new ArrayList<>();
                    }
                    fileOutputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }catch(Exception e)
                {
                    e.printStackTrace();
                }
                releaseLock();
                defaultUEH.uncaughtException(thread, ex);
            }
        };
        Thread.setDefaultUncaughtExceptionHandler(_unCaughtExceptionHandler);
    }

    /**
     * keep only one object
     * @param context instance of base activity
     * @return CBAnalytics instance.
     */
    public static CBAnalytics getInstance(Context context,String fileName) {
        if(INSTANCE==null) {
            synchronized (CBAnalytics.class) {
                if (INSTANCE == null) {
                    INSTANCE = new CBAnalytics(context, fileName);

                }
            }
        }
        return INSTANCE;
    }

    /**
     * Adding log information to a local file.
     * @param msg json string
     */

    public void log(final String msg) {
        resetTimer();
        if (mIsLocked) {
            try {
                mBuffer.add(msg);
            }catch(Exception e){
                e.printStackTrace();
            }

        } else {
            new AsyncTask<Void, Void, Void>() {
                @Override
                protected Void doInBackground(Void... voids) {
                    setLock();
                    try {
                        JSONObject newobject = new JSONObject(msg);
                        JSONArray jsonArray;
                        String temp = CBUtil.readFileInputStream(mcontext, fileName, Context.MODE_PRIVATE);
                        if(temp.equalsIgnoreCase(""))
                            jsonArray = new JSONArray();
                        else
                            jsonArray = new JSONArray(temp);

                        FileOutputStream fileOutputStream = mcontext.openFileOutput(fileName, Context.MODE_PRIVATE);
                        jsonArray.put(jsonArray.length(),newobject);
                        fileOutputStream.write((jsonArray.toString()).getBytes());
                        fileOutputStream.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                        mBuffer.add(msg);
                    }catch(JSONException e){
                        e.printStackTrace();

                        mBuffer.add(msg);
                    }catch(Exception e){
                        e.printStackTrace();

                        mBuffer.add(msg);
                    }
                    releaseLock();
                    return null;
                }
            }.execute(null, null, null);
        }
    }

    /**
     * Open file and read the content.
     * Send the contents to payu server
     * command should be sdkWs and var1 should be the actual data.
     */
    private void resetTimer() {
        if (mTimer != null) {
            mTimer.cancel();
        }
        mTimer = new Timer();
        mTimer.schedule(new TimerTask() {
            @Override
            public void run() {
                Thread.currentThread().setName("resetTimer");
                while (mIsLocked) ;
                setLock();
                if (isOnline()) {
                    String temp = "";
                    try {
                        temp = CBUtil.readFileInputStream(mcontext, fileName, Context.MODE_PRIVATE);
                    } finally {

                        // temp is a json array.
                        // mBuffer is a json array again.

                        try {
                            if(!temp.equalsIgnoreCase("")) {
                                JSONArray tempJsonArray = new JSONArray(temp);

                                if (mBuffer.size() > 0) {
                                    for (int i = 0; i < mBuffer.size(); i++) {
                                        JSONObject jsonObject = new JSONObject(mBuffer.get(i));
                                        tempJsonArray.put(jsonObject);
                                    }
                                }
                                if (tempJsonArray.length() > 0) {
                                    String postData = "command=sdkWsNew&var1=" + tempJsonArray.toString();
                                    HttpURLConnection conn= CBUtil.getHttpsConn(ANALYTICS_URL,postData);
                                    if(conn!=null) {
                                        int responseCode = conn.getResponseCode();
                                        // SH: A lot of nested try catch, can we move it all outside?
                                        if (responseCode == HttpURLConnection.HTTP_OK) {
                                                if (conn.getInputStream() != null) {
                                                    StringBuffer responseStringBuffer = CBUtil.getStringBufferFromInputStream(conn.getInputStream());
                                                    if(responseStringBuffer!=null) {
                                                        JSONObject jsonObject = new JSONObject(responseStringBuffer.toString());
                                                        if (jsonObject.has("status")) {
                                                            mcontext.deleteFile(fileName);
                                                            mBuffer = new ArrayList<>();
                                                        }
                                                    }
                                                }

                                        }
                                    }

                                }
                            }

                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
                if (mBuffer.size() > 0) {
                    resetTimer();
                }
                releaseLock();

            }
        }, TIMER_DELAY);
    }

    private synchronized void setLock() {
        mIsLocked = true;
    }

    private synchronized void releaseLock() {
        mIsLocked = false;
    }


    private boolean isOnline() {
        ConnectivityManager cm =
                (ConnectivityManager) mcontext.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo netInfo = cm.getActiveNetworkInfo();
        return netInfo != null && netInfo.isConnectedOrConnecting();
    }
}
