package com.paystack.android.ui

import android.content.Context
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.core.api.models.PaymentChannel
import com.paystack.android.core.api.models.PaystackError
import com.paystack.android.core.logging.Logger
import com.paystack.android.ui.data.dataComponent
import com.paystack.android.ui.data.transaction.TransactionRepository
import com.paystack.android.ui.models.PaymentChannelMenuItem
import com.paystack.android.ui.paymentchannels.PaymentChannelMenuBuilder
import com.paystack.android.ui.utilities.StringProvider
import com.paystack.android.ui.utilities.utilitiesComponent
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update

internal class PaymentProcessViewModel(
    val accessCodeData: AccessCodeData,
    private val transactionRepository: TransactionRepository,
    private val channelMenuBuilder: PaymentChannelMenuBuilder,
) : ViewModel() {
    private val _state =
        MutableStateFlow<PaymentProcessState>(PaymentProcessState.SelectPaymentChannel)
    val state = _state.asStateFlow()

    private val _menuItems = MutableStateFlow<List<PaymentChannelMenuItem>>(emptyList())
    val menuItems = _menuItems.asStateFlow()

    init {
        buildChannelMenu()
    }

    private fun buildChannelMenu() {
        val supportedChannels = accessCodeData.channels
            .filter { transactionRepository.getSupportedPaymentChannels().contains(it) }

        if (supportedChannels.isEmpty()) {
            val error = PaystackError("No supported payment methods on transaction")
            Logger.error(error)
            _state.value = PaymentProcessState.Error(error)
            return
        }

        channelMenuBuilder.build(supportedChannels, accessCodeData.channelOptions).let { items ->
            _menuItems.value = items

            if (items.size == 1) {
                val menuItem = items.first()
                onPaymentChannelSelected(menuItem.paymentChannel, menuItem.providerKey)
                return@let
            }

            _state.update { PaymentProcessState.SelectPaymentChannel }
        }
    }

    fun onPaymentChannelSelected(paymentChannel: PaymentChannel, providerKey: String) {
        _state.update {
            PaymentProcessState.PaymentChannelProcess(paymentChannel, providerKey)
        }
    }

    class Factory(
        private val context: Context,
        private val accessCodeData: AccessCodeData,
        private val transactionRepository: TransactionRepository = dataComponent().transactionRepository,
        private val stringProvider: StringProvider = utilitiesComponent(context).stringProvider,
    ) : ViewModelProvider.Factory {
        @Suppress("UNCHECKED_CAST")
        override fun <T : ViewModel> create(modelClass: Class<T>): T {
            if (modelClass.isAssignableFrom(PaymentProcessViewModel::class.java)) {
                return PaymentProcessViewModel(
                    accessCodeData = accessCodeData,
                    transactionRepository = transactionRepository,
                    channelMenuBuilder = PaymentChannelMenuBuilder(stringProvider),
                ) as T
            }

            throw IllegalArgumentException("Unknown ViewModel class")
        }
    }
}
