package com.paystack.android_sdk.ui.paymentchannels.card.auth.address

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.paystack.android_sdk.core.api.models.AccessCodeData
import com.paystack.android_sdk.ui.data.transaction.TransactionRepository
import com.paystack.android_sdk.ui.models.Charge
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

internal class AddressFormViewModel(
    private val transactionAccessData: AccessCodeData,
    private val transactionRepository: TransactionRepository
) : ViewModel() {

    private val _state: MutableStateFlow<AddressFormState> = MutableStateFlow(AddressFormState())
    val state: StateFlow<AddressFormState> = _state.asStateFlow()

    private val _result = MutableStateFlow<Result<Charge>?>(null)
    val result: StateFlow<Result<Charge>?> = _result.asStateFlow()

    fun setStreetAddress(addressText: String) {
        _state.update { it.copy(streetAddress = addressText) }
    }

    fun setZipCode(zipCode: String) {
        _state.update { it.copy(zipCode = zipCode) }
    }

    fun setSelectedState(index: Int) {
        _state.update { it.copy(selectedStateIndex = index) }
    }

    fun setCity(city: String) {
        _state.update { it.copy(city = city) }
    }

    fun submit() {
        viewModelScope.launch {
            _state.update { it.copy(isSubmitting = true) }
            val formState = state.value
            val selectedStateCode = formState.selectedStateCode
            if (selectedStateCode == null) {
                _state.update { it.copy(isSubmitting = false) }
                val message = "Unable to verify address because state is null"
                _result.value = Result.failure(IllegalStateException(message))
                return@launch
            }
            _result.value = transactionRepository.authenticateWithAddress(
                accessCode = transactionAccessData.accessCode,
                address = formState.streetAddress,
                city = formState.city,
                state = selectedStateCode,
                zipCode = formState.zipCode,
            )
            _state.update { it.copy(isSubmitting = false) }
        }
    }

    fun loadStateList(countryCode: String) {
        viewModelScope.launch {
            _state.update { it.copy(isLoadingStates = true) }
            transactionRepository.getAddressStateList(countryCode)
                .onSuccess { states -> _state.update { it.copy(stateList = states) } }
                .onFailure { e -> _result.value = Result.failure(e) }
            _state.update { it.copy(isLoadingStates = false) }
        }
    }
}
