package com.paystack.android_sdk.ui

import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.ExperimentalAnimationApi
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import com.paystack.android_sdk.ui.components.LoadingScreen
import com.paystack.android_sdk.ui.components.WarningBadge
import com.paystack.android_sdk.ui.components.success.PaymentSuccessful
import com.paystack.android_sdk.ui.components.views.text.SecuredByPaystackText
import com.paystack.android_sdk.ui.models.Charge
import com.paystack.android_sdk.ui.paymentchannels.card.CardPaymentUi
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
@OptIn(ExperimentalAnimationApi::class)
internal fun PaymentUiContent(
    paymentUiState: PaymentUiState,
    modifier: Modifier = Modifier,
    onComplete: (Charge) -> Unit,
    onError: (Throwable) -> Unit
) {
    Column(
        modifier = modifier.padding(bottom = PaystackTheme.dimensions.spacing.double),
        verticalArrangement = Arrangement.SpaceBetween,
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
        AnimatedContent(
            targetState = paymentUiState,
            label = "MainContentCrossfade"
        ) { paymentState ->
            when (paymentState) {
                is PaymentUiState.Initialization -> LoadingScreen()

                is PaymentUiState.PaymentProcess -> PaymentChannelUi(
                    paymentState = paymentState,
                    onComplete = onComplete,
                    onError = onError
                )

                is PaymentUiState.Success -> PaymentSuccessful(
                    amount = paymentState.charge.amount,
                    currencyCode = paymentState.charge.currency,
                    merchantName = paymentState.merchantName,
                )

                is PaymentUiState.Error -> PaymentError(paymentState.displayText)
            }
        }

        SecuredByPaystackText()
    }
}

@Composable
private fun PaymentChannelUi(
    paymentState: PaymentUiState.PaymentProcess,
    onComplete: (Charge) -> Unit,
    onError: (Throwable) -> Unit,
) {
    val isTestTransaction = paymentState.accessCodeData.isTestTransaction
    Column(horizontalAlignment = Alignment.CenterHorizontally) {
        if (isTestTransaction) {
            WarningBadge(text = stringResource(R.string.test))
            Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))
        }
        if (paymentState.selectedPaymentChannel == "card") {
            CardPaymentUi(
                modifier = Modifier.padding(horizontal = PaystackTheme.dimensions.spacing.double),
                onComplete = onComplete,
                onError = onError,
                transactionAccessData = paymentState.accessCodeData
            )
        } else {
            throw IllegalStateException() // TODO: remove when ViewModel logic is done
        }
    }
}

@Composable
private fun PaymentError(displayText: String) {
    Text(text = displayText, style = PaystackTheme.typography.heading.heading3)
}
