package com.paystack.android.core.api.extensions

import com.paystack.android.core.api.extensions.RemoteResponse.Error
import com.paystack.android.core.api.extensions.RemoteResponse.Success
import kotlinx.serialization.Serializable
import retrofit2.Response

/**
 * @param T type of body when the response is [Success].
 * @param E type of error when the response is [Error].
 */
internal sealed class RemoteResponse<T, E> {
    data class Success<T, E>(val body: T) : RemoteResponse<T, E>()
    data class Error<T, E>(val error: E) : RemoteResponse<T, E>()
    companion object {
        fun <T, E> createError(error: E): RemoteResponse<T, E> {
            return Error(error)
        }

        fun <T, E> createSuccess(data: T): RemoteResponse<T, E> {
            return Success(data)
        }
    }
}

/**
 * Custom representation for the returned json from backend.
 *
 * @param message success or error message from the backend
 * @param status this will be a true or false based on Service response.
 */

@Serializable
internal data class ApiResponse<T>(val status: Boolean, val message: String, val data: T?)

/**
 * Converts the retrofit [Response] into [RemoteResponse].
 *
 * @param T retrofit body type.
 * @param R remote response body type.
 * @param E remote response error type.
 *
 * Note that this method will return ERROR remote response if retrofit response body was null.
 */
internal fun <T, R, E> Response<T>.toRemoteResponse(
    bodyToRemote: (T) -> RemoteResponse<R, E>,
    remoteError: (error: RemoteError) -> E
): RemoteResponse<R, E> {
    return if (isSuccessful) {
        val body = body()
        if (body == null) {
            RemoteResponse.createError(remoteError(RemoteError.general(Exception("empty response"))))
        } else {
            bodyToRemote(body)
        }
    } else {
        RemoteResponse.createError(remoteError(RemoteError.Failed(code())))
    }
}

/**
 * Converts the retrofit [response] into [RemoteResponse] by [Response.toRemoteResponse],
 * and wrap it inside a try catch to safely map the exception to [RemoteResponse.Error].
 *
 * @param T retrofit body type.
 * @param R remote response body type.
 * @param E remote response error type.
 *
 * Note that this method will return [RemoteResponse.Error] if retrofit response body was null.
 */
internal fun <T, R, E> safeApiCall(
    response: Response<T>,
    bodyToRemote: (T) -> RemoteResponse<R, E>,
    remoteError: (RemoteError) -> E
): RemoteResponse<R, E> {
    return try {
        response.toRemoteResponse(bodyToRemote, remoteError)
    } catch (e: Exception) {
        RemoteResponse.createError(remoteError(RemoteError.General(e)))
    }
}

/**
 * Every Network request may have one of this two error.
 */
internal sealed class RemoteError {

    /**
     * Indicating that the error is a thrown exception.
     */
    data class General(val throwable: Throwable) : RemoteError()

    /**
     * Indicating that the error is that the response code is not within 200..299
     */
    data class Failed(val responseCode: Int) : RemoteError()

    fun asException(): Exception {
        return when (this) {
            is General -> Exception(this.throwable)
            is Failed -> Exception("Request failed with response code: ${this.responseCode}")
        }
    }

    companion object {
        fun failed(responseCode: Int): RemoteError {
            return Failed(responseCode)
        }

        fun general(throwable: Throwable): RemoteError {
            return General(throwable)
        }
    }
}

internal fun <T, E> ApiResponse<T>.toRemoteResponse(
    remoteError: (RemoteError) -> E
): RemoteResponse<T, E> {
    val body = this.data
    val success = this.status == true

    return if (success) {
        if (body == null) {
            RemoteResponse.createError(remoteError(RemoteError.general(Exception("ApiResponse body is null!"))))
        } else {
            RemoteResponse.createSuccess(body)
        }
    } else {
        RemoteResponse.createError(remoteError(RemoteError.general(Exception("ApiResponse error is null"))))
    }
}

/**
 * Converts the retrofit [response] that holds [ApiResponse] into [RemoteResponse],
 * and safely map the exceptions to [RemoteResponse.Error].
 *
 * This is a custom implementation depends on [ApiResponse] structure.
 *
 * @param T retrofit body type, and remote response body type.
 * @param E remote response error type.
 */
internal fun <T, E> safeApiResponseCall(
    response: Response<ApiResponse<T>>,
    remoteError: (RemoteError) -> E
): RemoteResponse<T, E> {
    return safeApiCall(
        response = response,
        bodyToRemote = {
            it.toRemoteResponse(remoteError)
        },
        remoteError = remoteError
    )
}

/**
 * Converts the retrofit [response] that holds [ApiResponse] into [RemoteResponse],
 * and safely map the exceptions to [RemoteResponse.Error].
 *
 * This is a custom implementation depends on [ApiResponse] structure.
 *
 * Note that the desired error type is defined as [Exception].
 *
 * @param T retrofit body type, and remote response body type.
 *
 * @return [RemoteResponse] with [T] body, and error of type [Exception].
 */
internal fun <T> safeApiResponseCall(
    response: Response<ApiResponse<T>>
): RemoteResponse<T, Exception> {
    return safeApiResponseCall(
        response = response,
        remoteError = {
            it.asException()
        }
    )
}
