package com.paystack.android.core.api.core.encryption

import android.util.Base64
import java.security.GeneralSecurityException
import java.security.KeyFactory
import java.security.NoSuchAlgorithmException
import java.security.PublicKey
import java.security.spec.InvalidKeySpecException
import java.security.spec.X509EncodedKeySpec
import javax.crypto.Cipher

internal class RsaEncryptionManager : EncryptionManager {
    /**
     * Encrypts a string using the public key.
     *
     * @param text The string to encrypt.
     * @param publicKey The public encryption key.
     * @return The encrypted string, encoded as Base64.
     * @throws GeneralSecurityException if there's a problem reading the public key,
     * If the requested algorithm is not supported by the underlying cryptographic library
     * or there's a failure reading the input data
     */
    @Throws(GeneralSecurityException::class)
    override fun encrypt(text: String, publicKey: String): String {
        return Base64.encodeToString(
            encrypt(text, readPublicKey(publicKey)),
            Base64.NO_WRAP
        )
    }

    /**
     * Encrypts a string using the public key.
     *
     * @param text The string to encrypt.
     * @param publicKey The public key used for the encryption.
     * @return The encrypted string, encoded as Base64.
     * @throws GeneralSecurityException If the requested algorithm is not supported by the
     * underlying cryptographic library or there's a failure reading the input data
     */
    @Throws(GeneralSecurityException::class)
    override fun encrypt(text: String, publicKey: PublicKey): ByteArray {
        val cipher = Cipher.getInstance(CIPHER)
        cipher.init(Cipher.ENCRYPT_MODE, publicKey)
        return cipher.doFinal(text.toByteArray())
    }

    /**
     * Reads a public key from a string.
     *
     * @param publicKey The public key string.
     * @return The public key object.
     * @throws NoSuchAlgorithmException If the requested algorithm is not supported by the
     * underlying cryptographic library.
     * @throws InvalidKeySpecException If the key specification is invalid.
     */
    @Throws(NoSuchAlgorithmException::class, InvalidKeySpecException::class)
    private fun readPublicKey(publicKey: String): PublicKey {
        val keyFactory = KeyFactory.getInstance(ALGORITHM)

        // Decode key into a ByteArray
        val keyBytes = Base64.decode(publicKey, Base64.NO_WRAP)

        val keySpec = X509EncodedKeySpec(keyBytes)
        return keyFactory.generatePublic(keySpec)
    }

    companion object {
        private const val ALGORITHM = "RSA"
        private const val CIPHER = "RSA/ECB/OAEPPadding"
    }
}
