package com.paystack.android.core.api.models

import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.descriptors.serialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder

@Serializable(TransactionStatusSerializer::class)
enum class TransactionStatus {
    Success,
    Failed,
    Pending,
    Timeout,
    SentOtp,
    SendBirthday,
    SendPin,
    SendAddress,
    SendPhone,

    /**
     * External transaction authentication is required.
     * The user should be redirected to [ChargeResponse.authUrl] to complete the transaction.
     * This is applicable to 3DS transactions.
     */
    OpenUrl
}

object TransactionStatusSerializer : KSerializer<TransactionStatus> {
    private val mapping = mapOf(
        "success" to TransactionStatus.Success,
        "failed" to TransactionStatus.Failed,
        "0" to TransactionStatus.Failed,
        "pending" to TransactionStatus.Pending,
        "timeout" to TransactionStatus.Timeout,
        "send_otp" to TransactionStatus.SentOtp,
        "send_birthday" to TransactionStatus.SendBirthday,
        "send_pin" to TransactionStatus.SendPin,
        "send_address" to TransactionStatus.SendAddress,
        "send_phone" to TransactionStatus.SendPhone,
        "open_url" to TransactionStatus.OpenUrl,
    )

    override val descriptor: SerialDescriptor = serialDescriptor<String>()

    override fun deserialize(decoder: Decoder): TransactionStatus {
        val value = decoder.decodeString()
        return mapping[value] ?: decoder.decodeSerializableValue(TransactionStatus.serializer())
    }

    override fun serialize(encoder: Encoder, value: TransactionStatus) {
        encoder.encodeString(value.name.lowercase())
    }
}
