package com.paystack.android.core.api.services.cards

import com.paystack.android.core.api.ApiComponent
import com.paystack.android.core.api.ApiRequest
import com.paystack.android.core.api.asApiRequest
import com.paystack.android.core.api.core.encryption.EncryptionManager
import com.paystack.android.core.api.models.AddressState
import com.paystack.android.core.api.models.CardParams
import com.paystack.android.core.api.models.ChargeRequest
import com.paystack.android.core.api.models.ChargeResponse
import com.paystack.android.core.api.models.RedirectAuthResponse
import com.paystack.android.core.api.models.SubmitAddressRequest
import com.paystack.android.core.api.models.SubmitBirthdayRequest
import com.paystack.android.core.api.models.SubmitOtpRequest
import com.paystack.android.core.api.models.SubmitPhoneRequest
import com.paystack.android.core.api.models.SubmitPinRequest
import com.paystack.android.core.api.services.transactions.TransactionApi
import com.paystack.android.core.api.utilities.isDateValid
import com.paystack.android.core.events.EventManager
import com.paystack.android.core.events.EventsComponent
import com.paystack.android.core.events.Subscription
import com.paystack.android.core.events.SubscriptionParams
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json

internal class CardPaymentsServiceImpl(
    private val transactionApi: TransactionApi,
    private val encryptionManager: EncryptionManager,
    private val eventManager: EventManager,
) : CardPaymentsService {
    private val json = Json { ignoreUnknownKeys = true }

    constructor(apiComponent: ApiComponent, eventsComponent: EventsComponent) : this(
        apiComponent.transactionApi,
        apiComponent.encryptionManager,
        eventsComponent.eventManager
    )

    override fun charge(
        accessCode: String,
        cardParams: CardParams,
        encryptionKey: String,
    ): ApiRequest<ChargeResponse> {
        val cardData = encryptionManager.encrypt(Json.encodeToString(cardParams), encryptionKey)
        val chargeRequest = ChargeRequest(
            accessCode = accessCode,
            encryptedCardData = cardData
        )
        return transactionApi.charge(chargeRequest).asApiRequest()
    }

    override fun authenticateWithPin(
        accessCode: String,
        pin: String,
        encryptionKey: String,
    ): ApiRequest<ChargeResponse> {
        val request = SubmitPinRequest(
            accessCode = accessCode,
            pin = encryptionManager.encrypt(pin, encryptionKey)
        )
        return transactionApi.submitPin(request).asApiRequest()
    }

    override fun authenticateWithAddress(
        accessCode: String,
        address: String,
        city: String,
        state: String,
        zipCode: String,
    ): ApiRequest<ChargeResponse> {
        val request = SubmitAddressRequest(
            accessCode = accessCode,
            address = address,
            city = city,
            state = state,
            zipCode = zipCode
        )
        return transactionApi.submitAddress(request).asApiRequest()
    }

    override fun getAddressStateList(countryCode: String): ApiRequest<List<AddressState>> {
        return transactionApi.getAvsStates(countryCode).asApiRequest()
    }

    override fun authenticateWithOtp(accessCode: String, otp: String): ApiRequest<ChargeResponse> {
        val request = SubmitOtpRequest(
            accessCode = accessCode,
            otp = otp
        )
        return transactionApi.submitOtp(request).asApiRequest()
    }

    override fun authenticateWithPhone(
        accessCode: String,
        phoneNumber: String
    ): ApiRequest<ChargeResponse> {
        val request = SubmitPhoneRequest(
            accessCode = accessCode,
            phone = phoneNumber
        )
        return transactionApi.submitPhone(request).asApiRequest()
    }

    override fun authenticateWithBirthday(
        accessCode: String,
        day: String,
        month: String,
        year: String
    ): ApiRequest<ChargeResponse> {
        val isValidDate = isDateValid(year, month, day)
        if (isValidDate) {
            val request = SubmitBirthdayRequest(
                accessCode = accessCode,
                birthday = "$year-$month-$day"
            )
            return transactionApi.submitBirthday(request).asApiRequest()
        } else {
            throw IllegalArgumentException("Invalid date")
        }
    }

    override fun listenFor3dsResult(
        transactionId: String,
        callback: RedirectAuthResultListener
    ) {
        val channelName = "3DS_$transactionId"
        var subscription: Subscription? = null
        subscription = eventManager.subscribe(
            params = SubscriptionParams(
                channelName = channelName,
                eventName = "response",
            ),
            onEvent = { result ->
                runCatching { json.decodeFromString<RedirectAuthResponse>(result) }
                    .onSuccess { transaction ->
                        callback.onResult(transaction)
                        subscription?.cancel()
                    }
                    .onFailure {
                        callback.onError(it)
                        subscription?.cancel()
                    }
            }
        )
    }
}
