package com.paystack.android.core.api.models

import android.os.Parcelable
import kotlinx.parcelize.Parcelize
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

/**
 * A [ChargeResponse] holds details of the payment and tracks its state
 */
@Serializable
@Parcelize
data class ChargeResponse constructor(

    val status: TransactionStatus,

    /**
     * The transaction amount expressed as a positive integer in the smallest currency unit.
     * (e.g. 10000 Kobo to represent ₦100 or 100 for $1)
     */
    val amount: Long,

    /**
     * Three-letter ISO currency code.
     * [Supported Currencies](https://paystack.com/api/#supported-currency)
     */
    val currency: String,

    /**
     * Unique transaction reference. Only -, ., = and alphanumeric characters allowed.
     */
    val reference: String,

    /**
     * The [PaymentChannel] used for this transaction.
     */
    val channel: PaymentChannel,

    /**
     * Indicates if this is a live or test transaction.
     */
    val domain: PaystackDomain,

    /**
     * Date the transaction was created.
     */
    @SerialName("transaction_date")
    val transactionDate: String,

    @SerialName("gateway_response")
    val gatewayResponse: String,

    /**
     * Fees incurred for this transaction
     */
    val fees: Long? = 0,

    /**
     * Custom data added to transaction by merchant
     */
    val metadata: Map<String, String>? = null,

    /**
     * Information about the paying customer
     */
    val customer: Customer? = null,

    /**
     * Transaction Error messages
     */
    @SerialName("message")
    val message: String? = null,

    /**
     * Message to display for certain transaction states.
     * This may be available when [TransactionStatus] requires user action.
     */
    @SerialName("display_text")
    val displayText: String? = null,

    /**
     * A URL to visit to authenticate transactions that require external validation.
     * This is available when [status] is [TransactionStatus.OpenUrl].
     * This is primarily used for 3DS transaction authentication
     */
    @SerialName("url")
    val authUrl: String? = null,

    /**
     * Authorization for the payment method that completed the transaction
     */
    val authorization: ChargeAuthorization? = null
) : Parcelable
