package com.paystack.android_sdk.core.api.transactions

import com.paystack.android_sdk.core.api.ApiRequest
import com.paystack.android_sdk.core.api.models.AccessCodeData
import com.paystack.android_sdk.core.api.models.AddressState
import com.paystack.android_sdk.core.api.models.CardParams
import com.paystack.android_sdk.core.api.models.ChargeResponse

interface PaystackTransactionService {
    fun verifyAccessCode(accessCode: String): ApiRequest<AccessCodeData>

    /**
     * Creates an [ApiRequest] that a card charge for a transaction with [accessCode].
     * The [encryptionKey] is used to encrypt [cardParams] before the charge attempt.
     *
     * @return An [ApiRequest] for [ChargeResponse]
     */
    fun chargeCard(
        accessCode: String,
        cardParams: CardParams,
        encryptionKey: String,
    ): ApiRequest<ChargeResponse>

    /**
     * Creates an [ApiRequest] gets the Charge state for a pending transaction.
     *
     * @return An [ApiRequest] for [ChargeResponse]
     */
    fun checkPendingCharge(accessCode: String): ApiRequest<ChargeResponse>

    /**
     * Creates an [ApiRequest] that attempts to authenticate a Transaction using the card PIN
     *
     * @return An [ApiRequest] for [ChargeResponse]
     */
    fun authenticateWithPin(
        accessCode: String,
        pin: String,
        encryptionKey: String,
    ): ApiRequest<ChargeResponse>

    /**
     * Creates an [ApiRequest] that verifies a card address for transaction authentication.
     *
     * @return An [ApiRequest] for [ChargeResponse]
     */
    fun authenticateWithAddress(
        accessCode: String,
        address: String,
        city: String,
        state: String,
        zipCode: String,
    ): ApiRequest<ChargeResponse>

    fun getAddressStateList(countryCode: String): ApiRequest<List<AddressState>>

    /**
     * Creates an [ApiRequest] that attempts to authenticate a Transaction using an OTP
     *
     * @return An [ApiRequest] for [ChargeResponse]
     */
    fun authenticateWithOtp(accessCode: String, otp: String): ApiRequest<ChargeResponse>

    /**
     * Creates an [ApiRequest] that attempts to authenticate a Transaction using a phone number
     *
     * @return An [ApiRequest] for [ChargeResponse]
     */
    fun authenticateWithPhone(accessCode: String, phoneNumber: String): ApiRequest<ChargeResponse>

    /**
     * Creates an [ApiRequest] that attempts to authenticate a Transaction with a birthday
     * @param accessCode the access code of the transaction
     * @param day the day of the birthday
     * @param month the month of the birthday
     * @param year the year of the birthday
     * @throws IllegalStateException if the date is invalid
     *
     * @return An [ApiRequest] for [ChargeResponse]
     */
    fun authenticateWithBirthday(
        accessCode: String,
        day: String,
        month: String,
        year: String
    ): ApiRequest<ChargeResponse>

    fun listenFor3dsResult(transactionId: String, callback: RedirectAuthResultListener)
}
