package com.paystack.android.core.logging

import android.os.Build
import android.util.Log
import android.util.Log.getStackTraceString
import androidx.annotation.RestrictTo
import com.paystack.android.core.Paystack
import java.util.regex.Pattern
import kotlin.math.min

@RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
object Logger {

    private val isEnabled: Boolean
        get() = if (Paystack.Builder.isInitialized) {
            Paystack.Builder.instance
                .config
                .enableLogging
        } else {
            false
        }

    /**
     * Log a verbose message
     */
    fun verbose(message: String?) {
        prepareLog(Log.VERBOSE, null, message)
    }

    /**
     * Log a verbose message and exception
     */
    fun verbose(t: Throwable?, message: String?) {
        prepareLog(Log.VERBOSE, t, message)
    }

    /**
     * Log a verbose exception
     */
    fun verbose(t: Throwable?) {
        prepareLog(Log.VERBOSE, t, null)
    }

    /**
     * Log a debug message
     */
    fun debug(message: String?) {
        prepareLog(Log.DEBUG, null, message)
    }

    /**
     * Log a debug message and exception
     */
    fun debug(t: Throwable?, message: String?) {
        prepareLog(Log.DEBUG, t, message)
    }

    /**
     * Log a debug exception
     */
    fun debug(t: Throwable?) {
        prepareLog(Log.DEBUG, t, null)
    }

    /**
     * Log a info message
     */
    fun info(message: String?) {
        prepareLog(Log.INFO, null, message)
    }

    /**
     * Log a info message and exception
     */
    fun info(t: Throwable?, message: String?) {
        prepareLog(Log.INFO, t, message)
    }

    /**
     * Log a info exception
     */
    fun info(t: Throwable?) {
        prepareLog(Log.INFO, t, null)
    }

    /**
     * Log a warning message
     */
    fun warn(message: String?) {
        prepareLog(Log.WARN, null, message)
    }

    /**
     * Log a warning message and exception
     */
    fun warn(t: Throwable?, message: String?) {
        prepareLog(Log.WARN, t, message)
    }

    /**
     * Log a warning exception
     */
    fun warn(t: Throwable?) {
        prepareLog(Log.WARN, t, null)
    }

    /**
     * Log a error message
     */
    fun error(message: String?) {
        prepareLog(Log.ERROR, null, message)
    }

    /**
     * This method logs all error events to the console
     *
     * @param errorMessage a message in the form of a String that needs to be emitted to the log
     * when this method is called.
     * @param throwable the cause of the error thrown.
     */
    @Deprecated(
        message = "Use error(t: Throwable?, message: String?) instead",
        replaceWith = ReplaceWith("Logger.error(throwable, errorMessage)")
    )
    fun error(errorMessage: String, throwable: Throwable?) {
        error(throwable, errorMessage)
    }

    /**
     * Log a error message and exception
     */
    fun error(t: Throwable?, message: String?) {
        prepareLog(Log.ERROR, t, message)
    }

    /**
     * Log a error exception
     */
    fun error(t: Throwable?) {
        prepareLog(Log.ERROR, t, null)
    }

    private fun prepareLog(priority: Int, t: Throwable?, message: String?) {
        if (isEnabled.not()) return

        val tag = getTag()
        var msg = message
        if (msg.isNullOrEmpty()) {
            if (t == null) {
                return
            }
            msg = getStackTraceString(t)
        } else {
            if (t != null) {
                msg += "\n" + getStackTraceString(t)
            }
        }

        log(priority, tag, msg)
    }

    private fun log(priority: Int, tag: String?, msg: String) {
        if (msg.length < MAX_LOG_LENGTH) {
            Log.println(priority, tag, msg)
            return
        }

        // Split by line, then ensure each line can fit into Log's maximum length.
        var i = 0
        val length = msg.length
        while (i < length) {
            var newline = msg.indexOf('\n', i)
            newline = if (newline != -1) newline else length
            do {
                val end = min(newline, i + MAX_LOG_LENGTH)
                val part = msg.substring(i, end)
                Log.println(priority, tag, part)
                i = end
            } while (i < newline)
            i++
        }
    }

    private fun getTag(): String {
        return Throwable().stackTrace
            .first { it.className != Logger::class.java.name }
            .let(::createStackElementTag)
    }

    /**
     * Extract the tag which should be used for the message from the `element`.
     */
    private fun createStackElementTag(element: StackTraceElement): String {
        var tag = element.className.substringAfterLast('.')
        val m = ANONYMOUS_CLASS.matcher(tag)
        if (m.find()) {
            tag = m.replaceAll("")
        }
        // Tag length limit was removed in API 26.
        return if (tag.length <= MAX_TAG_LENGTH || Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            tag
        } else {
            tag.substring(0, MAX_TAG_LENGTH)
        }
    }

    private const val MAX_LOG_LENGTH = 4000
    private const val MAX_TAG_LENGTH = 23
    private val ANONYMOUS_CLASS = Pattern.compile("(\\$\\d+)+$")
}
