/*
 * Copyright (c) 2014, 2017, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.svm.core.genscavenge;

import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.Platforms;
import org.graalvm.word.Pointer;

import com.oracle.svm.core.heap.ObjectVisitor;
import com.oracle.svm.core.log.Log;

/**
 * A Generation is a collection of one or more Spaces.
 */
public abstract class Generation {

    /** Final state. */
    protected final String name;

    /** Constructor for subclasses. */
    @Platforms(Platform.HOSTED_ONLY.class)
    Generation(String name) {
        this.name = name;
    }

    /**
     * Walk the Objects in this Space, passing each to a Visitor.
     *
     * @param visitor An ObjectVisitor.
     * @return True if all visits returned true, false otherwise.
     */
    public abstract boolean walkObjects(ObjectVisitor visitor);

    /** The name of this generation. */
    public String getName() {
        return name;
    }

    /** Report some statistics about the Generation to a Log. */
    public abstract Log report(Log log, boolean traceHeapChunks);

    /**
     * Promote an Object to this Generation, either by HeapChunk motion or copying. If the original
     * is copied, a forwarding pointer to the new Object is left in place of the original Object.
     *
     * This turns an Object from white to grey: the object is in this Generation, but has not yet
     * had its interior pointers visited.
     *
     * @param original The original Object to be promoted.
     * @return The promoted Object, either the original if promotion was done by HeapChunk motion,
     *         or a new Object if promotion was done by copying.
     *
     *         For debugging GR-9912:
     * @param objRef The address of the reference to the original object.
     * @param innerOffset The byte offset of a derived pointer from head of the original object.
     *            Subtract to get to the head of the original object, add to get back to a derived
     *            pointer inside the original object.
     * @param compressed Whether object references are compressed.
     */
    protected abstract Object promoteObject(Object original, Pointer objRef, int innerOffset, boolean compressed);

    /** Returns true if the space is a valid Space in this generation, else false. */
    protected abstract boolean isValidSpace(Space space);

    protected abstract boolean verify(HeapVerifier.Occasion occasion);
}
