/*-
 * Copyright (c) 2020, 2021 Oracle and/or its affiliates.  All rights reserved.
 *
 * Licensed under the Universal Permissive License v 1.0 as shown at
 *  https://oss.oracle.com/licenses/upl/
 */
package com.oracle.nosql.spring.data.core;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import oracle.nosql.driver.NoSQLException;
import oracle.nosql.driver.NoSQLHandle;
import oracle.nosql.driver.TableNotFoundException;
import oracle.nosql.driver.ops.DeleteRequest;
import oracle.nosql.driver.ops.DeleteResult;
import oracle.nosql.driver.ops.GetRequest;
import oracle.nosql.driver.ops.GetResult;
import oracle.nosql.driver.ops.PrepareRequest;
import oracle.nosql.driver.ops.PrepareResult;
import oracle.nosql.driver.ops.PreparedStatement;
import oracle.nosql.driver.ops.PutRequest;
import oracle.nosql.driver.ops.PutResult;
import oracle.nosql.driver.ops.QueryRequest;
import oracle.nosql.driver.ops.TableRequest;
import oracle.nosql.driver.ops.TableResult;
import oracle.nosql.driver.ops.WriteMultipleRequest;
import oracle.nosql.driver.util.LruCache;
import oracle.nosql.driver.values.FieldValue;
import oracle.nosql.driver.values.LongValue;
import oracle.nosql.driver.values.MapValue;

import com.oracle.nosql.spring.data.NosqlDbFactory;
import com.oracle.nosql.spring.data.config.AbstractNosqlConfiguration;
import com.oracle.nosql.spring.data.config.NosqlDbConfig;
import com.oracle.nosql.spring.data.core.convert.MappingNosqlConverter;
import com.oracle.nosql.spring.data.core.mapping.NosqlPersistentEntity;
import com.oracle.nosql.spring.data.core.query.NosqlQuery;
import com.oracle.nosql.spring.data.repository.support.NosqlEntityInformation;

import org.apache.commons.lang3.reflect.FieldUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.projection.SpelAwareProxyProjectionFactory;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;

public class NosqlTemplate implements NosqlOperations, ApplicationContextAware {
    public static final String JSON_COLUMN = "kv_json_";

    private static final Logger log =
        LoggerFactory.getLogger(NosqlTemplate.class);

    static final String TEMPLATE_CREATE_TABLE =
        "CREATE TABLE IF NOT EXISTS %s (%s %s %s, " +
            JSON_COLUMN + " JSON, PRIMARY KEY( %s ))";

    static final String TEMPLATE_GENERATED_ALWAYS =
        "GENERATED ALWAYS as IDENTITY (NO CYCLE)";

    static final String TEMPLATE_GENERATED_UUID =
        " AS UUID GENERATED BY DEFAULT";

    static final String TEMPLATE_DROP_TABLE =
        "DROP TABLE IF EXISTS %s ";

    static final String TEMPLATE_DELETE_ALL =
        "DELETE FROM %s ";

    static final String TEMPLATE_SELECT_ALL =
        "SELECT * FROM %s t";

    static final String TEMPLATE_COUNT =
        "SELECT count(*) FROM %s ";

    static final String TEMPLATE_UPDATE =
        "DECLARE $id %s; $json JSON; " +
        "UPDATE %s t SET t." + JSON_COLUMN + " = $json WHERE t.%s = $id";

    private final NosqlDbFactory nosqlDbFactory;
    private final NoSQLHandle nosqlClient;
    private final MappingNosqlConverter mappingNosqlConverter;
    private final SpelAwareProxyProjectionFactory projectionFactory;

    private LruCache<String, PreparedStatement> psCache;


    public static NosqlTemplate create(NosqlDbConfig nosqlDBConfig)
        throws ClassNotFoundException {
        Assert.notNull(nosqlDBConfig, "NosqlDbConfig should not be null.");
        return create(new NosqlDbFactory(nosqlDBConfig));
    }

    public static NosqlTemplate create(NosqlDbFactory nosqlDbFactory)
        throws ClassNotFoundException {
        Assert.notNull(nosqlDbFactory, "NosqlDbFactory should not be null.");
        AbstractNosqlConfiguration configuration =
            new AbstractNosqlConfiguration();
        return new NosqlTemplate(nosqlDbFactory,
            configuration.mappingNosqlConverter());
    }

    public NosqlTemplate(NosqlDbFactory nosqlDbFactory,
        MappingNosqlConverter mappingNosqlConverter) {
        Assert.notNull(nosqlDbFactory, "NosqlDbFactory should not be null.");
        this.nosqlDbFactory = nosqlDbFactory;
        nosqlClient = nosqlDbFactory.getNosqlClient();
        this.mappingNosqlConverter = mappingNosqlConverter;
        this.projectionFactory = new SpelAwareProxyProjectionFactory();
        psCache = new LruCache<>(nosqlDbFactory.getQueryCacheCapacity(),
            nosqlDbFactory.getQueryCacheLifetime());
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext)
        throws BeansException {
    }

    public String getTableName(Class<?> domainClass) {
        Assert.notNull(domainClass, "domainClass should not be null");

        return getNosqlEntityInformation(domainClass).getTableName();
    }

    @Override
    public boolean createTableIfNotExists(
        NosqlEntityInformation<?, ?> entityInformation) {
        String idColName = entityInformation.getIdField().getName();

        String idColType = entityInformation.getIdNosqlType().toString();
        if (entityInformation.getIdNosqlType() == FieldValue.Type.TIMESTAMP) {
            // For example: CREATE TABLE IF NOT EXISTS SensorIdTimestamp
            //   (time TIMESTAMP(3) , kv_json_ JSON, PRIMARY KEY( time ))
            idColType += "(" + nosqlDbFactory.getTimestampPrecision() + ")";
        }

        String autogen = "";
        if (entityInformation.isAutoGeneratedId() ) {
            if (entityInformation.getIdNosqlType() == FieldValue.Type.STRING) {
                autogen = TEMPLATE_GENERATED_UUID;
            } else {
                autogen = TEMPLATE_GENERATED_ALWAYS;
            }
        }

        String sql = String.format(TEMPLATE_CREATE_TABLE,
            entityInformation.getTableName(),
            idColName, idColType, autogen, idColName);

        TableRequest tableReq = new TableRequest().setStatement(sql)
            .setTableLimits(entityInformation.getTableLimits());

        TableResult tableRes = doTableRequest(entityInformation, tableReq);

        TableResult.State tableState = tableRes.getTableState();
        return tableState == TableResult.State.ACTIVE;
    }

    private TableResult doTableRequest(
        NosqlEntityInformation<?, ?> entityInformation,
        TableRequest tableReq) {

        if (entityInformation != null &&
            entityInformation.getTimeout() > 0) {
            tableReq.setTimeout(entityInformation.getTimeout());
        }

        TableResult tableRes;
        try {
            log.debug("DDL: {}", tableReq.getStatement());
            tableRes = nosqlClient.doTableRequest(tableReq,
                nosqlDbFactory.getTableReqTimeout(),
                nosqlDbFactory.getTableReqPollInterval());
        } catch (NoSQLException nse) {
            log.error("DDL: {}", tableReq.getStatement());
            log.error(nse.getMessage());
            throw MappingNosqlConverter.convert(nse);
        }
        return tableRes;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T insert(@NonNull T objectToSave) {
        Assert.notNull(objectToSave, "entityClass should not be null");

        return insert(getNosqlEntityInformation(
            (Class<T>) objectToSave.getClass()),
            objectToSave);
    }

    /**
     * If entity doesn't have autogen field objectToSave is wrote using put.
     * If entity has autogen field objectToSave should not have field set.
     */
    @Override
    public <T, ID> T insert(NosqlEntityInformation<T, ID> entityInformation,
        @NonNull T objectToSave) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");

        Assert.notNull(objectToSave, "objectToSave should not be null");

        final MapValue row = mappingNosqlConverter.convertObjToRow(
            objectToSave, entityInformation.isAutoGeneratedId());

        PutResult putRes = doPut(entityInformation, row, false);

        FieldValue id;
        if (entityInformation.isAutoGeneratedId()) {
            id = putRes.getGeneratedValue();
            if (id == null) {
                throw new IllegalStateException("Expected generated value is " +
                    "null.");
            }
            // for the case when id is autogenerated, the generated value is in
            // the result
            // id is set to the same object and returned
            objectToSave = populateIdIfNecesary(objectToSave, id);
        }

        return objectToSave;
    }

    private PutResult doPut(NosqlEntityInformation<?, ?> entityInformation,
        MapValue row, boolean ifPresent) {
        PutRequest putReq = new PutRequest()
            .setTableName(entityInformation.getTableName())
            .setValue(row);

        if (ifPresent) {
            putReq.setOption(PutRequest.Option.IfPresent);
        }

        if (entityInformation.isAutoGeneratedId()) {
            putReq.setReturnRow(true);
        }

        if (entityInformation.getTimeout() > 0) {
            putReq.setTimeout(entityInformation.getTimeout());
        }

        //todo set durability when supported by API

        PutResult putRes;
        try {
            try {
                putRes = nosqlClient.put(putReq);
            } catch (TableNotFoundException tnfe) {
                if (entityInformation.isAutoCreateTable()) {
                    createTableIfNotExists(entityInformation);
                    putRes = nosqlClient.put(putReq);
                } else {
                    throw tnfe;
                }
            }
        } catch (NoSQLException nse) {
            log.error("Put: table: {} key: {}", putReq.getTableName(),
                row.get(entityInformation.getIdColumnName()));
            log.error(nse.getMessage());
            throw MappingNosqlConverter.convert(nse);
        }

        assert putRes != null;
        return putRes;
    }

    private <T> T populateIdIfNecesary(T objectToSave, FieldValue id) {
        return mappingNosqlConverter.setId(objectToSave, id);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> void update(T objectToSave) {
        Assert.notNull(objectToSave, "entity should not be null");

        update(getNosqlEntityInformation((Class<T>) objectToSave.getClass()),
            objectToSave);
    }

    public <T, ID> void update(NosqlEntityInformation<T, ID> entityInformation,
        T objectToSave) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");
        Assert.notNull(objectToSave, "objectToSave should not be null");

        log.debug("execute update in table {}", entityInformation.getTableName());
        final MapValue row = mappingNosqlConverter
            .convertObjToRow(objectToSave, false);

        doUpdate(entityInformation, row);
    }

    private void doUpdate(NosqlEntityInformation<?, ?> entityInformation,
        MapValue row) {
        // When id is autogenerated, it's required to do a SQL update query
        if (entityInformation.isAutoGeneratedId()) {
            final String idColumnName = entityInformation.getIdColumnName();
            String sql = String.format(TEMPLATE_UPDATE,
                entityInformation.getIdNosqlType().name(),
                entityInformation.getTableName(),
                idColumnName);

            Map<String, FieldValue> params = new HashMap<>();
            //todo implement composite keys
            params.put("$id", row.get(idColumnName));
            params.put("$json", row.get(JSON_COLUMN));

            // Must read at least one result to execute query!!!
            runQueryNosqlParams(entityInformation, sql, params)
                .iterator()
                .next();
        } else {
            // otherwise do a regular put, which is faster, use less resources
            doPut(entityInformation, row, true);
        }
    }

    @Override
    public void deleteAll(NosqlEntityInformation<?, ?> entityInformation) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");

        String sql = String.format(TEMPLATE_DELETE_ALL,
            entityInformation.getTableName() );

        // Since this returns an Iterable the query isn't run until first
        // result is read. Must read at least one result.
        runQuery(entityInformation, sql).iterator().next();
//        log.debug("deleteAll(" + tableName + "): " + res);
    }

    @Override
    public <T, ID> void deleteAll(
        NosqlEntityInformation<T, ID> entityInformation,
        Iterable<? extends ID> ids) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");
        Assert.notNull(ids, "ids should not be null");

        // Cannot use nosqlClient.multiDelete or writeMultiple because this
        // can't guarantee all ids are in the same shard. For this see
        // deleteInShard() method.
        //todo supporting limiting parallelism (1000+ fails) requires
        // external libraries
        StreamSupport.stream(ids.spliterator(), false)
            .forEach(id -> deleteById(entityInformation, id));
    }

    /**
     * Deletes ids from one shard. Note: All ids must be in the same shard
     * otherwise it's an error. It uses
     * {@link NoSQLHandle#writeMultiple(WriteMultipleRequest)}.
     */
    public <T, ID> void deleteInShard(String tableName, Class<T> entityClass,
        Iterable<? extends ID> ids)
    {
        Assert.hasText(tableName, "Table name should not be null, " +
            "empty or only whitespaces");
        Assert.notNull(ids, "ids should not be null");
        Assert.notNull(entityClass, "entityClass should not be null");

        NosqlEntityInformation<?, ?> entityInformation =
            getNosqlEntityInformation(entityClass);

        WriteMultipleRequest wmReq = new WriteMultipleRequest();
        if (entityInformation.getTimeout() > 0) {
            wmReq.setTimeout(entityInformation.getTimeout());
        }

        //todo set durability when supported by API

        String idColumnName = getIdColumnName(entityClass);

        StreamSupport.stream(ids.spliterator(), true)
            .map(id -> mappingNosqlConverter.convertIdToPrimaryKey(idColumnName, id))
            .map(pk -> new DeleteRequest().setKey(pk).setTableName(tableName))
            .forEach(dr -> wmReq.add(dr, false));

        try {
            nosqlClient.writeMultiple(wmReq);
        } catch (NoSQLException nse) {
            log.error("WriteMultiple: table: {}", wmReq.getTableName());
            log.error(nse.getMessage());
            throw MappingNosqlConverter.convert(nse);
        }
    }

    /**
     * Drops table and returns true if result indicates table state changed to
     * DROPPED or DROPPING.
     * Uses {@link NosqlDbFactory#getTableReqTimeout()} and
     * {@link NosqlDbFactory#getTableReqPollInterval()} to check the result.
     */
    @Override
    public boolean dropTableIfExists(String tableName) {
        Assert.hasText(tableName, "tableName should not be null, " +
            "empty or only whitespaces");

        String sql = String.format(TEMPLATE_DROP_TABLE, tableName );

        TableRequest tableReq = new TableRequest().setStatement(sql);

        TableResult tableRes = doTableRequest(null, tableReq);

        return tableRes.getTableState() == TableResult.State.DROPPED ||
            tableRes.getTableState() == TableResult.State.DROPPING;
    }

    @Override
    public MappingNosqlConverter getConverter() {
        return mappingNosqlConverter;
    }

    public <T> NosqlEntityInformation<T, ?> getNosqlEntityInformation(
        Class<T> domainClass) {
        return new NosqlEntityInformation<>(domainClass);
    }

    @Override
    public <T, ID> T findById(ID id, Class<T> javaType) {
        return findById(getNosqlEntityInformation(javaType), id);
    }

    @Override
    public <T, ID> T findById(NosqlEntityInformation<T, ID> entityInformation,
        ID id) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");
        Assert.notNull(id, "id should not be null");

        log.debug("execute findById in table {}",
            entityInformation.getTableName());

        final String idColumName = mappingNosqlConverter
            .getIdProperty(entityInformation.getJavaType()).getName();

        final MapValue row = mappingNosqlConverter
            .convertIdToPrimaryKey(idColumName, id);

        GetResult getRes = doGet(entityInformation, row);

        return mappingNosqlConverter.read(entityInformation.getJavaType(),
            getRes.getValue());
    }

    private GetResult doGet(NosqlEntityInformation<?, ?> entityInformation,
        MapValue primaryKey) {

        GetRequest getReq = new GetRequest()
            .setTableName(entityInformation.getTableName())
            .setKey(primaryKey);

        if (entityInformation.getTimeout() > 0) {
            getReq.setTimeout(entityInformation.getTimeout());
        }

        getReq.setConsistency(entityInformation.getConsistency());

        GetResult getRes;

        try {
            getRes = nosqlClient.get(getReq);
        } catch (NoSQLException nse) {
            log.error("Get: table: {} key: {}", getReq.getTableName(),
                primaryKey);
            log.error(nse.getMessage());
            throw MappingNosqlConverter.convert(nse);
        }

        assert getRes != null;
        return getRes;
    }

    @Override
    public <T, ID> Iterable<T> findAllById(
        NosqlEntityInformation<T, ID> entityInformation, Iterable<ID> ids) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");
        Assert.notNull(ids, "Id list should not be null");

        //todo usage of limited parallel streams (10000+ fails) requires
        // external libs
        return IterableUtil.getIterableFromStream(
            StreamSupport.stream(ids.spliterator(), false)
                .map(x -> findById(entityInformation, x)));
    }

    @Override
    public <T, ID> void deleteById(
        NosqlEntityInformation<T, ID> entityInformation,
        ID id) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");        Assert.notNull(id, "id should not be null");

        log.debug("execute deleteById in table {}",
            entityInformation.getTableName());

        final String idColumnName = getIdColumnName(
            entityInformation.getJavaType());

        final MapValue row = mappingNosqlConverter.convertIdToPrimaryKey(
            idColumnName, id);

        doDelete(entityInformation, row);
    }

    private <T> String getIdColumnName(@NonNull Class<T> entityClass) {
        final NosqlPersistentEntity<?> entity =
            mappingNosqlConverter.getMappingContext().getPersistentEntity(entityClass);
        Assert.notNull(entity, "entity should not be null");
        Assert.notNull(entity.getIdProperty(), "entity.getIdProperty() should" +
            " not be null");
        return entity.getIdProperty().getName();
    }

    private DeleteResult doDelete(NosqlEntityInformation<?, ?> entityInformation,
        MapValue primaryKey) {

        DeleteRequest delReq = new DeleteRequest()
            .setTableName(entityInformation.getTableName())
            .setKey(primaryKey);

        if (entityInformation.getTimeout() > 0) {
            delReq.setTimeout(entityInformation.getTimeout());
        }

        //todo add durability setting when supported by API

        DeleteResult delRes;

        try {
            delRes = nosqlClient.delete(delReq);
        } catch (NoSQLException nse) {
            log.error("Delete: table: {} key: {}", delReq.getTableName(),
                primaryKey);
            log.error(nse.getMessage());
            throw MappingNosqlConverter.convert(nse);
        }

        assert delRes != null;
        return delRes;
    }

    @Override
    public <T> Iterable<T> findAll(Class<T> entityClass) {
        Assert.notNull(entityClass, "entityClass should not be null");

        return findAll(getNosqlEntityInformation(entityClass));
    }

    public <T> Iterable<T> findAll(
        NosqlEntityInformation<T, ?> entityInformation) {
        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");

        String sql = String.format(TEMPLATE_SELECT_ALL,
            entityInformation.getTableName());

        Iterable<MapValue> items = runQuery(entityInformation, sql);
        Stream<T> result = IterableUtil.getStreamFromIterable(items)
                .map(d -> getConverter()
                    .read(entityInformation.getJavaType(), d));
        return IterableUtil.getIterableFromStream(result);
    }

    @Override
    public long count(NosqlEntityInformation<?, ?> entityInformation) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");

        String sql = String.format(TEMPLATE_COUNT,
            entityInformation.getTableName());

        Iterable<MapValue> res = runQuery(entityInformation, sql);

        Assert.isTrue(res != null,
            "Result of a count query should not be null and should have a non" +
                " null iterator.");
        Iterator<MapValue> iterator = res.iterator();
        Assert.isTrue(iterator.hasNext(),
            "Result of count query iterator should have 1 result.");
        Collection<FieldValue> values = iterator.next().values();
//        Assert.isTrue(values.size() == 1, "Results of a count query " +
//            "collection should have 1 result.");
        FieldValue countField = values.iterator().next();
        Assert.isTrue(countField != null && countField.getType() ==
            FieldValue.Type.LONG,
            "Result of a count query should be of type LONG.");
        return countField.asLong().getValue();
    }

    @Override
    public <T> Iterable<T> findAll(
        NosqlEntityInformation<T, ?> entityInformation,
        Sort sort) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");
        String sql = String.format(TEMPLATE_SELECT_ALL,
            entityInformation.getTableName());

        sql += " " + orderBySql(entityInformation, sort);
//        log.debug("findAll(" + tableName + ", " + sort + "): SQL: " + sql);

        Iterable<MapValue> items = runQuery(entityInformation, sql);

        return IterableUtil.getIterableFromStream(
            IterableUtil.getStreamFromIterable(items)
            .map(d -> getConverter().read(entityInformation.getJavaType(), d)));
    }

    @Override
    public <T> Page<T> findAll(NosqlEntityInformation<T, ?> entityInformation,
        Pageable pageable) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");

        Map<String, FieldValue> params = new HashMap<>();

        String sql = limitOffsetSql(entityInformation, pageable, params);
//        log.debug("findAll(" + tableName + ", " + pageable + "): SQL: " + sql);

        Iterable<MapValue> items = runQueryNosqlParams(entityInformation, sql,
            params);

        List<T> result = IterableUtil.getStreamFromIterable(items)
            .map(d -> getConverter().read(entityInformation.getJavaType(), d))
            .collect(Collectors.toList());

        return new PageImpl<>(result, pageable, count(entityInformation));
    }

    private <T> String limitOffsetSql(
        NosqlEntityInformation<T, ?> entityInformation, Pageable pageable,
        @NonNull Map<String, FieldValue> params) {

        String sql = String.format(TEMPLATE_SELECT_ALL,
            entityInformation.getTableName());

        if (pageable == null || pageable.isUnpaged()) {
            return sql;
        }

        sql = sql + orderBySql(entityInformation, pageable.getSort());

        sql += " LIMIT $kv_limit_ OFFSET $kv_offset_";

        params.put("$kv_limit_", new LongValue(pageable.getPageSize()));
        params.put("$kv_offset_", new LongValue(pageable.getOffset()));

        sql = "DECLARE $kv_limit_ LONG; $kv_offset_ LONG; " + sql;

        return sql;
    }

    private <T> String orderBySql(
        NosqlEntityInformation<T, ?> entityInformation, Sort sort) {

        if (sort == null || sort.isUnsorted()) {
            return "";
        }

        String sql = sort.stream()
            .map(f -> ( "t." +
                convertProperty(entityInformation, f.getProperty()) + " " +
                (f.isAscending() ? "ASC" : "DESC")))
            .collect(Collectors.joining(",", "ORDER BY ", ""));
        return " " + sql;
    }

    private <T> String convertProperty(
        NosqlEntityInformation<T, ?> entityInformation,
        @NonNull String property) {

        Field field = FieldUtils.getField(entityInformation.getJavaType(),
            property);

        // if field == null can mean it's not accessible so it can still be a
        // valid non-id field
        if (field != null && field.equals(entityInformation.getIdField())) {
            return property;
        }

        return JSON_COLUMN + "." + property;
    }

    public void runTableRequest(String statement) {
        TableRequest tableRequest = new TableRequest();
        tableRequest.setStatement(statement);

        doTableRequest(null, tableRequest);
    }

    public Iterable<MapValue> runQuery(
        NosqlEntityInformation<?, ?> entityInformation,
        String query) {
        return runQueryNosqlParams(entityInformation, query, null);
    }

    /**
     *  javaParams is a Map of param_name to Java objects
     */
    public Iterable<MapValue> runQueryJavaParams(
        NosqlEntityInformation<?, ?> entityInformation,
        String query,
        Map<String, Object> javaParams) {
        Map<String, FieldValue> nosqlParams = null;

        if (javaParams != null) {
            nosqlParams = new HashMap<>(javaParams.size());

            for (Map.Entry<String, Object> e : javaParams.entrySet()) {
                FieldValue fieldValue =
                    mappingNosqlConverter.convertObjToFieldValue(e.getValue(),
                        null, false);
                nosqlParams.put(e.getKey(), fieldValue);
            }
        }

        return runQueryNosqlParams(entityInformation, query, nosqlParams);
    }

    /**
     * nosqlParams is a Map of param_name to FieldValue
     */
    public Iterable<MapValue> runQueryNosqlParams(
        NosqlEntityInformation<?, ?> entityInformation,
        String query,
        Map<String, FieldValue> nosqlParams) {

        PreparedStatement preparedStatement =
            getPreparedStatement(entityInformation, query);

        if (nosqlParams != null) {
            for (Map.Entry<String, FieldValue> e : nosqlParams.entrySet()) {
                preparedStatement.setVariable(e.getKey(), e.getValue());
            }
        }

        QueryRequest qReq = new QueryRequest()
            .setPreparedStatement(preparedStatement);

        if (entityInformation != null) {
            if (entityInformation.getTimeout() > 0) {
                qReq.setTimeout(entityInformation.getTimeout());
            }

            qReq.setConsistency(entityInformation.getConsistency());
        }

        log.debug("Q: {}", query);
        Iterable<MapValue> results = doQuery(qReq);

        return results;
    }

    /* Query execution for dynamic queries */
    @Override
    public <T> Iterable<MapValue> count(
        NosqlEntityInformation<T, ?> entityInformation, NosqlQuery query) {

        return executeMapValueQuery(entityInformation, query);
    }

    @Override
    public <T, ID> Iterable<T> delete(
        NosqlEntityInformation<T, ID> entityInformation, NosqlQuery query) {

        return IterableUtil.getIterableFromStream(
            IterableUtil.getStreamFromIterable(
                find(entityInformation, entityInformation.getJavaType(), query))
            .map(e -> {
                deleteById(entityInformation, entityInformation.getId(e));
                return e;
            }));
    }

    @SuppressWarnings("unchecked")
    public <S, T> Iterable<T> find(
        NosqlEntityInformation<S, ?> entityInformation,
        Class<T> targetType,
        NosqlQuery query) {

        Class entityType = entityInformation.getJavaType();
        Class<?> typeToRead = targetType.isInterface() ||
            targetType.isAssignableFrom(entityType)
            ? entityType
            : targetType;

        Iterable<MapValue> results = executeMapValueQuery(entityInformation,
            query);

        Stream<T> resStream = IterableUtil.getStreamFromIterable(results)
            .map(d -> {
                Object source = getConverter().read(typeToRead, d);
                T result = targetType.isInterface()
                    ? projectionFactory.createProjection(targetType, source)
                    : (T) source;
                return result;
            });

        return IterableUtil.getIterableFromStream(resStream);
    }

    private  <T> Iterable<MapValue> executeMapValueQuery(
        NosqlEntityInformation<T, ?> entityInformation, NosqlQuery query) {

        Assert.notNull(entityInformation, "Entity information " +
            "should not be null.");
        Assert.notNull(query, "Query should not be null.");

        Class<T> entityClass = entityInformation.getJavaType();

        String idPropertyName = ( entityClass == null ||
            mappingNosqlConverter.getIdProperty(entityClass) == null ? null :
            mappingNosqlConverter.getIdProperty(entityClass).getName());

        final Map<String, Object> params = new LinkedHashMap<>();
        String sql = query.generateSql(entityInformation.getTableName(), params,
            idPropertyName);

        PreparedStatement pStmt = getPreparedStatement(entityInformation, sql);

        for (Map.Entry<String, Object> param : params.entrySet()) {
            pStmt.setVariable(param.getKey(),
                mappingNosqlConverter.convertObjToFieldValue(param.getValue(),
                    null, false));
        }

        QueryRequest qReq = new QueryRequest().setPreparedStatement(pStmt);

        if (entityInformation.getTimeout() > 0) {
            qReq.setTimeout(entityInformation.getTimeout());
        }

        qReq.setConsistency(entityInformation.getConsistency());

        if (query.isCount()) {
            qReq.setLimit(1);
        }

        log.debug("Q: {}", sql);
//        System.out.println("Q: " + sql);
        return doQuery(qReq);
    }

    private PreparedStatement getPreparedStatement(
        NosqlEntityInformation<?, ?> entityInformation, String query) {
        PreparedStatement preparedStatement;

        preparedStatement = psCache.get(query);
        if (preparedStatement == null) {
            PrepareRequest pReq = new PrepareRequest()
                .setStatement(query);

            if (entityInformation != null) {
                if (entityInformation.getTimeout() > 0) {
                    pReq.setTimeout(entityInformation.getTimeout());
                }
            }

            try {
                log.debug("Prepare: {}", pReq.getStatement());
                PrepareResult pRes = nosqlClient.prepare(pReq);
                preparedStatement = pRes.getPreparedStatement();
                psCache.put(query, preparedStatement);
            } catch (NoSQLException nse) {
                log.error("Prepare: {}", pReq.getStatement());
                log.error(nse.getMessage());
                throw MappingNosqlConverter.convert(nse);
            }
        }
        return preparedStatement.copyStatement();
    }

    private Iterable<MapValue> doQuery(QueryRequest qReq) {
        return new IterableUtil.IterableImpl(nosqlClient, qReq);
    }
}
