package com.guagua.guagua.image.adapter

import android.content.Context
import android.graphics.Color
import android.graphics.ColorFilter
import android.text.TextUtils
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.annotation.NonNull
import androidx.core.view.isVisible
import androidx.recyclerview.widget.RecyclerView
import com.guagua.guagua.image.R
import com.guagua.guagua.image.data.ItemType
import com.guagua.guagua.image.data.MediaFile
import com.guagua.guagua.image.manager.ConfigManager
import com.guagua.guagua.image.manager.SelectionManager
import com.guagua.guagua.image.utils.FastClickChecker
import com.guagua.guagua.image.utils.Utils
import com.guagua.guagua.image.view.RoundImageView

/**
 * Copyright (C), 2020-2020, guagua
 * Author: lixiangchao
 * Date: 2020/4/27 16:30
 * Version: 1.0.0
 * Description: 图片选择适配器
 * History:
 * <author> <time> <version> <desc>
 */
class ImagePickerAdapter(
    private val mContext: Context,
    private val mMediaFileList: List<MediaFile>?
) : RecyclerView.Adapter<ImagePickerAdapter.BaseHolder>() {


    private val isShowCamera: Boolean
    val fastClickChecker = FastClickChecker()

    /**
     * 接口回调，将点击事件向外抛
     */
    private var mOnItemClickListener: OnItemClickListener? = null

    init {
        this.isShowCamera = ConfigManager.isShowCamera()
    }

    override fun getItemCount(): Int {
        if (mMediaFileList == null) {
            return 0
        }
        return if (isShowCamera) mMediaFileList.size + 1 else mMediaFileList.size
    }

    override fun getItemViewType(position: Int): Int {
        var position = position
        if (isShowCamera) {
            if (position == 0) {
                return ItemType.ITEM_TYPE_CAMERA
            }
            //如果有相机存在，position位置需要-1
            position--
        }
        return if (mMediaFileList!![position].duration > 0) {
            ItemType.ITEM_TYPE_VIDEO
        } else {
            ItemType.ITEM_TYPE_IMAGE
        }
    }

    /**
     * 获取item所对应的数据源
     *
     * @param position
     * @return
     */
    fun getMediaFile(position: Int): MediaFile? {
        return if (isShowCamera) {
            if (position == 0) {
                null
            } else mMediaFileList!![position - 1]
        } else mMediaFileList!![position]
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): BaseHolder {
        val view: View
        if (viewType == ItemType.ITEM_TYPE_CAMERA) {
            view = LayoutInflater.from(mContext).inflate(R.layout.item_recyclerview_camera, null)
            return BaseHolder(view)
        } else if (viewType == ItemType.ITEM_TYPE_IMAGE) {
            view = LayoutInflater.from(mContext).inflate(R.layout.item_recyclerview_image, null)
            return ImageHolder(view)
        } else {
            view = LayoutInflater.from(mContext).inflate(R.layout.item_recyclerview_video, null)
            return VideoHolder(view)
        }
    }


    override fun onBindViewHolder(@NonNull holder: BaseHolder, position: Int) {
        val itemType = getItemViewType(position)
        val mediaFile = getMediaFile(position)
        when (itemType) {
            //图片、视频Item
            ItemType.ITEM_TYPE_IMAGE, ItemType.ITEM_TYPE_VIDEO -> {
                val mediaHolder = holder as MediaHolder
                bindMedia(mediaHolder, mediaFile!!)
            }
            //相机Item
            else -> {
            }
        }
        //设置点击事件监听
        if (mOnItemClickListener != null) {
            holder.mSquareRelativeLayout.setOnClickListener(View.OnClickListener { view ->
                if (fastClickChecker.isFastClick(holder.mSquareRelativeLayout)) {
                    return@OnClickListener
                }
                mOnItemClickListener!!.onMediaClick(view, position)
            })

            if (holder is MediaHolder) {
                holder.mImageCheck.setOnClickListener { view ->
                    mOnItemClickListener!!.onMediaCheck(view, position)
                }
                holder.mImageChecked.setOnClickListener { view ->
                    mOnItemClickListener!!.onMediaCheck(view, position)
                }
            }
        }
    }


    /**
     * 绑定数据（图片、视频）
     *
     * @param mediaHolder
     * @param mediaFile
     */
    private fun bindMedia(mediaHolder: MediaHolder, mediaFile: MediaFile) {
        val imagePath = mediaFile.path
        if (!TextUtils.isEmpty(imagePath)) {
            //选择状态（仅是UI表现，真正数据交给SelectionManager管理）
            if (SelectionManager.isImageSelect(imagePath)) {
                mediaHolder.mImageCheck.visibility = View.GONE
                mediaHolder.mImageChecked.visibility = View.VISIBLE
                mediaHolder.mImageChecked.text = SelectionManager.indexOf(imagePath).toString()
            } else {
                mediaHolder.mImageView.visibility = View.VISIBLE
                mediaHolder.mImageCheck.visibility = View.VISIBLE
                mediaHolder.mImageCheck.setImageDrawable(mContext.resources.getDrawable(R.drawable.icon_image_check))
                mediaHolder.mImageChecked.visibility = View.GONE
                mediaHolder.mImageChecked.text = ""
            }

            try {
                imagePath?.let { ConfigManager.getImageLoader().loadImage(mediaHolder.mImageView, it) }
            } catch (e: Exception) {
                e.printStackTrace()
            }

            if (mediaHolder is ImageHolder) {
                //如果是gif图，显示gif标识
                val suffix = imagePath!!.substring(imagePath.lastIndexOf(".") + 1)
                if (suffix.toUpperCase() == "GIF") {
                    mediaHolder.mImageGif.visibility = View.VISIBLE
                } else {
                    mediaHolder.mImageGif.visibility = View.GONE
                }
            }

            if (mediaHolder is VideoHolder) {
                //如果是视频，需要显示视频时长
                val duration = Utils.getVideoDuration(mediaFile.duration)
                mediaHolder.mVideoDuration.text = duration
            }
        }
    }

    /**
     * 图片Item
     */
    class ImageHolder(itemView: View) : MediaHolder(itemView) {
        var mImageGif: ImageView
        init {
            mImageGif = itemView.findViewById(R.id.iv_item_gif)
        }
    }

    /**
     * 视频Item
     */
    class VideoHolder(itemView: View) : MediaHolder(itemView) {

        var mVideoDuration: TextView

        init {
            mVideoDuration = itemView.findViewById(R.id.tv_item_videoDuration)
        }
    }

    /**
     * 媒体Item
     */
    open class MediaHolder(itemView: View) : BaseHolder(itemView) {

        var mImageView: RoundImageView
        var mImageCheck: ImageView
        var mImageChecked: TextView

        init {
            mImageView = itemView.findViewById(R.id.iv_item_image)
            mImageCheck = itemView.findViewById(R.id.iv_item_check)
            mImageChecked = itemView.findViewById(R.id.tv_item_check)
        }
    }

    /**
     * 基础Item
     */
    open class BaseHolder(itemView: View) : RecyclerView.ViewHolder(itemView) {
        var mSquareRelativeLayout: RelativeLayout
        init {
            mSquareRelativeLayout = itemView.findViewById(R.id.srl_item)
        }
    }

    fun setOnItemClickListener(onItemClickListener: OnItemClickListener) {
        this.mOnItemClickListener = onItemClickListener
    }

    interface OnItemClickListener {
        fun onMediaClick(view: View, position: Int)

        fun onMediaCheck(view: View, position: Int)
    }
}