package com.onradar.sdk.model;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

public class RadarEvent {

    public enum RadarEventType {
        UNKNOWN, USER_ENTERED_GEOFENCE, USER_EXITED_GEOFENCE, USER_ENTERED_HOME, USER_EXITED_HOME, USER_ENTERED_OFFICE, USER_EXITED_OFFICE, USER_STARTED_TRAVELING, USER_STOPPED_TRAVELING, USER_ENTERED_PLACE, USER_EXITED_PLACE
    }

    public enum RadarEventConfidence {
        NONE, LOW, MEDIUM, HIGH
    }

    public enum RadarEventVerification {
        ACCEPT, UNVERIFY, REJECT
    }

    private String mId;
    private Date mCreatedAt;
    private boolean mLive;
    private RadarEventType mType;
    private RadarGeofence mGeofence;
    private RadarPlace mPlace;
    private RadarPlace[] mAlternatePlaces;
    private RadarPlace mVerifiedPlace;
    private RadarEventVerification mVerification;
    private RadarEventConfidence mConfidence;
    private float mDuration;

    public static RadarEvent[] eventsFromJSONArray(JSONArray array) throws JSONException, ParseException {
        int numEvents = array.length();
        RadarEvent[] events = new RadarEvent[numEvents];
        for (int i = 0; i < numEvents; i++) {
            JSONObject eventObj = array.getJSONObject(i);
            RadarEvent event = new RadarEvent(eventObj);
            events[i] = event;
        }
        return events;
    }

    public RadarEvent(String _id, Date createdAt, boolean live, RadarEventType type, RadarGeofence geofence, RadarPlace place, RadarPlace[] alternatePlaces, RadarPlace verifiedPlace, RadarEventVerification verification, RadarEventConfidence confidence, float duration) {
        mId = _id;
        mCreatedAt = createdAt;
        mLive = live;
        mType = type;
        mGeofence = geofence;
        mPlace = place;
        mAlternatePlaces = alternatePlaces;
        mVerifiedPlace = verifiedPlace;
        mVerification = verification;
        mConfidence = confidence;
        mDuration = duration;
    }

    public RadarEvent(JSONObject obj) throws JSONException, ParseException {
        if (obj == null) {
            return;
        }

        if (obj.has("_id") && !obj.isNull("_id")) {
            mId = obj.getString("_id");
        }

        if (obj.has("createdAt") && !obj.isNull("createdAt")) {
            String createdAtStr = obj.getString("createdAt");
            DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
            dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            mCreatedAt = dateFormat.parse(createdAtStr);
        }

        if (obj.has("live") && !obj.isNull("live")) {
            mLive = obj.getBoolean("live");
        }

        if (obj.has("type") && !obj.isNull("type")) {
            String typeStr = obj.getString("type");
            if (typeStr.equals("user.entered_geofence")) {
                mType = RadarEventType.USER_ENTERED_GEOFENCE;
            } else if (typeStr.equals("user.exited_geofence")) {
                mType = RadarEventType.USER_EXITED_GEOFENCE;
            } else if (typeStr.equals("user.entered_home")) {
                mType = RadarEventType.USER_ENTERED_HOME;
            } else if (typeStr.equals("user.exited_home")) {
                mType = RadarEventType.USER_EXITED_HOME;
            } else if (typeStr.equals("user.entered_office")) {
                mType = RadarEventType.USER_ENTERED_OFFICE;
            } else if (typeStr.equals("user.exited_office")) {
                mType = RadarEventType.USER_EXITED_OFFICE;
            } else if (typeStr.equals("user.started_traveling")) {
                mType = RadarEventType.USER_STARTED_TRAVELING;
            } else if (typeStr.equals("user.stopped_traveling")) {
                mType = RadarEventType.USER_STOPPED_TRAVELING;
            } else if (typeStr.equals("user.entered_place")) {
                mType = RadarEventType.USER_ENTERED_PLACE;
            } else if (typeStr.equals("user.exited_place")) {
                mType = RadarEventType.USER_EXITED_PLACE;
            } else {
                mType = RadarEventType.UNKNOWN;
            }
        }

        if (obj.has("geofence") && !obj.isNull("geofence")) {
            JSONObject geofenceObj = obj.getJSONObject("geofence");
            mGeofence = new RadarGeofence(geofenceObj);
        }

        if (obj.has("place") && !obj.isNull("place")) {
            JSONObject placeObj = obj.getJSONObject("place");
            mPlace = new RadarPlace(placeObj);
        }

        if (obj.has("alternatePlaces") && !obj.isNull("alternatePlaces")) {
            JSONArray alternatePlacesArr = obj.getJSONArray("alternatePlaces");
            int numAlternatePlaces = alternatePlacesArr.length();

            mAlternatePlaces = new RadarPlace[numAlternatePlaces];

            for (int i = 0; i < numAlternatePlaces; i++) {
                JSONObject alternatePlaceObj = alternatePlacesArr.getJSONObject(i);
                RadarPlace alternatePlace = new RadarPlace(alternatePlaceObj);
                mAlternatePlaces[i] = alternatePlace;
            }
        }

        if (obj.has("verifiedPlace") && !obj.isNull("verifiedPlace")) {
            JSONObject verifiedPlaceObj = obj.getJSONObject("verifiedPlace");
            mVerifiedPlace = new RadarPlace(verifiedPlaceObj);
        }

        if (obj.has("verification") && !obj.isNull("verification")) {
            int verificationInt = obj.getInt("verification");
            if (verificationInt == 1) {
                mVerification = RadarEventVerification.ACCEPT;
            } else if (verificationInt == -1) {
                mVerification = RadarEventVerification.REJECT;
            } else {
                mVerification = RadarEventVerification.UNVERIFY;
            }
        }

        if (obj.has("confidence") && !obj.isNull("confidence")) {
            int confidenceInt = obj.getInt("confidence");
            if (confidenceInt == 3) {
                mConfidence = RadarEventConfidence.HIGH;
            } else if (confidenceInt == 2) {
                mConfidence = RadarEventConfidence.MEDIUM;
            } else if (confidenceInt == 1) {
                mConfidence = RadarEventConfidence.LOW;
            } else {
                mConfidence = RadarEventConfidence.NONE;
            }
        }

        if (obj.has("duration") && !obj.isNull("duration")) {
            mDuration = (float)obj.getDouble("duration");
        }
    }

    /**
     * @return The unique ID for the event, provided by Radar.
     */
    public @NonNull String getId() {
        return mId;
    }

    /**
     * @return The datetime when the event was created.
     */
    public @NonNull Date getCreatedAt() {
        return mCreatedAt;
    }

    /**
     * @return A boolean indicating whether the event was generated for a user created with your live API key.
     */
    public boolean getLive() {
        return mLive;
    }

    /**
     * @return The type of event.
     */
    public @NonNull RadarEventType getType() {
        return mType;
    }

    /**
     * @return The geofence for which the event was generated. May be null for non-geofence events.
     */
    public @Nullable RadarGeofence getGeofence() {
        return mGeofence;
    }

    /**
     * @return The place for which the event was generated. May be null for non-place events.
     */
    public @Nullable RadarPlace getPlace() {
        return mPlace;
    }

    /**
     * @return For place entry events, alternate place candidates. May be null for non-place events.
     */
    public @Nullable RadarPlace[] getAlternatePlaces() {
        return mAlternatePlaces;
    }

    /**
     * @return For accepted place entry events, the verified place. May be null for non-place events.
     */
    public @Nullable RadarPlace getVerifiedPlace() {
        return mVerifiedPlace;
    }

    /**
     * @return The confidence level of the event.
     */
    public @NonNull RadarEventConfidence getConfidence() {
        return mConfidence;
    }

    /**
     * @return The duration between entry and exit events, in minutes, for exit events. 0 for entry events.
     */
    public float getDuration() {
        return mDuration;
    }

}