package com.onradar.sdk.model;

import android.location.Location;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;

public class RadarUser {

    private String mId;
    private String mUserId;
    private String mDeviceId;
    private String mDescription;
    private JSONObject mMetadata;
    private Location mLocation;
    private RadarGeofence[] mGeofences;
    private RadarPlace mPlace;
    private RadarUserInsights mInsights;

    public RadarUser(String _id, String userId, String deviceId, String description, JSONObject metadata, Location location, RadarGeofence[] geofences, RadarPlace place, RadarUserInsights insights) {
        mId = _id;
        mUserId = userId;
        mDeviceId = deviceId;
        mDescription = description;
        mMetadata = metadata;
        mLocation = location;
        mGeofences = geofences;
        mPlace = place;
        mInsights = insights;
    }

    public RadarUser(JSONObject obj) throws JSONException, ParseException {
        if (obj == null) {
            return;
        }

        if (obj.has("_id") && !obj.isNull("_id")) {
            mId = obj.getString("_id");
        }

        if (obj.has("userId") && !obj.isNull("userId")) {
            mUserId = obj.getString("userId");
        }

        if (obj.has("deviceId") && !obj.isNull("deviceId")) {
            mDeviceId = obj.getString("deviceId");
        }

        if (obj.has("description") && !obj.isNull("description")) {
            mDescription = obj.getString("description");
        }

        if (obj.has("metadata") && !obj.isNull("metadata")) {
            mMetadata = obj.getJSONObject("metadata");
        }

        if (obj.has("location") && !obj.isNull("location")) {
            JSONObject locationObj = obj.getJSONObject("location");
            if (locationObj.has("coordinates")) {
                JSONArray coordinatesArr = locationObj.getJSONArray("coordinates");
                double longitude = coordinatesArr.getDouble(0);
                double latitude = coordinatesArr.getDouble(1);
                Location location = new Location("");
                location.setLongitude(longitude);
                location.setLatitude(latitude);
                if (obj.has("locationAccuracy") && !obj.isNull("locationAccuracy")) {
                    float accuracy = (float)obj.getDouble("locationAccuracy");
                    location.setAccuracy(accuracy);
                }
                mLocation = location;
            }
        }

        if (obj.has("geofences") && !obj.isNull("geofences")) {
            JSONArray geofencesArr = obj.getJSONArray("geofences");
            int numGeofences = geofencesArr.length();

            mGeofences = new RadarGeofence[numGeofences];

            for (int i = 0; i < numGeofences; i++) {
                JSONObject geofenceObj = geofencesArr.getJSONObject(i);
                RadarGeofence geofence = new RadarGeofence(geofenceObj);
                mGeofences[i] = geofence;
            }
        }

        if (obj.has("place") && !obj.isNull("place")) {
            mPlace = new RadarPlace((obj.getJSONObject("place")));
        }

        if (obj.has("insights") && !obj.isNull("insights")) {
            mInsights = new RadarUserInsights(obj.getJSONObject("insights"));
        }
    }

    /**
     * @return The unique ID for the user, provided by Radar.
     */
    public @NonNull String getId() {
        return mId;
    }

    /**
     * @return The unique ID for the user, provided when you identified the user.
     */
    public @Nullable String getUserId() {
        return mUserId;
    }

    /**
     * @return The device ID of the user.
     */
    public @Nullable String getDeviceId() {
        return mUserId;
    }

    /**
     * @return An optional description for the user, provided when you started tracking the user.
     */
    public @Nullable String getDescription() {
        return mDescription;
    }

    /**
     * @return An optional set of custom key-value pairs for the user.
     */
    public @Nullable JSONObject getMetadata() {
        return mMetadata;
    }

    /**
     * @return The user's last known location.
     */
    public @NonNull Location getLocation() {
        return mLocation;
    }

    /**
     * @return An array of the user's last known geofences. May be null or empty if the user is not in any geofences.
     */
    public @Nullable RadarGeofence[] getGeofences() {
        return mGeofences;
    }

    /**
     * @return The user's last known place.  May be null if the user is not at a place, or if Places is turned off.
     */
    public @Nullable RadarPlace getPlace() {
        return mPlace;
    }

    /**
     * @return Learned insights for the user. May be null if no insights are available, or if Insights is turned off.
     */
    public @Nullable RadarUserInsights getInsights() {
        return mInsights;
    }

}