package com.onradar.sdk.model;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class RadarPlace {

    private String mId;
    private String mFacebookId;
    private String mName;
    private String[] mCategories;
    private RadarChain mChain;

    public RadarPlace(String _id, String facebookId, String name, String[] categories, RadarChain chain) {
        mId = _id;
        mFacebookId = facebookId;
        mName = name;
        mCategories = categories;
        mChain = chain;
    }

    public RadarPlace(JSONObject obj) throws JSONException {
        if (obj == null) {
            return;
        }

        if (obj.has("_id") && !obj.isNull("_id")) {
            mId = obj.getString("_id");
        }

        if (obj.has("facebookId") && !obj.isNull("facebookId")) {
            mFacebookId = obj.getString("facebookId");
        }

        if (obj.has("name") && !obj.isNull("name")) {
            mName = obj.getString("name");
        }

        if (obj.has("categories") && !obj.isNull("categories")) {
            JSONArray categoriesArray = obj.getJSONArray("categories");
            int numCategories = categoriesArray.length();
            mCategories = new String[numCategories];

            for (int i = 0; i < numCategories; i++) {
                mCategories[i] = categoriesArray.getString(i);
            }
        }

        if (obj.has("chain") && !obj.isNull("chain")) {
            JSONObject chainObj = obj.getJSONObject("chain");
            mChain = new RadarChain(chainObj);
        }
    }

    /**
     * @return The unique ID for the place, provided by Radar.
     */
    public String getId() {
        return mId;
    }

    /**
     * @return A Facebook ID for the place, if known.
     */
    public String getFacebookId() {
        return mFacebookId;
    }

    /**
     * @return The name of the place.
     */
    public String getName() {
        return mName;
    }

    /**
     * @return The categories of the place.
     */
    public String[] getCategories() {
        return mCategories;
    }

    /**
     * @return The chain of the place, if known.
     */
    public RadarChain getChain() {
        return mChain;
    }

    /**
     * @return A boolean indicating whether the place is part of the specified chain.
     */
    public boolean isChain(String slug) {
        if (mChain == null || mChain.getSlug() == null)
            return false;

        return mChain.getSlug().equals(slug);
    }

    /**
     * @return A boolean indicating whether the place has the specified category.
     */
    public boolean hasCategory(String category) {
        if (mCategories == null)
            return false;

        for (int i = 0; i < mCategories.length; i++) {
            if (mCategories[i].equals(category))
                return true;
        }

        return false;
    }

}