package com.onradar.sdk;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.support.annotation.NonNull;

import com.onradar.sdk.Radar.RadarStatus;
import com.onradar.sdk.model.RadarEvent;
import com.onradar.sdk.model.RadarUser;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;

public abstract class RadarReceiver extends BroadcastReceiver {

    public static final String RECEIVED = "com.onradar.sdk.RECEIVED";

    @Override
    public void onReceive(Context context, Intent intent) {
        if (intent != null) {
            if (intent.getAction().equals(RECEIVED)) {
                String response = intent.getStringExtra("response");
                Location location = intent.getExtras().getParcelable("location");

                if (response != null && location != null) {
                    try {
                        JSONObject responseObj = new JSONObject(response);

                        JSONArray eventsArr = responseObj.getJSONArray("events");
                        int numEvents = eventsArr.length();
                        RadarEvent[] events = new RadarEvent[numEvents];
                        for (int i = 0; i < numEvents; i++) {
                            JSONObject eventObj = eventsArr.getJSONObject(i);
                            RadarEvent event = new RadarEvent(eventObj);
                            events[i] = event;
                        }

                        JSONObject userObj = responseObj.getJSONObject("user");
                        RadarUser user = new RadarUser(userObj);

                        if (events.length > 0) {
                            this.onEventsReceived(context, events, user);
                        }

                        this.onLocationUpdated(context, location, user);
                    } catch (JSONException | ParseException e) {
                        this.onError(context, RadarStatus.ERROR_UNKNOWN);
                    }
                } else {
                    RadarStatus status = RadarStatus.fromOrdinal(intent.getIntExtra("status", -1));
                    if (status != RadarStatus.SUCCESS && status != RadarStatus.UNKNOWN)
                        this.onError(context, status);
                }
            }
        }
    }

    /**
     * Tells the receiver that events were received for the current user. Note that events can also be delivered server-side via webhooks.
     *
     * @param context The context.
     * @param events The events received.
     * @param user The current user.
     */
    public abstract void onEventsReceived(@NonNull Context context, @NonNull RadarEvent[] events, @NonNull RadarUser user);

    /**
     * Tells the receiver that the current user's location was updated.
     *
     * @param context The context.
     * @param location The location.
     * @param user The current user.
     */
    public void onLocationUpdated(@NonNull Context context, @NonNull Location location, @NonNull RadarUser user) {

    }

    /**
     * Tells the receiver that a request failed.
     *
     * @param context The context.
     * @param status The status.
     */
    public abstract void onError(@NonNull Context context, @NonNull RadarStatus status);

}
