package com.onradar.sdk;

import android.app.Activity;
import android.content.Context;
import android.location.Location;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.onradar.sdk.model.RadarEvent;

import org.json.JSONObject;

import java.util.Map;

public class Radar {

    public enum RadarStatus {
        SUCCESS,
        ERROR_PUBLISHABLE_KEY,
        ERROR_USER_ID,
        ERROR_PERMISSIONS,
        ERROR_LOCATION,
        ERROR_NETWORK,
        ERROR_UNAUTHORIZED,
        ERROR_SERVER,
        ERROR_UNKNOWN,
        UNKNOWN;

        public static RadarStatus fromOrdinal(int ordinal) {
            switch (ordinal) {
                case 0:
                    return RadarStatus.SUCCESS;
                case 1:
                    return RadarStatus.ERROR_PUBLISHABLE_KEY;
                case 2:
                    return RadarStatus.ERROR_USER_ID;
                case 3:
                    return RadarStatus.ERROR_PERMISSIONS;
                case 4:
                    return RadarStatus.ERROR_LOCATION;
                case 5:
                    return RadarStatus.ERROR_NETWORK;
                case 6:
                    return RadarStatus.ERROR_UNAUTHORIZED;
                case 7:
                    return RadarStatus.ERROR_SERVER;
                case 8:
                    return RadarStatus.ERROR_UNKNOWN;
                default:
                    return RadarStatus.UNKNOWN;
            }
        }
    }

    public enum RadarPlacesProvider {
        NONE,
        FACEBOOK;
    }

    public enum RadarPriority {
        RESPONSIVENESS,
        EFFICIENCY;
    }

    /**
     * Initializes the Radar SDK. You must call this method in the onCreate() method of your main activity. Before calling this method, your publishable API key must have been added as a string resource in your project.
     *
     * @param context The main activity.
     */
    public static void initialize(@NonNull Context context) {
        _Radar.getSharedInstance().initialize(context, null);
    }

    /**
     * Initializes the Radar SDK. You must call this method in the onCreate() method of your main activity.
     *
     * @param context The main activity.
     * @param publishableKey Your publishable API key.
     */
    public static void initialize(@NonNull Context context, @NonNull String publishableKey) {
        _Radar.getSharedInstance().initialize(context, publishableKey);
    }

    /**
     * Identifies the user. You must call this method once before calling trackOnce() or startTracking().
     *
     * @param userId A stable unique ID for the user.
     */
    public static void setUserId(@NonNull String userId) {
        _Radar.getSharedInstance().setUserId(userId);
    }

    /**
     * Reidentifies the user, changing their user ID without creating a new user. To reidentify the user, you must call this method with the user's old stable unique ID, call setUserId() with the user's new stable unique ID, then track the user's location.
     *
     * @param oldUserId The old stable unique ID for the user. If null, the previous ID will be cleared.
     */
    public static void reidentifyUser(@Nullable String oldUserId) {
        _Radar.getSharedInstance().reidentifyUser(oldUserId);
    }

    /**
     * Sets an optional description for the user, displayed in the dashboard.
     *
     * @param description An optional description for the user. If null, the previous description will be cleared.
     */
    public static void setDescription(@Nullable String description) {
        _Radar.getSharedInstance().setDescription(description);
    }

    /**
     * Sets an optional set of custom key-value pairs for the user.
     *
     * @param metadata A set of custom key-value pairs for the user. Must have 16 or fewer keys and values of type string, boolean, or number. If nil, the previous metadata will be cleared.
     **/
    public static void setMetadata(@Nullable JSONObject metadata) {
        _Radar.getSharedInstance().setMetadata(metadata);
    }

    /**
     * Sets the provider for Places data.
     *
     * @param provider The provider for Places data.
     */
    public static void setPlacesProvider(RadarPlacesProvider provider) {
        _Radar.getSharedInstance().setPlacesProvider(provider);
    }

    /**
     * Determines whether the user has granted location permissions for the app.
     *
     * @return A boolean indicating whether the user has granted location permissions for the app.
     */
    public static boolean checkSelfPermissions() {
        return _Radar.getSharedInstance().checkSelfPermissions();
    }

    /**
     * Requests permission to track the user's location.
     *
     * @param activity The activity from which to show the permissions prompt.
     */
    public static void requestPermissions(@NonNull Activity activity) {
        _Radar.getSharedInstance().requestPermissions(activity);
    }

    /**
     * Tracks the user's location once in the foreground. Before calling this method, you must have called setUserId() once to identify the user, and the user must have granted location permissions for the app.
     *
     * @param callback An optional callback.
     */
    public static void trackOnce(@Nullable RadarCallback callback) {
        _Radar.getSharedInstance().trackOnce(callback);
    }

    /**
     * Sets the priority for background tracking.
     *
     * @param priority The priority for background tracking. RESPONSIVENESS, the default, triggers more frequent wakeups for better responsiveness and reliability. EFFICIENCY triggers less frequent wakeups for better battery efficiency and vitals.
     */
    public static void setTrackingPriority(RadarPriority priority) {
        _Radar.getSharedInstance().setTrackingPriority(priority);
    }

    /**
     * Starts tracking the user's location in the background. Before calling this method, you must have called setUserId() once to identify the user, and the user must have granted location permissions for the app
     */
    public static void startTracking() {
        _Radar.getSharedInstance().startTracking();
    }

    /**
     * Stops tracking the user's location in the background.
     */
    public static void stopTracking() {
        _Radar.getSharedInstance().stopTracking();
    }

    /**
     * Returns a boolean indicating whether the user's location is being tracked in the background.
     *
     * @return A boolean indicating whether the user's location is being tracked in the background.
     */
    public static boolean isTracking() {
        return _Radar.getSharedInstance().isTracking();
    }

    /**
     * Manually updates the user's location. Before calling this method, you must have called setUserId() once to identify the user.
     *
     * @param location A location for the user. Must have a valid latitude, longitude, and accuracy.
     * @param callback An optional callback.
     */
    public static void updateLocation(@NonNull Location location, @Nullable RadarCallback callback) {
        _Radar.getSharedInstance().updateLocation(location, callback);
    }

    /**
     * Accepts an event. Events can be accepted or rejected after user check-ins or other forms of verification. Event verifications will be used to improve the accuracy and confidence level of future events.
     *
     * @param eventId The ID of the event to accept.
     * @param verifiedPlaceId For place entry events, the ID of the verified place. May be null.
     */
    public static void acceptEvent(@NonNull String eventId, @Nullable String verifiedPlaceId) {
        _Radar.getSharedInstance().acceptEvent(eventId, verifiedPlaceId);
    }

    /**
     * Rejects an event. Events can be accepted or rejected after user check-ins or other forms of verification. Event verifications will be used to improve the accuracy and confidence level of future events.
     *
     * @param eventId The ID of the event to reject.
     */
    public static void rejectEvent(@NonNull String eventId) {
        _Radar.getSharedInstance().rejectEvent(eventId);
    }

    /**
     * Returns a boolean indicating whether the user's wi-fi is enabled. Location accuracy and reliability are greatly improved when wi-fi is enabled.
     *
     * @return A boolean indicating whether the user's wi-fi is enabled.
     */
    public static boolean isWifiEnabled() {
        return _Radar.getSharedInstance().isWifiEnabled();
    }

}
