/*
 * Decompiled with CFR 0.152.
 */
package com.okta.sdk.impl.http;

import com.okta.commons.lang.Assert;
import com.okta.commons.lang.Strings;
import com.okta.sdk.impl.config.ClientConfiguration;
import com.okta.sdk.impl.http.HttpHeaders;
import com.okta.sdk.impl.http.QueryString;
import com.okta.sdk.impl.http.Request;
import com.okta.sdk.impl.http.RequestExecutor;
import com.okta.sdk.impl.http.Response;
import com.okta.sdk.impl.http.RestException;
import com.okta.sdk.impl.http.support.BackoffStrategy;
import java.io.InputStream;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RetryRequestExecutor
implements RequestExecutor {
    private static final Logger log = LoggerFactory.getLogger(RetryRequestExecutor.class);
    private static final int DEFAULT_MAX_BACKOFF_IN_MILLISECONDS = 20000;
    private static final int DEFAULT_MAX_RETRIES = 4;
    private int maxRetries = 4;
    private int maxElapsedMillis = 0;
    private BackoffStrategy backoffStrategy;
    private final RequestExecutor delegate;

    public RetryRequestExecutor(ClientConfiguration clientConfiguration, RequestExecutor delegate) {
        this.delegate = delegate;
        if (clientConfiguration.getRetryMaxElapsed() >= 0) {
            this.maxElapsedMillis = clientConfiguration.getRetryMaxElapsed() * 1000;
        }
        if (clientConfiguration.getRetryMaxAttempts() > 0) {
            this.maxRetries = clientConfiguration.getRetryMaxAttempts();
        }
    }

    /*
     * Loose catch block
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public Response executeRequest(Request request) throws RestException {
        Assert.notNull((Object)request, (String)"Request argument cannot be null.");
        int retryCount = 0;
        Response response = null;
        String requestId = null;
        Timer timer = new Timer();
        QueryString originalQuery = new QueryString();
        originalQuery.putAll(request.getQueryString());
        HttpHeaders originalHeaders = new HttpHeaders();
        originalHeaders.putAll(request.getHeaders());
        while (true) {
            try {
                do {
                    if (retryCount > 0) {
                        InputStream content;
                        request.setQueryString(originalQuery);
                        request.setHeaders(originalHeaders);
                        if (requestId == null) {
                            requestId = this.getRequestId(response);
                        }
                        if ((content = request.getBody()) != null && content.markSupported()) {
                            content.reset();
                        }
                        try {
                            this.pauseBeforeRetry(retryCount, response, timer.split());
                        }
                        catch (RestException e) {
                            if (log.isDebugEnabled()) {
                                log.warn("Unable to pause for retry: {}", (Object)e.getMessage(), (Object)e);
                            } else {
                                log.warn("Unable to pause for retry: {}", (Object)e.getMessage());
                            }
                            if (response != null) return response;
                            throw new RestException("Unable to execute HTTP request: " + e.getMessage(), e);
                        }
                    }
                    this.setOktaHeaders(request, requestId, ++retryCount);
                } while (this.shouldRetry(response = this.doExecuteRequest(request), retryCount, timer.split()));
                return response;
            }
            catch (SocketException | SocketTimeoutException e) {
                if (!this.shouldRetry(retryCount, timer.split())) {
                    throw new RestException("Unable to execute HTTP request: " + e.getMessage(), e);
                }
                log.debug("Retrying on {}: {}", (Object)e.getClass().getName(), (Object)e.getMessage());
                continue;
            }
            catch (RestException e) {
                if (!e.isRetryable() || !this.shouldRetry(retryCount, timer.split())) throw e;
                continue;
            }
            break;
        }
        catch (Exception e) {
            throw new RestException("Unable to execute HTTP request: " + e.getMessage(), e);
        }
    }

    protected Response doExecuteRequest(Request request) {
        return this.delegate.executeRequest(request);
    }

    private void pauseBeforeRetry(int retries, Response response, long timeElapsed) throws RestException {
        long delay = -1L;
        long timeElapsedLeft = (long)this.maxElapsedMillis - timeElapsed;
        if (!this.shouldRetry(retries, timeElapsed)) {
            throw this.failedToRetry();
        }
        if (this.backoffStrategy != null) {
            delay = Math.min(this.backoffStrategy.getDelayMillis(retries), timeElapsedLeft);
        } else if (response != null && response.getHttpStatus() == 429) {
            delay = this.get429DelayMillis(response);
            if (!this.shouldRetry(retries, timeElapsed + delay)) {
                throw this.failedToRetry();
            }
            log.debug("429 detected, will retry in {}ms, attempt number: {}", (Object)delay, (Object)retries);
        }
        if (delay < 0L) {
            delay = Math.min(this.getDefaultDelayMillis(retries), timeElapsedLeft);
        }
        if (delay < 0L) {
            throw this.failedToRetry();
        }
        log.debug("Retryable condition detected, will retry in {}ms, attempt number: {}", (Object)delay, (Object)retries);
        try {
            Thread.sleep(delay);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RestException(e.getMessage(), e);
        }
    }

    private long get429DelayMillis(Response response) {
        long resetLimit = this.getRateLimitResetValue(response);
        if (resetLimit == -1L) {
            return -1L;
        }
        Date requestDate = this.dateFromHeader(response);
        if (requestDate == null) {
            return -1L;
        }
        long waitUntil = resetLimit * 1000L;
        long requestTime = requestDate.getTime();
        long delay = Math.max(waitUntil - requestTime + 1000L, 1000L);
        log.debug("429 wait: Math.max({} - {} + 1s), 1s = {})", new Object[]{waitUntil, requestTime, delay});
        return delay;
    }

    private Date dateFromHeader(Response response) {
        Date result = null;
        long dateLong = response.getHeaders().getDate();
        if (dateLong > 0L) {
            result = new Date(dateLong);
        }
        return result;
    }

    private long getDefaultDelayMillis(int retries) {
        long scaleFactor = 300L;
        long result = (long)(Math.pow(2.0, retries) * (double)scaleFactor);
        return Math.min(result, 20000L);
    }

    private boolean shouldRetry(int retryCount, long timeElapsed) {
        return !(this.maxRetries <= 0 && this.maxElapsedMillis <= 0 || this.maxRetries > 0 && retryCount > this.maxRetries || this.maxElapsedMillis > 0 && timeElapsed >= (long)this.maxElapsedMillis);
    }

    private boolean shouldRetry(Response response, int retryCount, long timeElapsed) {
        int httpStatus = response.getHttpStatus();
        return this.shouldRetry(retryCount, timeElapsed) && (httpStatus == 429 || httpStatus == 503 || httpStatus == 504);
    }

    private RestException failedToRetry() {
        return new RestException("Cannot retry request, next request will exceed retry configuration.");
    }

    private long getRateLimitResetValue(Response response) {
        return response.getHeaders().getOrDefault("X-Rate-Limit-Reset", Collections.emptyList()).stream().filter(value -> !Strings.isEmpty((Object)value)).filter(value -> value.chars().allMatch(Character::isDigit)).map(Long::parseLong).filter(value -> value > 0L).min(Comparator.naturalOrder()).orElse(-1L);
    }

    private String getRequestId(Response response) {
        if (response != null) {
            return response.getHeaders().getFirst("X-Okta-Request-Id");
        }
        return null;
    }

    private void setOktaHeaders(Request request, String requestId, int retryCount) {
        if (Strings.hasText((String)requestId)) {
            request.getHeaders().add("X-Okta-Retry-For", requestId);
        }
        if (retryCount > 1) {
            request.getHeaders().add("X-Okta-Retry-Count", Integer.toString(retryCount));
        }
    }

    @Deprecated
    public BackoffStrategy getBackoffStrategy() {
        return this.backoffStrategy;
    }

    @Deprecated
    public void setBackoffStrategy(BackoffStrategy backoffStrategy) {
        this.backoffStrategy = backoffStrategy;
    }

    @Deprecated
    public int getNumRetries() {
        return this.maxRetries;
    }

    @Deprecated
    public void setNumRetries(int numRetries) {
        this.maxRetries = numRetries;
    }

    @Deprecated
    int getMaxElapsedMillis() {
        return this.maxElapsedMillis;
    }

    @Deprecated
    void setMaxElapsedMillis(int maxElapsedMillis) {
        this.maxElapsedMillis = maxElapsedMillis;
    }

    private static class Timer {
        private long startTime = System.currentTimeMillis();

        private Timer() {
        }

        long split() {
            return System.currentTimeMillis() - this.startTime;
        }
    }
}

