/*
 * oauth2-oidc-sdk
 *
 * Copyright 2012-2021, Connect2id Ltd and contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use
 * this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */

package com.nimbusds.oauth2.sdk.token;


import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.http.HTTPResponse;
import com.nimbusds.oauth2.sdk.rar.AuthorizationDetail;
import com.nimbusds.oauth2.sdk.util.JSONArrayUtils;
import com.nimbusds.oauth2.sdk.util.JSONObjectUtils;
import com.nimbusds.oauth2.sdk.util.MultivaluedMapUtils;
import com.nimbusds.oauth2.sdk.util.StringUtils;
import net.minidev.json.JSONArray;
import net.minidev.json.JSONObject;

import java.net.URI;
import java.util.List;
import java.util.Map;


/**
 * Access token parse utilities.
 */
public class AccessTokenParseUtils {
	
	
	/**
	 * Parses a {@code token_type} from a JSON object and ensures it
	 * matches the specified.
	 *
	 * @param jsonObject The JSON object. Must not be {@code null}.
	 * @param type       The expected token type. Must not be {@code null}.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static void parseAndEnsureTypeFromJSONObject(final JSONObject jsonObject, final AccessTokenType type)
		throws ParseException {
		
		if (! new AccessTokenType(JSONObjectUtils.getNonBlankString(jsonObject, "token_type")).equals(type)) {
			throw new ParseException("The token type must be " + type);
		}
	}
	
	
	/**
	 * Parses an {code access_token} value from a JSON object.
	 *
	 * @param params The JSON object. Must not be {@code null}.
	 *
	 * @return The access token value.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static String parseValueFromJSONObject(final JSONObject params)
		throws ParseException {
		
		return JSONObjectUtils.getNonBlankString(params, "access_token");
	}
	
	
	/**
	 * Parses an access token {@code expires_in} parameter from a JSON
	 * object.
	 *
	 * @param jsonObject The JSON object. Must not be {@code null}.
	 *
	 * @return The access token lifetime, in seconds, zero if not
	 *         specified.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static long parseLifetimeFromJSONObject(final JSONObject jsonObject)
		throws ParseException {
		
		if (jsonObject.containsKey("expires_in")) {
			// Lifetime can be a JSON number or string
			if (jsonObject.get("expires_in") instanceof Number) {
				return JSONObjectUtils.getLong(jsonObject, "expires_in");
			} else {
				String lifetimeStr = JSONObjectUtils.getNonBlankString(jsonObject, "expires_in");
				try {
					return Long.parseLong(lifetimeStr);
				} catch (NumberFormatException e) {
					throw new ParseException("expires_in must be an integer");
				}
			}
		}
		
		return 0L;
	}
	
	
	/**
	 * Parses a {@code scope} parameter from a JSON object.
	 *
	 * @param jsonObject The JSON object. Must not be {@code null}.
	 *
	 * @return The scope, {@code null} if not specified.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static Scope parseScopeFromJSONObject(final JSONObject jsonObject)
		throws ParseException {
		
		return Scope.parse(JSONObjectUtils.getString(jsonObject, "scope", null));
	}


	/**
	 * Parses an {@code authorization_details} parameter from a JSON
	 * object.
	 *
	 * @param jsonObject The JSON object. Must not be {@code null}.
	 *
	 * @return The authorisation details, {@code null} if not specified.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static List<AuthorizationDetail> parseAuthorizationDetailsFromJSONObject(final JSONObject jsonObject)
		throws ParseException {

		JSONArray jsonArray = JSONObjectUtils.getJSONArray(jsonObject, "authorization_details", null);

		if (jsonArray == null) {
			return null;
		}

		return AuthorizationDetail.parseList(JSONArrayUtils.toJSONObjectList(jsonArray));
	}

	
	/**
	 * Parses an {@code issued_token_type} parameter from a JSON object.
	 *
	 * @param jsonObject The JSON object. Must not be {@code null}.
	 *
	 * @return The issued token type, {@code null} if not specified.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static TokenTypeURI parseIssuedTokenTypeFromJSONObject(final JSONObject jsonObject)
		throws ParseException {
		
		String issuedTokenTypeString = JSONObjectUtils.getString(jsonObject, "issued_token_type", null);

		if (issuedTokenTypeString == null) {
			return null;
		}

		try {
			return TokenTypeURI.parse(issuedTokenTypeString);
		} catch (ParseException e) {
			throw new ParseException("Invalid issued_token_type", e);
		}
	}


	private static class GenericTokenSchemeError extends TokenSchemeError {

		private static final long serialVersionUID = -8049139536364886132L;

		public GenericTokenSchemeError(final AccessTokenType scheme,
					       final String code,
					       final String description,
					       final int httpStatusCode) {
			super(scheme, code, description, httpStatusCode, null, null, null);
		}

		@Override
		public TokenSchemeError setDescription(String description) {
			return this;
		}

		@Override
		public TokenSchemeError appendDescription(String text) {
			return this;
		}

		@Override
		public TokenSchemeError setHTTPStatusCode(int httpStatusCode) {
			return this;
		}

		@Override
		public TokenSchemeError setURI(URI uri) {
			return this;
		}

		@Override
		public TokenSchemeError setRealm(String realm) {
			return this;
		}

		@Override
		public TokenSchemeError setScope(Scope scope) {
			return this;
		}
	}


	private static TokenSchemeError getTypedMissingTokenError(final AccessTokenType type) {
		if (AccessTokenType.BEARER.equals(type)) {
			return BearerTokenError.MISSING_TOKEN;
		} else if (AccessTokenType.DPOP.equals(type)) {
			return DPoPTokenError.MISSING_TOKEN;
		} else {
			return new GenericTokenSchemeError(type, null, null, HTTPResponse.SC_UNAUTHORIZED);
		}
	}


	private static TokenSchemeError getTypedInvalidRequestError(final AccessTokenType type) {
		if (AccessTokenType.BEARER.equals(type)) {
			return BearerTokenError.INVALID_REQUEST;
		} else if (AccessTokenType.DPOP.equals(type)) {
			return DPoPTokenError.INVALID_REQUEST;
		} else {
			return new GenericTokenSchemeError(type, "invalid_request", "Invalid request", HTTPResponse.SC_BAD_REQUEST);
		}
	}
	
	
	/**
	 * Parses an access token value from an {@code Authorization} HTTP
	 * request header.
	 *
	 * @param header The {@code Authorization} header value, {@code null}
	 *               if not specified.
	 * @param type   The expected access token type, such as
	 *               {@link AccessTokenType#BEARER} or
	 *               {@link AccessTokenType#DPOP}. Must not be
	 *               {@code null}.
	 *
	 * @return The access token value.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static String parseValueFromAuthorizationHeader(final String header,
							       final AccessTokenType type)
		throws ParseException {
		
		if (StringUtils.isBlank(header)) {
			TokenSchemeError schemeError = getTypedMissingTokenError(type);
			throw new ParseException("Missing HTTP Authorization header", schemeError);
		}
		
		String[] parts = header.split("\\s", 2);
		
		if (parts.length != 2) {
			TokenSchemeError schemeError = getTypedInvalidRequestError(type);
			throw new ParseException("Invalid HTTP Authorization header value", schemeError);
		}
		
		if (! parts[0].equalsIgnoreCase(type.getValue())) {
			TokenSchemeError schemeError = getTypedInvalidRequestError(type);
			throw new ParseException("Token type must be " + type, schemeError);
		}
		
		if (StringUtils.isBlank(parts[1])) {
			TokenSchemeError schemeError = getTypedInvalidRequestError(type);
			throw new ParseException("Invalid HTTP Authorization header value: Missing token", schemeError);
		}
		
		return parts[1];
	}
	
	
	/**
	 * Parses an {@code access_token} values from a query or form
	 * parameters.
	 *
	 * @param parameters The parameters. Must not be {@code null}.
	 * @param type       The expected access token type, such as
	 *                   {@link AccessTokenType#BEARER} or
	 *                   {@link AccessTokenType#DPOP}. Must not be
	 *                   {@code null}.
	 *
	 * @return The access token value.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static String parseValueFromQueryParameters(final Map<String, List<String>> parameters,
							   final AccessTokenType type)
		throws ParseException {
		
		if (! parameters.containsKey("access_token")) {
			TokenSchemeError schemeError = getTypedMissingTokenError(type);
			throw new ParseException("Missing access token parameter", schemeError);
		}
		
		String accessTokenValue = MultivaluedMapUtils.getFirstValue(parameters, "access_token");
		
		if (StringUtils.isBlank(accessTokenValue)) {
			TokenSchemeError schemeError = getTypedInvalidRequestError(type);
			throw new ParseException("Blank / empty access token", schemeError);
		}
		
		return accessTokenValue;
	}
	
	
	/**
	 * Parses an {@code access_token} value from a query or form
	 * parameters.
	 *
	 * @param parameters The query parameters. Must not be {@code null}.
	 *
	 * @return The access token value.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static String parseValueFromQueryParameters(final Map<String, List<String>> parameters)
		throws ParseException {
		
		String accessTokenValue = MultivaluedMapUtils.getFirstValue(parameters, "access_token");
		
		if (StringUtils.isBlank(accessTokenValue)) {
			throw new ParseException("Missing access token");
		}
		
		return accessTokenValue;
	}
	
	
	/**
	 * Determines the access token type from an {@code Authorization} HTTP
	 * request header.
	 *
	 * @param header The {@code Authorization} header value. Must not be
	 *               {@code null}.
	 *
	 * @return The access token type.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static AccessTokenType determineAccessTokenTypeFromAuthorizationHeader(final String header)
		throws ParseException {

		String[] parts = header.split("\\s", 2);

		if (parts.length < 2 || StringUtils.isBlank(parts[0]) || StringUtils.isBlank(parts[1])) {
			throw new ParseException("Invalid Authorization header");
		}

		if (parts[0].equalsIgnoreCase(AccessTokenType.BEARER.getValue())) {
			return AccessTokenType.BEARER;
		}

		if (parts[0].equalsIgnoreCase(AccessTokenType.DPOP.getValue())) {
			return AccessTokenType.DPOP;
		}

		return new AccessTokenType(parts[0]);
	}
	
	
	private AccessTokenParseUtils() {}
}
