/**
 * 
 */
package com.newrelic.agent.xray;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.newrelic.agent.Agent;
import com.newrelic.agent.IRPMService;
import com.newrelic.agent.commands.AbstractCommand;
import com.newrelic.agent.commands.CommandException;

/**
 * Used when the collector sends us an active_xray_sessions command as part of get_agent_commands. Passes the payload of
 * active_xray_sessions into xRaySessionService.processSessionsList() method which compares the incoming list to the
 * list of X-Ray sessions we already know about and starts/stops agent X-Ray sessions as needed.
 * 
 * @author ddelany
 * 
 */
public class StartXRayCommand extends AbstractCommand {

    public final static String COMMAND_NAME = "active_xray_sessions";
    private final static String DISABLED_MESSAGE = "The X-Ray service is disabled";
    private IXRaySessionService xRaySessionService;

    public StartXRayCommand(XRaySessionService xRaySessionService) {
        super(COMMAND_NAME);
        this.xRaySessionService = xRaySessionService;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.newrelic.agent.commands.Command#process(com.newrelic.agent.IRPMService, java.util.Map)
     */
    @SuppressWarnings("rawtypes")
    public Map<?, ?> process(IRPMService rpmService, Map arguments) throws CommandException {
        if (xRaySessionService.isEnabled()) {
            return processEnabled(rpmService, arguments);
        }
        return processDisabled(rpmService, arguments);
    }

    @SuppressWarnings("rawtypes")
    private Map processDisabled(IRPMService rpmService, Map arguments) {
        Agent.LOG.debug(DISABLED_MESSAGE);
        try {
            this.xRaySessionService.stop();
        } catch (Exception e) {
            Agent.LOG.warning("Error disabling X-Ray Session service: " + e.getMessage());
        }
        return Collections.EMPTY_MAP;
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    private Map processEnabled(IRPMService rpmService, Map arguments) {
        Object xray_ids = arguments.remove("xray_ids");
        List<Long> xrayIds;
        if (xray_ids instanceof List) {
            xrayIds = (List<Long>) xray_ids;
        } else {
            xrayIds = Collections.emptyList();
        }
        return this.xRaySessionService.processSessionsList(xrayIds, rpmService);
    }

}
