// ========================================================================
// Copyright 2004-2005 Mort Bay Consulting Pty. Ltd.
// ------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at 
// http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ========================================================================

package com.newrelic.agent.util;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.regex.Pattern;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;

import com.newrelic.org.apache.axis.encoding.Base64;

/**
 * String helper methods.
 * 
 * @author sdaubin
 * 
 */
public class Strings {
    public static final String NEWRELIC_DEPENDENCY_INTERNAL_PACKAGE_PREFIX = "com/newrelic/agent/deps/";
    private static final String NEWRELIC_DEPENDENCY_PACKAGE_PREFIX = "com.newrelic.agent.deps.";
    private static final StringMap NO_OP_STRING_MAP = new StringMap() {

        @Override
        public Object addString(String string) {
            return string;
        }

        @Override
        public Map<Object, String> getStringMap() {
            return ImmutableMap.<Object, String>of();
        }
        
    };

    private Strings() {
    }

    public static boolean isBlank(String str) {
        return str == null || str.length() == 0;
    }

    public static Collection<String> trim(Collection<String> strings) {
        Collection<String> trimmedList = new ArrayList<String>(strings.size());
        for (String string : strings) {
            trimmedList.add(string.trim());
        }
        return trimmedList;
    }

    public static String unquoteDatabaseName(String s) {
        int index = s.indexOf('.');
        if (index > 0) {
            return new StringBuilder(s.length()).append(unquote(s.substring(0, index))).append('.').append(
                    unquote(s.substring(index + 1))).toString();
        } else {
            return unquote(s);
        }
    }

    public static String removeBrackets(String s){
        String result = s.replace("[", "");
        result = result.replace("]", "");
        return result;
    }

    /**
     * Join a list of strings with a delimiter.
     * 
     * @param delimiter
     * @param strings
     * @return
     */
    public static String join(char delimiter, String... strings) {
        if (strings.length == 0) {
            return null;
        } else if (strings.length == 1) {
            return strings[0];
        } else {
            int length = strings.length - 1;
            for (String s : strings) {
                length += s.length();
            }
            StringBuilder sb = new StringBuilder(length);
            sb.append(strings[0]);
            for (int i = 1; i < strings.length; i++) {
                if (!strings[i].isEmpty()) {
                    sb.append(delimiter).append(strings[i]);
                }
            }
            return sb.toString();
        }
    }

    public static String join(String... strings) {
        if (strings.length == 0) {
            return null;
        } else if (strings.length == 1) {
            return strings[0];
        } else {
            int length = 0;
            for (String s : strings) {
                length += s.length();
            }
            StringBuilder sb = new StringBuilder(length);
            for (int i = 0; i < strings.length; i++) {
                if (!strings[i].isEmpty()) {
                    sb.append(strings[i]);
                }
            }
            return sb.toString();
        }
    }

    /**
     * Similar to {@link String#split(String)}, but the delimiter is not a regular expression.
     * 
     * @param string
     * @param delimiter
     * @return
     */
    public static String[] split(String string, String delimiter) {
        StringTokenizer tokenizer = new StringTokenizer(string, delimiter);
        List<String> segments = new ArrayList<String>(4);
        while (tokenizer.hasMoreTokens()) {
            segments.add(tokenizer.nextToken());
        }
        return segments.toArray(new String[segments.size()]);
    }

    /**
     * Unquote a string.
     */
    public static String unquote(String string) {
        if (string == null || string.length() < 2) {
            return string;
        }

        char first = string.charAt(0);
        char last = string.charAt(string.length() - 1);
        if (first != last || (first != '"' && first != '\'' && first != '`')) {
            return string;
        }

        return string.substring(1, string.length() - 1);
    }

    /**
     * Returns true if the string is empty (null or of zero length).
     * 
     * @param string
     * @return
     */
    public static boolean isEmpty(String string) {
        return string == null || string.length() == 0;
    }

    /**
     * We use jarjar to repackage third party libraries. The side effect is that it make it hard to reference these
     * libraries in our instrumentation because the class names get rewritten - org.apache.commons.ClassName becomes
     * com.newrelic.org.apache.commons.ClassName.
     * 
     * This method rewrites these repackages class names to their original package.
     * 
     * @param className
     * @return
     */
    public static String fixClassName(String className) {
        return trimName(className, NEWRELIC_DEPENDENCY_PACKAGE_PREFIX);
    }

    private static String trimName(String fullName, String prefix) {
        if (fullName.startsWith(prefix)) {
            return fullName.substring(prefix.length());
        }
        return fullName;
    }

    /**
     * We use jarjar to repackage third party libraries. The side effect is that it make it hard to reference these
     * libraries in our instrumentation because the class names get rewritten - org/apache/commons/ClassName becomes
     * com/newrelic/org/apache/commons/ClassName.
     * 
     * This method rewrites these repackages class names to their original package.
     * 
     * @param className
     * @return
     */
    public static String fixInternalClassName(String className) {
        className = className.replace('.', '/');
        return trimName(className, NEWRELIC_DEPENDENCY_INTERNAL_PACKAGE_PREFIX);
    }

    /**
     * Returns a regex pattern for the given glob. This isn't a complete implementation - I just escaped enough stuff to
     * cover my use case.
     * 
     * @return
     */
    public static String getGlobPattern(String glob) {
        StringBuilder b = new StringBuilder().append('^');
        for (char c : glob.toCharArray()) {
            switch (c) {
            case '.':
                b.append("\\.");
                break;
            case '/':
                b.append("\\/");
                break;
            case '*':
                b.append(".*");
                break;
            default:
                b.append(c);
            }
        }
        return b.toString();
    }

    public static Pattern getPatternFromGlobs(List<String> globs) {
        List<String> patterns = Lists.newArrayListWithCapacity(globs.size());
        for (String glob : globs) {
            patterns.add('(' + getGlobPattern(glob) + ')');
        }
        String pattern = Joiner.on('|').join(patterns);
        return Pattern.compile(pattern);
    }
    
    /**
     * Returns a no op StringMap that just returns the strings passed into it. 
     * @return
     */
    public static StringMap getPassthroughStringMap() {
        return NO_OP_STRING_MAP;
    }

    public static StringMap newStringMap() {
        return new StringMap() {
            private final HashFunction hashFunction = Hashing.murmur3_128(394852370);
            private final Charset charSet = Charset.forName("UTF-8");

            final Map<Object,String> stringMap = Maps.newConcurrentMap();
            
            @Override
            public Map<Object, String> getStringMap() {
                return stringMap;
            }
            
            @Override
            public Object addString(String string) {
                if (null == string) {
                    return null;
                }
                
                String key;
                if (string.length() < 12) {
                    key = string;
                } else {
                    HashCode hash = hashFunction.newHasher().putString(string, charSet).hash();
                    byte[] asBytes = hash.asBytes();
                    key = Base64.encode(asBytes, 0, 8);
                }
                
                stringMap.put(key, string);

                return key;
            }
        };
    }

    // This comes from Apache commons-lang. I didn't want to pull in the full library for these two methods. This is
    // licensed under the Apache License v2 which is safe to include here.
    public static boolean containsIgnoreCase(final CharSequence str, final CharSequence searchStr) {
        if (str == null || searchStr == null) {
            return false;
        }
        final int len = searchStr.length();
        final int max = str.length() - len;
        for (int i = 0; i <= max; i++) {
            if (regionMatches(str, true, i, searchStr, 0, len)) {
                return true;
            }
        }
        return false;
    }

    // This comes from Apache commons-lang. I didn't want to pull in the full library for these two methods. This is
    // licensed under the Apache License v2 which is safe to include here.
    private static boolean regionMatches(final CharSequence cs, final boolean ignoreCase, final int thisStart,
            final CharSequence substring, final int start, final int length)    {
        if (cs instanceof String && substring instanceof String) {
            return ((String) cs).regionMatches(ignoreCase, thisStart, (String) substring, start, length);
        }
        int index1 = thisStart;
        int index2 = start;
        int tmpLen = length;

        // Extract these first so we detect NPEs the same as the java.lang.String version
        final int srcLen = cs.length() - thisStart;
        final int otherLen = substring.length() - start;

        // Check for invalid parameters
        if (thisStart < 0 || start < 0 || length < 0) {
            return false;
        }

        // Check that the regions are long enough
        if (srcLen < length || otherLen < length) {
            return false;
        }

        while (tmpLen-- > 0) {
            final char c1 = cs.charAt(index1++);
            final char c2 = substring.charAt(index2++);

            if (c1 == c2) {
                continue;
            }

            if (!ignoreCase) {
                return false;
            }

            // The same check as in String.regionMatches():
            if (Character.toUpperCase(c1) != Character.toUpperCase(c2)
                    && Character.toLowerCase(c1) != Character.toLowerCase(c2)) {
                return false;
            }
        }

        return true;
    }
}
