package com.newrelic.agent.transport;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import com.newrelic.agent.Agent;
import com.newrelic.agent.ForceDisconnectException;
import com.newrelic.agent.ForceRestartException;
import com.newrelic.agent.LicenseException;
import com.newrelic.agent.MaxPayloadException;
import com.newrelic.agent.MetricData;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.config.LaspPolicies;
import com.newrelic.agent.errors.TracedError;
import com.newrelic.agent.profile.ProfileData;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.service.analytics.AnalyticsEvent;
import com.newrelic.agent.service.analytics.CustomInsightsEvent;
import com.newrelic.agent.service.analytics.ErrorEvent;
import com.newrelic.agent.service.analytics.SpanEvent;
import com.newrelic.agent.service.module.Jar;
import com.newrelic.agent.service.module.Module;
import com.newrelic.agent.sql.SqlTrace;
import com.newrelic.agent.stats.StatsWorks;
import com.newrelic.agent.trace.TransactionTrace;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.StatusLine;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContextBuilder;
import org.apache.http.conn.ssl.StrictHostnameVerifier;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicHeader;
import org.apache.http.protocol.HttpContext;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.JSONStreamAware;
import org.json.simple.JSONValue;
import org.json.simple.parser.JSONParser;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLHandshakeException;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.SocketException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.rmi.UnexpectedException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * A class for sending and receiving New Relic data.
 *
 * This class is thread-safe.
 */
public class DataSenderImpl implements DataSender {

    private static final String MODULE_TYPE = "Jars";
    private static final int DEFAULT_REQUEST_TIMEOUT_IN_SECONDS = 120;
    private static final int PROTOCOL_VERSION = 17;
    private static final String BEFORE_LICENSE_KEY_URI_PATTERN = "/agent_listener/invoke_raw_method?method={0}";
    private static final String AFTER_LICENSE_KEY_URI_PATTERN = "&marshal_format=json&protocol_version=";
    private static final String LICENSE_KEY_URI_PATTERN = "&license_key={0}";
    private static final String RUN_ID_PATTERN = "&run_id={1}";
    private static final String CONNECT_METHOD = "connect";
    private static final String METRIC_DATA_METHOD = "metric_data";
    private static final String GET_AGENT_COMMANDS_METHOD = "get_agent_commands";
    private static final String AGENT_COMMAND_RESULTS_METHOD = "agent_command_results";
    private static final String GET_PRECONNECT_HOST_METHOD = "preconnect";
    private static final String ERROR_DATA_METHOD = "error_data";
    private static final String PROFILE_DATA_METHOD = "profile_data";
    public static final String ERROR_EVENT_DATA_METHOD = "error_event_data";
    public static final String ANALYTIC_DATA_METHOD = "analytic_event_data";
    public final static String SPAN_EVENT_DATA_METHOD = "span_event_data";
    public static final String CUSTOM_ANALYTIC_DATA_METHOD = "custom_event_data";
    private static final String UPDATE_LOADED_MODULES_METHOD = "update_loaded_modules";
    private static final String MODULE_METADATA_METHOD = "module_metadata";
    private static final String SHUTDOWN_METHOD = "shutdown";
    private static final String SQL_TRACE_DATA_METHOD = "sql_trace_data";
    private static final String TRANSACTION_SAMPLE_DATA_METHOD = "transaction_sample_data";
    private static final String USER_AGENT_HEADER_VALUE = initUserHeaderValue();
    private static final String GZIP = "gzip";
    public static final String DEFLATE_ENCODING = "deflate";
    public static final String GZIP_ENCODING = "gzip";
    private static final String IDENTITY_ENCODING = "identity";
    private static final String EXCEPTION_MAP_RETURN_VALUE_KEY = "return_value";
    private static final String AGENT_RUN_ID_KEY = "agent_run_id";
    private static final String SSL_KEY = "ssl";
    private static final Object NO_AGENT_RUN_ID = null;
    private static final String NULL_RESPONSE = "null";
    private static final String TIMEOUT_PROPERTY = "timeout";
    private static final int COMPRESSION_LEVEL = Deflater.DEFAULT_COMPRESSION;
    private static final String GET_XRAY_PARMS_METHOD = "get_xray_metadata";
    private static final String REDIRECT_HOST = "redirect_host";
    private static final String SECURITY_POLICIES = "security_policies";
    private static final String MAX_PAYLOAD_SIZE_IN_BYTES = "max_payload_size_in_bytes";
    private static final String METADATA = "request_headers_map";
    private static final int DEFAULT_MAX_PAYLOAD_SIZE_IN_BYTES = 1000000;

    // As of P17 these are the only agent endpoints that actually contain data in the response payload for a successful request
    private static final Set<String> METHODS_WITH_RESPONSE_BODY = ImmutableSet.of("connect", "get_agent_commands", "get_xray_metadata", "preconnect",
            "profile_data");

    private volatile String host;
    private final int port;
    private volatile String protocol;
    private final HttpHost proxy;
    private final Credentials proxyCredentials;
    private final int defaultTimeoutInMillis;
    private volatile boolean auditMode;
    private Set<String> auditModeEndpoints;
    private volatile Object agentRunId = NO_AGENT_RUN_ID;
    private final String agentRunIdUriPattern;
    private final String noAgentRunIdUriPattern;
    private final boolean usePrivateSSL;
    private final String caBundlePath;
    private final boolean useSSL;
    private final SSLContext sslContext;
    private final DataSenderListener dataSenderListener;
    private final String compressedEncoding;
    private final boolean putForDataSend;
    private Map<String, Boolean> policiesJson;
    private volatile int maxPayloadSizeInBytes = DEFAULT_MAX_PAYLOAD_SIZE_IN_BYTES;
    private volatile Map<String, String> metadata;

    private static String initUserHeaderValue() {
        String arch = "unknown";
        String javaVersion = "unknown";
        try {
            arch = System.getProperty("os.arch");
            javaVersion = System.getProperty("java.version");
        } catch (Exception e) {
        }
        return MessageFormat.format("NewRelic-JavaAgent/{0} (java {1} {2})", Agent.getVersion(), javaVersion, arch);
    }

    public DataSenderImpl(AgentConfig config) {
        this(config, null);
    }

    public DataSenderImpl(AgentConfig config, DataSenderListener dataSenderListener) {
        auditMode = config.isAuditMode();
        auditModeEndpoints = config.getAuditModeConfig().getEndpoints();
        Agent.LOG.info(MessageFormat.format("Setting audit_mode to {0}", auditMode));
        host = config.getHost();
        port = config.getPort();
        useSSL = config.isSSL();
        protocol = useSSL ? "https" : "http";
        Agent.LOG.info(MessageFormat.format("Setting protocol to \"{0}\"", protocol));
        String proxyHost = config.getProxyHost();
        Integer proxyPort = config.getProxyPort();
        String proxyScheme = config.getProxyScheme();
        usePrivateSSL = config.isUsePrivateSSL();
        caBundlePath = config.getCaBundlePath();
        sslContext = createSSLContext();
        if (proxyHost != null && proxyPort != null) {
            Agent.LOG.fine(MessageFormat.format("Using proxy host {0}:{1}", proxyHost, Integer.toString(proxyPort)));
            proxy = new HttpHost(proxyHost, proxyPort, proxyScheme);
            proxyCredentials = getProxyCredentials(config.getProxyUser(), config.getProxyPassword());
        } else {
            proxy = null;
            proxyCredentials = null;
        }
        defaultTimeoutInMillis = config.getProperty(TIMEOUT_PROPERTY, DEFAULT_REQUEST_TIMEOUT_IN_SECONDS) * 1000;
        String licenseKeyUri = MessageFormat.format(LICENSE_KEY_URI_PATTERN, config.getLicenseKey());
        noAgentRunIdUriPattern = BEFORE_LICENSE_KEY_URI_PATTERN + licenseKeyUri + AFTER_LICENSE_KEY_URI_PATTERN + PROTOCOL_VERSION;
        agentRunIdUriPattern = noAgentRunIdUriPattern + RUN_ID_PATTERN;
        this.dataSenderListener = dataSenderListener;
        this.compressedEncoding = config.getCompressedContentEncoding();
        this.putForDataSend = config.isPutForDataSend();
    }

    public static KeyStore getKeyStore(String caBundlePath) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException {
        KeyStore keystore = KeyStore.getInstance(KeyStore.getDefaultType());

        InputStream in = DataSenderImpl.class.getResourceAsStream("/nrcerts");
        if (null == in) {
            Agent.LOG.fine("Unable to find NR trust store");
        } else {
            Collection<X509Certificate> caCerts = new LinkedList<>();
            if (caBundlePath != null) {
                Agent.LOG.log(Level.FINEST, "Checking ca_bundle_path at: {0}", caBundlePath);

                InputStream is = new FileInputStream(caBundlePath);
                try {
                    CertificateFactory cf = CertificateFactory.getInstance("X.509");
                    while (is.available() > 0) {
                        try {
                            caCerts.add((X509Certificate) cf.generateCertificate(is));
                        } catch (Throwable t) {
                            Agent.LOG.log(Level.FINEST, "Unable to generate ca_bundle_path certificate", t);
                        }
                    }
                } finally {
                    is.close();
                }

                Agent.LOG.log(Level.FINEST, "Found: {0} certificates", caCerts.size());
            }

            try {
                keystore.load(in, null);

                int i = 1;
                for (X509Certificate caCert : caCerts) {
                    if (caCert != null) {
                        String alias = "ca_bundle_path_" + i;
                        keystore.setCertificateEntry(alias, caCert);

                        Agent.LOG.log(Level.FINEST, "Installed certificate {0} at alias: {1}", i, alias);
                        if (Agent.isDebugEnabled()) {
                            Agent.LOG.log(Level.FINEST, "Installed certificate {0} at alias: {1}", caCert, alias);
                        }
                    }
                    i++;
                }
            } finally {
                in.close();
            }
        }

        Agent.LOG.finer("SSL Keystore Provider: " + keystore.getProvider().getName());

        return keystore;
    }

    private SSLContext createSSLContext() {
        SSLContextBuilder sslContextBuilder = new SSLContextBuilder();
        try {
            if (usePrivateSSL && useSSL) {
                sslContextBuilder.loadTrustMaterial(getKeyStore(caBundlePath));
            }

            return sslContextBuilder.build();
        } catch (Exception e) {
            Agent.LOG.log(Level.WARNING, e, "Unable to create SSL context");
            return null;
        }
    }

    private Credentials getProxyCredentials(final String proxyUser, final String proxyPass) {
        if (proxyUser != null && proxyPass != null) {
            Agent.LOG.info(MessageFormat.format("Setting Proxy Authenticator for user {0}", proxyUser));
            return new UsernamePasswordCredentials(proxyUser, proxyPass);
        }
        return null;
    }

    private void checkAuditMode() {
        boolean auditMode2 = ServiceFactory.getConfigService().getLocalAgentConfig().isAuditMode();
        if (auditMode != auditMode2) {
            auditMode = auditMode2;
            Agent.LOG.info(MessageFormat.format("Setting audit_mode to {0}", auditMode));
        }

        Set<String> auditModeEndpoints2 = ServiceFactory.getConfigService()
                .getLocalAgentConfig()
                .getAuditModeConfig()
                .getEndpoints();
        if (auditModeEndpoints != auditModeEndpoints2) {
            auditModeEndpoints = auditModeEndpoints2;
            Agent.LOG.info(MessageFormat.format("Setting audit_mode.endpoints to {0}", auditModeEndpoints));
        }
    }

    @VisibleForTesting
    void setAgentRunId(Object runId) {
        agentRunId = runId;
        if (runId != NO_AGENT_RUN_ID) {
            Agent.LOG.info("Agent run id: " + runId);
        }
    }

    @Override
    public Map<String, Object> connect(Map<String, Object> startupOptions) throws Exception {
        String redirectHost = parsePreconnectAndReturnHost();
        if (redirectHost != null) {
            host = redirectHost;
            Agent.LOG.info(MessageFormat.format("Collector redirection to {0}:{1}", host, Integer.toString(port)));
        } else if (ServiceFactory.getConfigService().getDefaultAgentConfig().laspEnabled()) {
            throw new ForceDisconnectException("The agent did not receive one or more security policies that it expected and will shut down."
                    + " Please contact support.");
        }
        return doConnect(startupOptions);
    }

    private String parsePreconnectAndReturnHost() throws Exception {
        AgentConfig agentConfig = ServiceFactory.getConfigService().getDefaultAgentConfig();

        InitialSizedJsonArray params = new InitialSizedJsonArray(1);
        JSONObject token = new JSONObject();

        if (agentConfig.laspEnabled()) {
            token.put("security_policies_token", agentConfig.securityPoliciesToken());
        }
        params.add(token);
        Object response = invokeNoRunId(GET_PRECONNECT_HOST_METHOD, compressedEncoding, params);

        if (response != null) {
            Map<?, ?> returnValue = (Map<?, ?>) response;
            String host = returnValue.get(REDIRECT_HOST).toString();

            JSONObject policies = (JSONObject) returnValue.get(SECURITY_POLICIES);
            this.policiesJson = LaspPolicies.validatePolicies(policies);

            return host;
        }

        return null;
    }

    @SuppressWarnings("unchecked")
    private Map<String, Object> doConnect(Map<String, Object> startupOptions) throws Exception {
        InitialSizedJsonArray params = new InitialSizedJsonArray(1);
        if (policiesJson != null && !policiesJson.isEmpty()) {
            startupOptions.put("security_policies", LaspPolicies.convertToConnectPayload(policiesJson));
        }
        params.add(startupOptions);
        Object response = invokeNoRunId(CONNECT_METHOD, compressedEncoding, params);
        if (!(response instanceof Map)) {
            throw new UnexpectedException(MessageFormat.format("Expected a map of connection data, got {0}", response));
        }
        Map<String, Object> data = (Map<String, Object>) response;
        if (data.containsKey(MAX_PAYLOAD_SIZE_IN_BYTES)) {
            Object maxPayloadSize = data.get(MAX_PAYLOAD_SIZE_IN_BYTES);
            if (maxPayloadSize instanceof Number) {
                maxPayloadSizeInBytes = Integer.valueOf(((Number) maxPayloadSize).intValue());
                Agent.LOG.log(Level.INFO, "Max payload size is {0} bytes", maxPayloadSizeInBytes);
            }
        }

        if (data.containsKey(METADATA)) {
            final Object metadata = data.get(METADATA);
            if (metadata instanceof Map) {
                this.metadata = (Map<String, String>) metadata;
            } else {
                Agent.LOG.log(Level.WARNING, "Expected a map but got {0}. Not setting metadata", metadata);
            }
        } else {
            Agent.LOG.log(Level.WARNING, "Did not receive metadata on connect");
        }

        if (data.containsKey(AGENT_RUN_ID_KEY)) {
            Object runId = data.get(AGENT_RUN_ID_KEY);
            setAgentRunId(runId);
        } else {
            throw new UnexpectedException(MessageFormat.format("Missing {0} connection parameter", AGENT_RUN_ID_KEY));
        }
        Object ssl = data.get(SSL_KEY);
        if (Boolean.TRUE.equals(ssl)) {
            Agent.LOG.info("Setting protocol to \"https\"");
            protocol = "https";
        }
        ServiceFactory.getConfigService().setLaspPolicies(policiesJson);

        return data;
    }

    @Override
    @SuppressWarnings("unchecked")
    public List<List<?>> getAgentCommands() throws Exception {
        checkAuditMode();
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID) {
            return Collections.emptyList();
        }
        InitialSizedJsonArray params = new InitialSizedJsonArray(1);
        params.add(runId);

        Object response = invokeRunId(GET_AGENT_COMMANDS_METHOD, compressedEncoding, runId, params);
        if (response == null || NULL_RESPONSE.equals(response)) {
            return Collections.emptyList();
        }
        try {
            return (List<List<?>>) response;
        } catch (ClassCastException e) {
            Agent.LOG.warning(MessageFormat.format("Invalid response from New Relic when getting agent commands: {0}", e));
            throw e;
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<?> getXRayParameters(Collection<Long> ids) throws Exception {
        if (ids.size() > 0) {
            checkAuditMode();
            Object runId = agentRunId;
            if (runId == NO_AGENT_RUN_ID) {
                return Collections.emptyList();
            }
            JSONArray params = new JSONArray();
            params.add(runId);

            for (Long s : ids) {
                params.add(s);
            }

            Object response = invokeRunId(GET_XRAY_PARMS_METHOD, compressedEncoding, runId, params);
            if (response == null || NULL_RESPONSE.equals(response)) {
                return Collections.emptyList();
            }
            try {
                return (List<List<?>>) response;
            } catch (ClassCastException e) {
                Agent.LOG.warning(MessageFormat.format("Invalid response from New Relic when getting agent X Ray parameters: {0}", e));
                throw e;
            }
        } else {
            Agent.LOG.info("Attempted to fetch X-Ray Session metadata with no session IDs");
        }
        return Collections.emptyList();
    }

    @Override
    public void sendCommandResults(Map<Long, Object> commandResults) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID || commandResults.isEmpty()) {
            return;
        }

        InitialSizedJsonArray params = new InitialSizedJsonArray(2);
        params.add(runId);
        params.add(commandResults);

        invokeRunId(AGENT_COMMAND_RESULTS_METHOD, compressedEncoding, runId, params);
    }

    @Override
    public void sendErrorData(List<TracedError> errors) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID || errors.isEmpty()) {
            return;
        }

        InitialSizedJsonArray params = new InitialSizedJsonArray(2);
        params.add(runId);
        params.add(errors);

        invokeRunId(ERROR_DATA_METHOD, compressedEncoding, runId, params);
    }

    @Override
    public void sendErrorEvents(int reservoirSize, int eventsSeen, Collection<ErrorEvent> errorEvents) throws Exception {
        sendAnalyticEventsForReservoir(ERROR_EVENT_DATA_METHOD, compressedEncoding, reservoirSize, eventsSeen, errorEvents);
    }

    @Override
    public void sendAnalyticsEvents(int reservoirSize, int eventsSeen, Collection<? extends AnalyticsEvent> events) throws Exception {
        sendAnalyticEventsForReservoir(ANALYTIC_DATA_METHOD, compressedEncoding, reservoirSize, eventsSeen, events);
    }

    @Override
    public void sendCustomAnalyticsEvents(int reservoirSize, int eventsSeen, Collection<? extends CustomInsightsEvent> events) throws Exception {
        sendAnalyticEventsForReservoir(CUSTOM_ANALYTIC_DATA_METHOD, compressedEncoding, reservoirSize, eventsSeen, events);
    }

    @Override
    public void sendSpanEvents(int reservoirSize, int eventsSeen, Collection<SpanEvent> events) throws Exception {
        sendAnalyticEventsForReservoir(SPAN_EVENT_DATA_METHOD, compressedEncoding, reservoirSize, eventsSeen, events);
    }

    private void sendAnalyticEventsForReservoir(String method, String encoding, int reservoirSize, int eventsSeen,
            Collection<? extends AnalyticsEvent> events) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID || events.isEmpty()) {
            return;
        }
        InitialSizedJsonArray params = new InitialSizedJsonArray(3);
        params.add(runId);

        JSONObject metadata = new JSONObject();
        metadata.put("reservoir_size", reservoirSize);
        metadata.put("events_seen", eventsSeen);
        params.add(metadata);

        params.add(events);
        invokeRunId(method, encoding, runId, params);
    }

    @Override
    @SuppressWarnings("unchecked")
    public void sendMetricData(long beginTimeMillis, long endTimeMillis, List<MetricData> metricData) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID || metricData.isEmpty()) {
            return;
        }

        InitialSizedJsonArray params = new InitialSizedJsonArray(4);
        params.add(runId);
        params.add(beginTimeMillis / 1000);
        params.add(endTimeMillis / 1000);
        params.add(metricData);

        invokeRunId(METRIC_DATA_METHOD, compressedEncoding, runId, params);
    }

    @Override
    @SuppressWarnings("unchecked")
    public List<Long> sendProfileData(List<ProfileData> profiles) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID || profiles.isEmpty()) {
            return Collections.emptyList();
        }

        InitialSizedJsonArray params = new InitialSizedJsonArray(2);
        params.add(runId);
        params.add(profiles);

        Object response = invokeRunId(PROFILE_DATA_METHOD, getEncodingForComplexCompression(), runId, params);
        if (response == null || NULL_RESPONSE.equals(response)) {
            return Collections.emptyList();
        }
        try {
            return (List<Long>) response;
        } catch (ClassCastException e) {
            Agent.LOG.warning(MessageFormat.format("Invalid response from New Relic sending profiles: {0}", e));
            throw e;
        }
    }

    /**
     * Sends the jars with versions to the collector.
     *
     * @param pJars The new jars which need to be sent to the collector.
     */
    @Override
    public void sendModules(List<Jar> pJars) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID || pJars == null || pJars.isEmpty()) {
            return;
        }
        InitialSizedJsonArray params = new InitialSizedJsonArray(2);

        // Module type must always be first - it should always be jars
        params.add(MODULE_TYPE);
        params.add(pJars);

        invokeRunId(UPDATE_LOADED_MODULES_METHOD, compressedEncoding, runId, params);
    }

    /**
     * Some of our data calls are json documents of base 64 encoded strings with gzipped json docs inside of them.
     * We normally send these requests with IDENTITY encoding because a large portion of the payload is already compressed.
     * When the "simple_compression" flag is on, we directly include the json docs instead of compressing them, and we
     * DEFLATE the entire json document instead.
     *
     * @return the type of encoding to use based on the simple_compression configuration value
     */
    private String getEncodingForComplexCompression() {
        return ServiceFactory.getConfigService().getDefaultAgentConfig().isSimpleCompression() ? compressedEncoding : IDENTITY_ENCODING;
    }

    @Override
    public void sendModuleMetadata(Module module) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID) {
            throw new Exception("Not connected");
        }

        /**
         * For now we are not limiting the size of the module payload. We will assume that the limit is large
         * enough to accommodate the compressed & stripped + base64 encoded jars we are sending up.
         *
         * As an example, the guava-18.0 jar is 1646791 bytes (1.6 MB)
         */
        InitialSizedJsonArray params = new InitialSizedJsonArray(1);
        params.add(module);

        invokeRunId(MODULE_METADATA_METHOD, IDENTITY_ENCODING, runId, params);
    }

    @Override
    public void sendSqlTraceData(List<SqlTrace> sqlTraces) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID || sqlTraces.isEmpty()) {
            return;
        }

        InitialSizedJsonArray params = new InitialSizedJsonArray(1);
        params.add(sqlTraces);

        invokeRunId(SQL_TRACE_DATA_METHOD, getEncodingForComplexCompression(), runId, params);
    }

    @Override
    public void sendTransactionTraceData(List<TransactionTrace> traces) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID || traces.isEmpty()) {
            return;
        }

        InitialSizedJsonArray params = new InitialSizedJsonArray(2);
        params.add(runId);
        params.add(traces);

        invokeRunId(TRANSACTION_SAMPLE_DATA_METHOD, getEncodingForComplexCompression(), runId, params);
    }

    // The fix for JAVA-2965 assumes RPMService.shutdown() is the only caller of this method.
    // There's no way to avoid this bogus assumption short of a major rewrite of this layer.
    @Override
    public void shutdown(long timeMillis) throws Exception {
        Object runId = agentRunId;
        if (runId == NO_AGENT_RUN_ID) {
            return;
        }

        InitialSizedJsonArray params = new InitialSizedJsonArray(2);
        params.add(runId);
        params.add(timeMillis);
        int requestTimeoutInMillis = 10000;
        try {
            invokeRunId(SHUTDOWN_METHOD, compressedEncoding, runId, requestTimeoutInMillis, params);
        } finally {
            setAgentRunId(NO_AGENT_RUN_ID);
        }
    }

    @VisibleForTesting
    void setMaxPayloadSizeInBytes(int payloadSizeInBytes) {
        maxPayloadSizeInBytes = payloadSizeInBytes;
    }

    private Object invokeRunId(String method, String encoding, Object runId, JSONStreamAware params) throws Exception {
        return invokeRunId(method, encoding, runId, defaultTimeoutInMillis, params);
    }

    private Object invokeRunId(String method, String encoding, Object runId, int timeoutInMillis, JSONStreamAware params) throws Exception {
        String uri = MessageFormat.format(agentRunIdUriPattern, method, runId.toString());
        return invoke(method, encoding, uri, params, timeoutInMillis);
    }

    private Object invokeNoRunId(String method, String encoding, JSONStreamAware params) throws Exception {
        String uri = MessageFormat.format(noAgentRunIdUriPattern, method);
        return invoke(method, encoding, uri, params, defaultTimeoutInMillis);
    }

    private Object invoke(String method, String encoding, String uri, JSONStreamAware params, int timeoutInMillis) throws Exception {
        // ReadResult should be from a valid 2xx response at this point otherwise send method throws an exception here
        ReadResult readResult = send(method, encoding, uri, params, timeoutInMillis);
        Map<?, ?> responseMap = null;
        String responseBody = readResult.getResponseBody();

        if (responseBody != null && !responseBody.isEmpty()) {
            try {
                responseMap = getResponseMap(responseBody);
            } catch (Exception e) {
                Agent.LOG.log(Level.WARNING, "Error parsing response JSON({0}) from NewRelic: {1}", method, e.toString());
                Agent.LOG.log(Level.FINEST, "Invalid response JSON({0}): {1}", method, responseBody);
                throw e;
            }
        } else if (METHODS_WITH_RESPONSE_BODY.contains(method)) {
            // Only log this if it's a method that we would expect to have data in the response payload
            Agent.LOG.log(Level.FINER, "Response was null ({0})", method);
        }

        if (responseMap != null) {
            if (dataSenderListener != null) {
                dataSenderListener.dataReceived(method, encoding, uri, responseMap);
            }

            try {
                return responseMap.get(EXCEPTION_MAP_RETURN_VALUE_KEY);
            } catch (ClassCastException ex) {
                Agent.LOG.log(Level.WARNING, "Error parsing response JSON({0}) from NewRelic: {1}", method, ex.toString());
                return null;
            }
        } else {
            return null;
        }
    }

    /*
     * As of Protocol 17 agents MUST NOT depend on the content of the response body for any behavior; just the integer
     * response code value. The previous behavior of a 200 (“OK”) with an exact string in the body that should be
     * matched/parsed has been deprecated.
     */
    private ReadResult connectAndSend(String method, String encoding, String uri, JSONStreamAware params, int timeoutInMillis) throws Exception {
        CloseableHttpClient conn = null;
        try {
            int statusCode;
            conn = createHttpClient(encoding, uri, timeoutInMillis);
            byte[] data = writeData(encoding, params);

            /* We don’t enforce max_payload_size_in_bytes for error_data (aka error traces). Instead we halve the
             * payload and try again. See RPMService sendErrorData
             */
            if (data.length > maxPayloadSizeInBytes && !method.equals("error_data")) {
                ServiceFactory.getStatsService().doStatsWork(StatsWorks.getIncrementCounterWork(
                        MessageFormat.format(MetricNames.SUPPORTABILITY_PAYLOAD_SIZE_EXCEEDS_MAX, method), 1));
                String msg = MessageFormat.format("Payload of size {0} exceeded maximum size {1} for {2} method ",
                        data.length, maxPayloadSizeInBytes, method);
                throw new MaxPayloadException(msg);
            }

            HttpUriRequest request = createRequest(method, uri, timeoutInMillis, data);
            HttpContext context = createHttpContext();

            long requestSent = System.currentTimeMillis();
            // here's where we get the response back from agent endpoints
            try (CloseableHttpResponse response = conn.execute(request, context)) {
                long requestDuration = System.currentTimeMillis() - requestSent;

                ServiceFactory.getStatsService().doStatsWork(StatsWorks.getRecordResponseTimeWork(
                        MessageFormat.format(MetricNames.SUPPORTABILITY_AGENT_ENDPOINT_DURATION, method), requestDuration));

                // check status line of the response, can be null
                StatusLine statusLine = response.getStatusLine();
                if (statusLine == null) {
                    throw new Exception("The http response has no status line");
                }

                if (auditMode && methodWhitelisted(method)) {
                    String msg = MessageFormat.format("Sent JSON({0}) to: {1}, with payload: {2}", method, request.getURI(),
                            DataSenderWriter.toJSONString(params));
                    Agent.LOG.info(msg);
                }

                statusCode = statusLine.getStatusCode();

                // Create supportability metric for all response codes
                ServiceFactory.getStatsService().doStatsWork(StatsWorks.getIncrementCounterWork(
                        MessageFormat.format(MetricNames.SUPPORTABILITY_HTTP_CODE, statusCode), 1));

                // Comply with spec and send supportability metric only for error responses
                if (statusCode != HttpResponseCode.OK && statusCode != HttpResponseCode.ACCEPTED) {
                    ServiceFactory.getStatsService().doStatsWork(StatsWorks.getIncrementCounterWork(
                            MessageFormat.format(MetricNames.SUPPORTABILITY_AGENT_ENDPOINT_HTTP_ERROR, statusCode), 1));
                    ServiceFactory.getStatsService().doStatsWork(StatsWorks.getIncrementCounterWork(
                            MessageFormat.format(MetricNames.SUPPORTABILITY_AGENT_ENDPOINT_ATTEMPTS, method), 1));
                }

                String responseMessage = statusLine.getReasonPhrase();
                // HttpError exceptions are typically handled in RPMService or the harvestable service for the requested endpoint
                if (statusCode == HttpResponseCode.PROXY_AUTHENTICATION_REQUIRED) {
                    // agent receives a 407 response due to a misconfigured proxy (not from NR backend), throw exception
                    final Header proxyAuthHeader = response.getFirstHeader("Proxy-Authenticate");
                    if (proxyAuthHeader != null) {
                        String authField = proxyAuthHeader.getValue();
                        throw new HttpError("Proxy Authentication Mechanism Failed: " + authField, statusCode, data.length);
                    } else {
                        throw new HttpError("Proxy Authentication Mechanism Failed: " + "null Proxy-Authenticate header", statusCode, data.length);
                    }
                } else if (statusCode == HttpResponseCode.UNAUTHORIZED) {
                    // received 401 Unauthorized, throw exception instead of parsing LicenseException from 200 response body
                    throw new LicenseException(responseMessage);
                } else if (statusCode == HttpResponseCode.CONFLICT) {
                    // received 409 Conflict, throw exception instead of parsing ForceRestartException from 200 response body
                    throw new ForceRestartException(responseMessage);
                } else if (statusCode == HttpResponseCode.GONE) {
                    // received 410 Gone, throw exception instead of parsing ForceDisconnectException from 200 response body
                    throw new ForceDisconnectException(responseMessage);
                } else if (statusCode != HttpResponseCode.OK && statusCode != HttpResponseCode.ACCEPTED) {
                    // response is bad (neither 200 nor 202), throw generic HttpError exception
                    Agent.LOG.log(Level.FINER, "Connection http status code: {0}", statusCode);
                    throw HttpError.create(statusCode, host, data.length);
                }

                // received successful 2xx response
                String responseBody = readResponseBody(response);
                if (auditMode && methodWhitelisted(method)) {
                    Agent.LOG.info(MessageFormat.format("Received JSON({0}): {1}", method, responseBody));
                }
                if (dataSenderListener != null) {
                    dataSenderListener.dataSent(method, encoding, uri, data);
                }
                return ReadResult.create(statusCode, responseBody);
            }
        } finally {
            if (conn != null) {
                conn.close();
            }
        }
    }

    private boolean methodWhitelisted(String method) {
        if (auditModeEndpoints != null && auditModeEndpoints.size() > 0) {
            return auditModeEndpoints.contains(method);
        }
        return true;
    }

    private HttpContext createHttpContext() {
        HttpClientContext context = new HttpClientContext();

        if (proxy != null && proxyCredentials != null) {
            CredentialsProvider credentialsProvider = new BasicCredentialsProvider();
            credentialsProvider.setCredentials(new AuthScope(proxy), proxyCredentials);
            context.setCredentialsProvider(credentialsProvider);
        }

        return context;
    }

    private ReadResult send(String method, String encoding, String uri, JSONStreamAware params, int timeoutInMillis) throws Exception {
        try {
            return connectAndSend(method, encoding, uri, params, timeoutInMillis);
        } catch (MalformedURLException e) {
            Agent.LOG.log(Level.SEVERE, "You have requested a connection to New Relic via a protocol which is unavailable in your runtime: {0}", e.toString());
            throw new ForceDisconnectException(e.toString());
        } catch (SocketException e) {
            if (e.getCause() instanceof java.security.NoSuchAlgorithmException) {
                String msg = MessageFormat.format("You have requested a connection to New Relic via an algorithm which is unavailable in your runtime: {0}."
                        + " This may also be indicative of a corrupted keystore or trust store on your server.", e.getCause().toString());
                Agent.LOG.error(msg);
                // this is a recoverable error. Try again later
            } else {
                Agent.LOG.log(Level.INFO, "A socket exception was encountered while sending data to New Relic ({0})."
                        + " Please check your network / proxy settings.", e.toString());
                if (Agent.LOG.isLoggable(Level.FINE)) {
                    Agent.LOG.log(Level.FINE, "Error sending JSON({0}): {1}", method, DataSenderWriter.toJSONString(params));
                }
                Agent.LOG.log(Level.FINEST, e, e.toString());
            }
            throw e;
        } catch (HttpError e) {
            // These errors are logged upstream of this call.
            throw e;
        } catch (Exception e) {
            if (e instanceof SSLHandshakeException) {
                Agent.LOG.log(Level.INFO, "Unable to connect to New Relic due to an SSL error."
                        + " Consider enabling -Djavax.net.debug=all to debug your SSL configuration such as your trust store.", e);
            }
            Agent.LOG.log(Level.INFO, "Remote {0} call failed : {1}.", method, e.toString());
            if (Agent.LOG.isLoggable(Level.FINE)) {
                Agent.LOG.log(Level.FINE, "Error sending JSON({0}): {1}", method, DataSenderWriter.toJSONString(params));
            }
            Agent.LOG.log(Level.FINEST, e, e.toString());
            throw e;
        }
    }

    private HttpUriRequest createRequest(String method, String uri, int requestTimeoutInMillis, byte[] data) throws MalformedURLException, URISyntaxException {
        URL url = new URL(protocol, host, port, uri);
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(requestTimeoutInMillis)
                .setConnectionRequestTimeout(requestTimeoutInMillis)
                .setSocketTimeout(requestTimeoutInMillis).build();

        RequestBuilder requestBuilder;
        if (putForDataSend) {
            requestBuilder = RequestBuilder.put();
        } else {
            requestBuilder = RequestBuilder.post();
        }
        requestBuilder
                .setUri(url.toURI())
                .setEntity(new ByteArrayEntity(data))
                .setConfig(config);

        final Map<String, String> metadata = this.metadata;
        boolean isConnectOrPreconnect = method.equals(CONNECT_METHOD) || method.equals(GET_PRECONNECT_HOST_METHOD);
        if (metadata != null && !isConnectOrPreconnect) {
            for (Map.Entry<String, String> entry : metadata.entrySet()) {
                requestBuilder.addHeader(entry.getKey(), entry.getValue());
            }
        }

        return requestBuilder.build();
    }

    private CloseableHttpClient createHttpClient(String encoding, String uri, int requestTimeoutInMillis) throws Exception {
        HttpClientBuilder builder = HttpClientBuilder.create();
        builder.setUserAgent(USER_AGENT_HEADER_VALUE).setDefaultHeaders(Arrays.<Header>asList(new BasicHeader(
                        "Connection", "Keep-Alive"), new BasicHeader("CONTENT-TYPE", "application/json"),
                new BasicHeader("ACCEPT-ENCODING", GZIP), new BasicHeader("CONTENT-ENCODING", encoding)));
        builder.setDefaultSocketConfig(SocketConfig.custom()
                .setSoTimeout(requestTimeoutInMillis)
                .setSoKeepAlive(true)
                .build());

        RequestConfig.Builder requestBuilder = RequestConfig.custom()
                .setConnectTimeout(requestTimeoutInMillis)
                .setConnectionRequestTimeout(requestTimeoutInMillis)
                .setSocketTimeout(requestTimeoutInMillis);

        builder.setDefaultRequestConfig(requestBuilder.build());
        builder.setHostnameVerifier(new StrictHostnameVerifier());

        if (proxy != null) {
            builder.setProxy(proxy);
        }

        if (sslContext != null) {
            builder.setSSLSocketFactory(new SSLConnectionSocketFactory(this.sslContext));
        }

        return builder.build();
    }

    private byte[] writeData(String encoding, JSONStreamAware params) throws IOException {
        ByteArrayOutputStream outStream = new ByteArrayOutputStream();
        Writer out = null;
        try {
            OutputStream os = getOutputStream(outStream, encoding);
            out = new OutputStreamWriter(os, StandardCharsets.UTF_8);
            JSONValue.writeJSONString(params, out);
            out.flush();
        } finally {
            if (out != null) {
                out.close();
            }
        }
        return outStream.toByteArray();
    }

    private OutputStream getOutputStream(OutputStream out, String encoding) throws IOException {
        if (DEFLATE_ENCODING.equals(encoding)) {
            return new DeflaterOutputStream(out, new Deflater(COMPRESSION_LEVEL));
        } else if (GZIP_ENCODING.equals(encoding)) {
            return new GZIPOutputStream(out);
        } else {
            return out;
        }
    }

    private String readResponseBody(CloseableHttpResponse response) throws Exception {
        HttpEntity entity = response.getEntity();
        if (entity == null) {
            throw new Exception("The http response entity was null");
        }
        InputStream is = entity.getContent();
        BufferedReader in = getBufferedReader(response, is);
        try {
            StringBuilder responseBody = new StringBuilder();
            String line;
            while ((line = in.readLine()) != null) {
                responseBody.append(line);
            }
            return responseBody.toString();
        } finally {
            in.close();
            is.close();
        }
    }

    private Map<?, ?> getResponseMap(String responseBody) throws Exception {
        JSONParser parser = new JSONParser();
        Object response = parser.parse(responseBody);
        return Map.class.cast(response);
    }

    private BufferedReader getBufferedReader(CloseableHttpResponse response, InputStream is) throws IOException {
        Header encodingHeader = response.getFirstHeader("content-encoding");
        if (encodingHeader != null) {
            String encoding = encodingHeader.getValue();
            if (GZIP.equals(encoding)) {
                is = new GZIPInputStream(is);
            }
        }
        return new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8));
    }
}
