package com.newrelic.agent.tracers;

import com.newrelic.agent.Agent;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.Transaction;
import com.newrelic.agent.TransactionActivity;
import com.newrelic.agent.bridge.TracedMethod;
import com.newrelic.agent.bridge.TransactionNamePriority;
import com.newrelic.agent.instrumentation.AgentWrapper;
import com.newrelic.agent.instrumentation.pointcuts.play.PlayTransactionStateImpl;
import com.newrelic.agent.util.Strings;
import com.newrelic.api.agent.ExternalParameters;
import com.newrelic.api.agent.InboundHeaders;
import com.newrelic.api.agent.OutboundHeaders;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.text.MessageFormat;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

/**
 * Base class for all tracers. This implements {@link InvocationHandler#invoke(Object, Method, Object[])}
 *
 * @author sdaubin
 */
public abstract class AbstractTracer implements Tracer {

    private TransactionActivity transactionActivity;
    private Set<String> rollupMetricNames;
    private Set<String> exclusiveRollupMetricNames;
    private String customPrefix = "Custom";
    // doesn't need to be thread safe since this flag affects the decision to registerAsync
    private Boolean trackChildThreads = null;

    private final long startTimeInMillis;
    AtomicReference<Long> finishTime = new AtomicReference<Long>(null);

    // Tracers MUST NOT store references to the Transaction. Why: tracers are stored in the TransactionActivity,
    // and Activities can be reparented from one Transaction to another by the public APIs that support async.

    /**
     * Create a tracer on the current thread.
     *
     * @param transaction the transaction that owns the activity on the current thread. Must not be null.
     */
    public AbstractTracer(Transaction transaction) {
        this(transaction.getTransactionActivity());
    }

    /**
     * Create a tracer on the current thread.
     *
     * @param txa the activity for the current thread. For the benefit of legacy Play framework instrumentation, the
     * value is allowed to be null. See {@link MethodExitTracerNoSkip} and {@link PlayTransactionStateImpl}.
     */
    public AbstractTracer(TransactionActivity txa) {
        this.transactionActivity = txa;
        this.startTimeInMillis = System.currentTimeMillis();
    }

    /**
     * Get the transaction that currently owns the activity that owns this tracer.
     *
     * @return the transaction that currently owns the activity that owns this tracer.
     */
    public final Transaction getTransaction() {
        return transactionActivity.getTransaction();
    }

    /**
     * Get the transaction activity that owns this tracer.
     *
     * @return the transaction activity that owns this tracer. This value does not change during the life of the tracer.
     */
    @Override
    public final TransactionActivity getTransactionActivity() {
        return transactionActivity;
    }

    @Override
    public void setTransactionActivity(TransactionActivity txa) {
        transactionActivity = txa;
    }

    protected Object getInvocationTarget() {
        return null;
    }

    @Override
    public final Object invoke(Object methodName, Method method, Object[] args) {
        try {
            if (args == null) {
                Agent.LOG.severe("Tracer.finish() was invoked with no arguments");
            } else if (AgentWrapper.SUCCESSFUL_METHOD_INVOCATION == methodName) {
                if (args.length == 2) {
                    finish((Integer) args[0], args[1]);
                } else {
                    Agent.LOG.severe(MessageFormat.format(
                            "Tracer.finish(int, Object) was invoked with {0} arguments(s)", args.length));
                }
            } else if (AgentWrapper.UNSUCCESSFUL_METHOD_INVOCATION == methodName) {
                if (args.length == 1) {
                    finish((Throwable) args[0]);
                } else {
                    Agent.LOG.severe(MessageFormat.format("Tracer.finish(Throwable) was invoked with {0} arguments(s)",
                            args.length));
                }
            } else {
                Agent.LOG.severe(MessageFormat.format("Tracer.finish was invoked with an unknown method: {0}",
                        methodName));
            }
        } catch (RetryException e) {
            return invoke(methodName, method, args);
        } catch (Throwable t) {
            if (Agent.LOG.isLoggable(Level.FINE)) {
                String msg = MessageFormat.format(
                        "An error occurred finishing method tracer {0} for signature {1} : {2}", getClass().getName(),
                        getClassMethodSignature(), t.toString());
                if (Agent.LOG.isLoggable(Level.FINEST)) {
                    Agent.LOG.log(Level.FINEST, msg, t);
                } else {
                    Agent.LOG.fine(msg);
                }
            }
        }
        return null;
    }

    @Override
    public abstract ClassMethodSignature getClassMethodSignature();

    @Override
    public boolean isChildHasStackTrace() {
        return false;
    }

    @Override
    public void nameTransaction(TransactionNamePriority priority) {
        try {
            ClassMethodSignature classMethodSignature = getClassMethodSignature();
            Object invocationTarget = getInvocationTarget();
            String className = invocationTarget == null ? classMethodSignature.getClassName()
                    : invocationTarget.getClass().getName();
            String txName = className + "/" + classMethodSignature.getMethodName();
            Agent.LOG.log(Level.FINER, "Setting transaction name using instrumented class and method: {0}", txName);
            Transaction tx = transactionActivity.getTransaction();
            tx.setTransactionName(priority, false, customPrefix, txName);
        } catch (Throwable t) {
            Agent.LOG.log(Level.FINEST, "nameTransaction", t);
        }
    }

    @Override
    public TracedMethod getParentTracedMethod() {
        return getParentTracer();
    }

    @Override
    public boolean isLeaf() {
        return false;
    }

    @Override
    public boolean isAsync() {
        return false;
    }

    protected Set<String> getRollupMetricNames() {
        return rollupMetricNames;
    }

    protected Set<String> getExclusiveRollupMetricNames() {
        return exclusiveRollupMetricNames;
    }

    @Override
    public void addRollupMetricName(String... metricNameParts) {
        if (rollupMetricNames == null) {
            rollupMetricNames = new HashSet<String>();
        }
        rollupMetricNames.add(Strings.join(MetricNames.SEGMENT_DELIMITER, metricNameParts));
    }

    @Override
    public void setRollupMetricNames(String... metricNames) {
        rollupMetricNames = new HashSet<String>(metricNames.length);
        for (String metricName : metricNames) {
            rollupMetricNames.add(metricName);
        }
    }

    @Override
    public void addExclusiveRollupMetricName(String... metricNameParts) {
        if (exclusiveRollupMetricNames == null) {
            exclusiveRollupMetricNames = new HashSet<String>();
        }
        exclusiveRollupMetricNames.add(Strings.join(MetricNames.SEGMENT_DELIMITER, metricNameParts));
    }

    @Override
    public void setCustomMetricPrefix(String prefix) {
        this.customPrefix = prefix;
    }

    @Override
    public void setTrackChildThreads(boolean shouldTrack) {
        this.trackChildThreads = shouldTrack;
    }

    @Override
    public boolean trackChildThreads() {
        if (null == this.trackChildThreads) {
            TracedMethod parent = this.getParentTracedMethod();
            if (null == parent) {
                return false;
            } else {
                return parent.trackChildThreads();
            }
        }
        return this.trackChildThreads;
    }

    @Override
    public void addOutboundRequestHeaders(OutboundHeaders outboundHeaders) {
        Agent.LOG.severe("addOutboundRequestHeaders is only supported on subclasses of DefaultTracer: {0}");
    }

    @Override
    public void readInboundResponseHeaders(InboundHeaders inboundResponseHeaders) {
        Agent.LOG.severe("readInboundResponseHeaders is only supported on subclasses of DefaultTracer: {0}");
    }

    @Override
    public void reportAsExternal(ExternalParameters externalParameters) {
        Agent.LOG.severe("reportAsExternal is only supported on subclasses of DefaultTracer: {0}");
    }

    @Override
    public void reportAsExternal(com.newrelic.agent.bridge.external.ExternalParameters externalParameters) {
        Agent.LOG.severe("reportAsExternal is only supported on subclasses of DefaultTracer: {0}");
    }

    @Override
    public void markFinishTime() {
        finishTime.compareAndSet(null, System.nanoTime());
    }

    @Override
    public long getStartTimeInMillis() {
        return startTimeInMillis;
    }

    @Override
    public ExternalParameters getExternalParameters() {
        return null;
    }
}
