package com.newrelic.agent.service.module;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.newrelic.agent.Agent;
import com.newrelic.agent.IRPMService;
import com.newrelic.agent.commands.AbstractCommand;
import com.newrelic.agent.commands.CommandException;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;

/**
 * The `send_module_metadata` command is invoked with a list of module checksums to be sent to RPM. This map is stored under the
 * `module_checksums` key. The agent will send each of these modules to the New Relic service individually calling
 * {@link IRPMService#sendModuleMetadata(Module)}.  The command returns a map of checksums to a value indicating
 * whether or not the module data was sent (either `success` or `missing`).
 */
public class SendModuleMetadataCommand extends AbstractCommand {

    private final JarCollectorServiceProcessor processor;

    public SendModuleMetadataCommand(JarCollectorServiceProcessor processor) {
        super("send_module_metadata");
        this.processor = processor;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public Map process(IRPMService rpmService, Map arguments) throws CommandException {
        @SuppressWarnings("unchecked")
        List<String> checksums = (List<String>) arguments.get("module_checksums");

        Map<String, URL> checksumToUrl = Maps.newHashMap(processor.getSha1ChecksumToURL());
        // only retain the checksums we're interested in
        checksumToUrl.keySet().retainAll(checksums);

        Map<String, String> response = Maps.newHashMap();
        List<File> jars = Lists.newArrayList();

        try {
            for (Entry<String, URL> entry : checksumToUrl.entrySet()) {
                try {
                    File file = ClassMetadataFiles.createClassMetadataFile(entry.getValue());
                    jars.add(file);
                    Module module = new Module(entry.getKey(), file);

                    Agent.LOG.log(Level.FINER, "Sending module metadata with checksum {0} from {1}", entry.getKey(), file.getAbsolutePath());

                    try {
                        rpmService.sendModuleMetadata(module);
                        response.put(entry.getKey(), "success");
                    } catch (Exception e) {
                        // HttpError/LicenseException handled here
                        response.put(entry.getKey(), e.getMessage());
                    }

                } catch (IOException e) {
                    Agent.LOG.log(Level.FINEST, e, "{0}", e.getMessage());
                    response.put(entry.getKey(), e.getMessage());
                }
            }

            checksums.removeAll(response.keySet());

            for (String checksum : checksums) {
                response.put(checksum, "missing");
            }

            return response;
        } finally {
            for (File jar : jars) {
                jar.delete();
            }
        }
    }
}
