package com.newrelic.agent.service.analytics;

import com.newrelic.agent.DistributedTracePayloadImpl;
import com.newrelic.agent.bridge.TransportType;
import com.newrelic.agent.config.DistributedTracingConfig;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.service.ServiceUtils;
import com.newrelic.agent.stats.ApdexPerfZone;
import com.newrelic.agent.tracing.DistributedTraceService;
import com.newrelic.agent.transaction.TimeoutCause;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;

import java.io.IOException;
import java.io.Writer;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;

import static com.google.common.collect.Maps.newHashMap;

public class TransactionEvent extends AnalyticsEvent {
    static final float UNASSIGNED = Float.NEGATIVE_INFINITY;
    static final int UNASSIGNED_INT = Integer.MIN_VALUE;

    /**
     * Required. Always 'Transaction' for now. (static until needs to be different)
     */
    static final String TYPE = "Transaction";

    final String guid;
    final String parentSpanId;
    final String parentId;
    final String referrerGuid;
    final String tripId;
    final Integer referringPathHash;
    Integer pathHash;
    final String alternatePathHashes;
    final ApdexPerfZone apdexPerfZone;
    final String syntheticsResourceId;
    final String syntheticsMonitorId;
    final String syntheticsJobId;
    final int port;
    final boolean error;
    final boolean decider;

    /**
     * Required. Full metric name of the transaction
     */
    final String name;
    /**
     * Required. Duration of this transaction. Does not include queue time.
     */
    final float duration;

    final float totalTime;

    float timeToFirstByte = UNASSIGNED;

    float timeToLastByte = UNASSIGNED;

    /**
     * Optional. Equivalent to 'WebFrontend/QueueTime' metric.
     */
    float queueDuration = UNASSIGNED;

    /**
     * Optional. Equivalent to 'External/all' metric.
     */
    float externalDuration = UNASSIGNED;

    /**
     * Optional. Equivalent to 'External/all' call count.
     */
    float externalCallCount = UNASSIGNED;

    /**
     * Optional. Equivalent to 'Datastore/all' metric.
     */
    float databaseDuration = UNASSIGNED;

    /**
     * Optional. Equivalent to 'Datastore/all' call count.
     */
    float databaseCallCount = UNASSIGNED;

    /**
     * Optional. Equivalent to 'GC/cumulative' metric. Time spent in garbage collection across all transactions during
     * the timespan of this transaction.
     */
    float gcCumulative = UNASSIGNED;

    /**
     * Optional. Agent custom parameters. This includes request parameters.
     */
    Map<String, Object> agentAttributes;

    String appName;

    /**
     * Transaction timeout cause.
     * https://pages.datanerd.us/engineering-management/architecture-notes/notes/123/#nrtimeoutCause
     */
    final String timeoutCause;

    final float priority;

    final Map<String, Object> distributedTraceIntrinsics;

    public TransactionEvent(String appName, String subType, long timestamp, String name, float duration, String guid,
            String referringGuid, Integer port, String tripId, Integer referringPathHash, String alternatePathHashes,
            ApdexPerfZone apdexPerfZone, String syntheticsResourceId, String syntheticsMonitorId,
            String syntheticsJobId, boolean error, float pTotalTime, TimeoutCause timeoutCause,
            DistributedTracePayloadImpl payload, TransportType transportType, long transportDuration,
            long largestTransportDuration, String parentId, String parentSpanId, float priority) {
        super(TYPE, timestamp);
        this.name = name;
        this.duration = duration;
        this.guid = guid;
        this.referrerGuid = referringGuid;
        this.tripId = tripId;
        this.referringPathHash = referringPathHash;
        this.alternatePathHashes = alternatePathHashes;
        this.port = port == null ? UNASSIGNED_INT : port;
        this.appName = appName;
        this.apdexPerfZone = apdexPerfZone;
        this.syntheticsResourceId = syntheticsResourceId;
        this.syntheticsMonitorId = syntheticsMonitorId;
        this.syntheticsJobId = syntheticsJobId;
        this.error = error;
        this.totalTime = pTotalTime;
        this.timeoutCause = timeoutCause == null ? null : timeoutCause.cause;
        this.priority = priority;
        this.decider = (payload == null || payload.priority == null);
        this.parentId = parentId;
        this.parentSpanId = parentSpanId;

        DistributedTracingConfig distributedTracingConfig = ServiceFactory.getConfigService().getDefaultAgentConfig().getDistributedTracingConfig();
        if (distributedTracingConfig.isEnabled()) {
            // Better CAT
            DistributedTraceService distributedTraceService = ServiceFactory.getDistributedTraceService();
            this.distributedTraceIntrinsics = distributedTraceService.getIntrinsics(payload, guid,
                    tripId, transportType, transportDuration, largestTransportDuration, parentId, parentSpanId, priority);
        } else {
            this.distributedTraceIntrinsics = null;
        }
    }

    public float getDuration() {
        return duration;
    }

    public float getTotalTime() {
        return totalTime;
    }

    public float getTTFB() {
        return timeToFirstByte;
    }

    public float getTTLB() {
        return timeToLastByte;
    }

    public int getPort() {
        return port;
    }

    public String getName() {
        return name;
    }

    public float getExternalCallCount() {
        return externalCallCount;
    }

    public float getExternalDuration() {
        return externalDuration;
    }

    public float getDatabaseCallCount() {
        return databaseCallCount;
    }

    public float getDatabaseDuration() {
        return databaseDuration;
    }

    public boolean isError() {
        return error;
    }

    public String getGuid() {
        return guid;
    }

    public String getTripId() {
        return tripId;
    }

    public Integer getPathHash() {
        return pathHash;
    }

    public String getAlternatePathHashes() {
        return alternatePathHashes;
    }

    public String getReferrerGuid() {
        return referrerGuid;
    }

    public Integer getReferringPathHash() {
        return referringPathHash;
    }

    public String getApdexPerfZone() {
        if (apdexPerfZone != null) {
            return apdexPerfZone.getZone();
        }
        return null;
    }

    public String getTimeoutCause() {
        return timeoutCause;
    }

    public float getPriority() {
        return priority;
    }

    public Map<String, Object> getDistributedTraceIntrinsics() {
        return distributedTraceIntrinsics;
    }

    public String getParentId() {
        return parentId;
    }

    public String getParenSpanId() {
        return parentSpanId;
    }

    /*
     * The data should go up as 3 hashes. Example: [ { "webDuration":value, "databaseDuration":value, "timestamp":value,
     * "name":"value", "duration":value, "type":"value" }, { "user_param1":"value", "user_param2":value, }, {
     * "agent_param1": "value", "agent_param2": value }
     */
    @SuppressWarnings("unchecked")
    @Override
    public void writeJSONString(Writer out) throws IOException {
        JSONObject obj = new JSONObject();
        obj.put("type", type);
        obj.put("timestamp", timestamp);
        obj.put("name", name);
        obj.put("duration", duration);
        obj.put("error", error);
        obj.put("totalTime", totalTime);
        obj.put("priority", priority);

        if (timeToFirstByte != UNASSIGNED) {
            obj.put("timeToFirstByte", timeToFirstByte);
        }
        if (timeToLastByte != UNASSIGNED) {
            obj.put("timeToLastByte", timeToLastByte);
        }
        if (apdexPerfZone != null) {
            obj.put("apdexPerfZone", apdexPerfZone.getZone());
        }
        if (parentId != null) {
            obj.put("parentId", parentId);
        }
        if (parentSpanId != null) {
            obj.put("parentSpanId", parentSpanId);
        }

        DistributedTracingConfig distributedTracingConfig = ServiceFactory.getConfigService().getDefaultAgentConfig().getDistributedTracingConfig();
        if (!distributedTracingConfig.isEnabled()) {
            if (tripId != null) {
                obj.put("nr.tripId", tripId); // prefixed to be hidden by Insights
            }
            if (guid != null) {
                obj.put("nr.guid", guid); // prefixed to be hidden by Insights
            }
            if (pathHash != null) {
                obj.put("nr.pathHash", ServiceUtils.intToHexString(pathHash)); // prefixed to be hidden by Insights
            }
            if (referringPathHash != null) {
                obj.put("nr.referringPathHash", ServiceUtils.intToHexString(referringPathHash));
            }
            if (alternatePathHashes != null) {
                obj.put("nr.alternatePathHashes", alternatePathHashes); // prefixed to be hidden by Insights
            }
            if (referrerGuid != null) {
                obj.put("nr.referringTransactionGuid", referrerGuid); // prefixed to be hidden by Insights
            }
        }
        if (this.syntheticsResourceId != null) {
            obj.put("nr.syntheticsResourceId", this.syntheticsResourceId);
        }
        if (this.syntheticsMonitorId != null) {
            obj.put("nr.syntheticsMonitorId", this.syntheticsMonitorId);
        }
        if (this.syntheticsJobId != null) {
            obj.put("nr.syntheticsJobId", this.syntheticsJobId);
        }
        if (port != UNASSIGNED_INT) {
            obj.put("port", port);
        }
        if (queueDuration != UNASSIGNED) {
            obj.put("queueDuration", queueDuration);
        }
        if (externalDuration != UNASSIGNED) {
            obj.put("externalDuration", externalDuration);
        }
        if (externalCallCount > 0) {
            obj.put("externalCallCount", externalCallCount);
        }
        if (databaseDuration != UNASSIGNED) {
            obj.put("databaseDuration", databaseDuration);
        }
        if (databaseCallCount > 0) {
            obj.put("databaseCallCount", databaseCallCount);
        }
        if (gcCumulative != UNASSIGNED) {
            obj.put("gcCumulative", gcCumulative);
        }
        if (timeoutCause != null) {
            obj.put("nr.timeoutCause", timeoutCause);
        }

        if (distributedTraceIntrinsics != null && !distributedTraceIntrinsics.isEmpty()) {
            obj.putAll(distributedTraceIntrinsics);
        }

        Map<String, ? extends Object> filteredUserAtts = getUserFilteredMap(userAttributes);
        Map<String, ? extends Object> filteredAgentAtts = getFilteredMap(agentAttributes);
        if (filteredAgentAtts.isEmpty()) {
            if (filteredUserAtts.isEmpty()) {
                JSONArray.writeJSONString(Collections.singletonList(obj), out);
            } else {
                JSONArray.writeJSONString(Arrays.asList(obj, filteredUserAtts), out);
            }
        } else {
            JSONArray.writeJSONString(Arrays.asList(obj, filteredUserAtts, filteredAgentAtts), out);
        }
    }

    private Map<String, ? extends Object> getFilteredMap(Map<String, Object> input) {
        return ServiceFactory.getAttributesService().filterEventAttributes(appName, input);
    }

    private Map<String, ? extends Object> getUserFilteredMap(Map<String, Object> input) {
        // user attributes should have already been filtered for high security - this is just extra protection
        // high security is per an account - meaning it can not be different for various application names within a
        // JVM - so we can just check the default agent config
        if (!ServiceFactory.getConfigService().getDefaultAgentConfig().isHighSecurity()) {
            return getFilteredMap(input);
        } else {
            return Collections.emptyMap();
        }
    }

    @Override
    public boolean isValid() {
        // We don't need to validate the type "Transaction" every time.
        return true;
    }

    @Override
    public boolean decider() {
        return decider;
    }

    @Override
    public Map<String, Object> getAttributesCopy() {
        Map<String, Object> map = newHashMap(super.getAttributesCopy());
        map.putAll(this.agentAttributes);
        return map;
    }
}
