package com.newrelic.agent.service.analytics;

import com.google.common.collect.ComparisonChain;
import com.newrelic.agent.Agent;
import com.newrelic.agent.DistributedTracePayloadImpl;
import com.newrelic.agent.Harvestable;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.TransactionData;
import com.newrelic.agent.TransactionListener;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.config.AgentConfigListener;
import com.newrelic.agent.config.SpanEventsConfig;
import com.newrelic.agent.service.AbstractService;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsEngine;
import com.newrelic.agent.stats.StatsService;
import com.newrelic.agent.stats.StatsWork;
import com.newrelic.agent.stats.TransactionStats;
import com.newrelic.agent.tracers.Tracer;
import com.newrelic.agent.transport.HttpError;
import com.newrelic.agent.util.TimeConversion;
import com.newrelic.api.agent.DatastoreParameters;
import com.newrelic.api.agent.HttpParameters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

/**
 * The {@link SpanEventsServiceImpl} collects span events and transmits them to the collectors.
 * <p>
 * Ideally, all span events are stored until harvest and then transmitted. If the number of events exceeds a
 * configurable limit, events are replaced using a "reservoir" priority sampling algorithm.
 * <p>
 * This service can be configured using {@code span_events} with {@code enabled} or {@code max_samples_stored}.
 */
public class SpanEventsServiceImpl extends AbstractService implements AgentConfigListener, SpanEventsService, TransactionListener {

    private DistributedSamplingPriorityQueue<SpanEvent> reservoir;
    private List<Harvestable> harvestables = new ArrayList<>();

    private volatile SpanEventsConfig spanEventsConfig;

    // this value is not modifiable via config, but can be overridden by the collector
    private volatile int maxSamplesStored;

    public SpanEventsServiceImpl() {
        super(SpanEventsServiceImpl.class.getSimpleName());
        AgentConfig agentConfig = ServiceFactory.getConfigService().getDefaultAgentConfig();
        spanEventsConfig = agentConfig.getSpanEventsConfig();
        maxSamplesStored = spanEventsConfig.getMaxSamplesStored();
        ServiceFactory.getConfigService().addIAgentConfigListener(this);
        ServiceFactory.getTransactionService().addTransactionListener(this);
    }

    @Override
    public void dispatcherTransactionFinished(TransactionData transactionData, TransactionStats transactionStats) {
        // If this transaction is sampled and span events are enabled we should generate all of the transaction segment events
        if (transactionData.sampled() && isSpanEventsEnabled()) {
            // This is where all Transaction Segment Spans gets created. To only send specific types of Span Events, handle that here.
            Tracer rootTracer = transactionData.getRootTracer();
            try {
                createAndStoreSpanEvent(rootTracer, transactionData, true);
            } catch (Throwable t) {
                Agent.LOG.log(Level.FINER, t, "An error occurred creating span event for tracer: {0} in tx: {1}", rootTracer, transactionData);
            }

            Collection<Tracer> tracers = transactionData.getTracers();
            for (Tracer tracer : tracers) {
                if (tracer.isTransactionSegment()) {
                    try {
                        createAndStoreSpanEvent(tracer, transactionData, false);
                    } catch (Throwable t) {
                        Agent.LOG.log(Level.FINER, t, "An error occurred creating span event for tracer: {0} in tx: {1}", tracer, transactionData);
                    }
                }
            }
        }
    }

    private void createAndStoreSpanEvent(Tracer tracer, TransactionData transactionData, boolean isRoot) {
        boolean crossProcessOnly = spanEventsConfig.isCrossProcessOnly();
        if (crossProcessOnly && !isCrossProcessTracer(tracer)) {
            // We are in "cross_process_only" mode and we have a non datastore/external tracer. Return before we create anything.
            return;
        }

        DistributedTracePayloadImpl inboundPayload = transactionData.getSpanProxy().getInboundDistributedTracePayload();

        SpanEvent.SpanEventBuilder builder = SpanEvent.builder()
                .setAppName(transactionData.getApplicationName())
                .setGuid(tracer.getGuid())
                .setTraceId(inboundPayload != null ? inboundPayload.traceId : transactionData.getGuid())
                .setSampled(transactionData.sampled())
                .setParentId(getParentId(tracer, transactionData, crossProcessOnly))
                .setTransactionId(transactionData.getGuid())
                .setDurationInSeconds((float) tracer.getDuration() / TimeConversion.NANOSECONDS_PER_SECOND)
                .setName(tracer.getTransactionSegmentName())
                .setTimestamp(tracer.getStartTimeInMillis())
                .setPriority(transactionData.getPriority())
                .setExternalParameterAttributes(tracer.getExternalParameters())
                .setIsRootSpanEvent(isRoot)
                .setDecider(inboundPayload == null || inboundPayload.priority == null);

        storeEvent(builder.build());
    }

    private boolean isCrossProcessTracer(Tracer tracer) {
        return tracer.getExternalParameters() instanceof HttpParameters || tracer.getExternalParameters() instanceof DatastoreParameters;
    }

    private String getParentId(Tracer tracer, TransactionData transactionData, boolean crossProcessOnly) {
        if (crossProcessOnly) {
            // Cross process only uses transactionId for parenting instead of the parentId attribute so we do not have a parentId here
            return null;
        }

        // This is the non cross_process_only case where we "parent" using the parent tracer
        // or the inbound payload id if this is the first/root tracer and we have an inbound payload
        Tracer parentSegment = getParentTracerWithSpan(tracer.getParentTracer());
        DistributedTracePayloadImpl inboundPayload = transactionData.getInboundDistributedTracePayload();

        if (parentSegment != null) {
            return parentSegment.getGuid();
        } else if (inboundPayload != null) {
            // If we have an inbound payload we can use the id from the payload since it should be the id of the span that initiated this trace
            return inboundPayload.guid;
        } else {
            return null;
        }
    }

    @Override
    public Tracer getParentTracerWithSpan(Tracer parentTracer) {
        while (parentTracer != null && !parentTracer.isTransactionSegment()) {
            parentTracer = parentTracer.getParentTracer();
        }
        return parentTracer;
    }

    public void harvestEvents(final String appName) {
        if (!spanEventsConfig.isEnabled()) {
            reservoir = null;
            return;
        }
        if (maxSamplesStored <= 0) {
            clearReservoir();
            return;
        }

        long startTimeInNanos = System.nanoTime();

        int decidedLast = AdaptiveSampling.decidedLast(reservoir, spanEventsConfig.getTargetSamplesStored());

        // save a reference to the old reservoir to finish harvesting, and create a new one
        final DistributedSamplingPriorityQueue<SpanEvent> toSend = reservoir;
        createDistributedSamplingReservoir(decidedLast);

        if (toSend != null && toSend.size() > 0) {
            try {
                ServiceFactory.getRPMService(appName)
                        .sendSpanEvents(maxSamplesStored, toSend.getNumberOfTries(), Collections.unmodifiableList(toSend.asList()));
                final long durationInNanos = System.nanoTime() - startTimeInNanos;
                ServiceFactory.getStatsService().doStatsWork(new StatsWork() {
                    @Override
                    public void doWork(StatsEngine statsEngine) {
                        recordSupportabilityMetrics(statsEngine, durationInNanos, toSend);
                    }

                    @Override
                    public String getAppName() {
                        return appName;
                    }
                });

                if (toSend.size() < toSend.getNumberOfTries()) {
                    int dropped = toSend.getNumberOfTries() - toSend.size();
                    Agent.LOG.log(Level.WARNING, "Dropped {0} span events out of {1}.", dropped, toSend.getNumberOfTries());
                }
            } catch (HttpError e) {
                if (!e.discardHarvestData()) {
                    Agent.LOG.log(Level.FINE, "Unable to send span events. Unsent events will be included in the next harvest.", e);
                    // Save unsent data by merging it with toSend data using reservoir algorithm
                    reservoir.retryAll(toSend);
                } else {
                    // discard harvest data
                    toSend.clear();
                    Agent.LOG.log(Level.FINE, "Unable to send span events. Unsent events will be dropped.", e);
                }
            } catch (Exception e) {
                // discard harvest data
                toSend.clear();
                Agent.LOG.log(Level.FINE, "Unable to send span events. Unsent events will be dropped.", e);
            }
        }
    }

    private void recordSupportabilityMetrics(StatsEngine statsEngine, long durationInNanos, DistributedSamplingPriorityQueue<SpanEvent> reservoir) {
        int sent = reservoir.size();
        int seen = reservoir.getNumberOfTries();
        statsEngine.getStats(MetricNames.SUPPORTABILITY_SPAN_EVENT_TOTAL_EVENTS_SENT)
                .incrementCallCount(sent);
        statsEngine.getStats(MetricNames.SUPPORTABILITY_SPAN_EVENT_TOTAL_EVENTS_SEEN)
                .incrementCallCount(seen);
        statsEngine.getStats(MetricNames.SUPPORTABILITY_SPAN_EVENT_TOTAL_EVENTS_DISCARDED)
                .incrementCallCount(seen - sent);
        statsEngine.getResponseTimeStats(MetricNames.SUPPORTABILITY_SPAN_SERVICE_EVENT_HARVEST_TRANSMIT)
                .recordResponseTime(durationInNanos, TimeUnit.NANOSECONDS);
    }

    @Override
    public boolean isEnabled() {
        return spanEventsConfig.isEnabled();
    }

    private boolean isSpanEventsEnabled() {
        return spanEventsConfig.isEnabled() && maxSamplesStored > 0;
    }

    @Override
    protected void doStart() throws Exception {
        if (isEnabled()) {
            // track feature for angler
            StatsService statsService = ServiceFactory.getServiceManager().getStatsService();
            statsService.getMetricAggregator().incrementCounter(MetricNames.SUPPORTABILITY_SPAN_EVENTS);
        }
    }

    @Override
    protected void doStop() throws Exception {
        ServiceFactory.getTransactionService().removeTransactionListener(this);
        removeHarvestables();
        clearReservoir();
    }

    private void removeHarvestables() {
        for (Harvestable harvestable : harvestables) {
            ServiceFactory.getHarvestService().removeHarvestable(harvestable);
        }
    }

    @Override
    public void storeEvent(SpanEvent event) {
        if (isSpanEventsEnabled()) {
            DistributedSamplingPriorityQueue<SpanEvent> reservoir = getOrCreateDistributedSamplingReservoir();
            reservoir.add(event);
        }
    }

    public int getMaxSamplesStored() {
        return maxSamplesStored;
    }

    /**
     * For the setting to take effect, the reservoir must be re-created. Callers should harvest pending events before calling this.
     */
    public void setMaxSamplesStored(int maxSamplesStored) {
        this.maxSamplesStored = maxSamplesStored;
        createDistributedSamplingReservoir(0);
    }

    /**
     * Empty the current reservoir.
     */
    public void clearReservoir() {
        if (reservoir != null) {
            reservoir.clear();
        }
    }

    /**
     * There is only one event reservoir for Distributed Tracing so there only needs to be one harvestable for it.
     */
    @Override
    public void addHarvestableToService(String appName) {
        final String primaryApplication = ServiceFactory.getConfigService().getDefaultAgentConfig().getApplicationName();
        if (primaryApplication.equals(appName)) {
            Harvestable harvestable = new SpanEventHarvestableImpl(this, appName);
            ServiceFactory.getHarvestService().addHarvestable(harvestable);
            harvestables.add(harvestable);
        }
    }

    @Override
    public void configChanged(String appName, AgentConfig agentConfig) {
        boolean wasEnabled = isEnabled();
        spanEventsConfig = agentConfig.getSpanEventsConfig();

        if (!wasEnabled && isEnabled()) {
            // track feature for angler
            StatsService statsService = ServiceFactory.getServiceManager().getStatsService();
            statsService.getMetricAggregator().incrementCounter(MetricNames.SUPPORTABILITY_SPAN_EVENTS);
        }
    }

    @Override
    public DistributedSamplingPriorityQueue<SpanEvent> getOrCreateDistributedSamplingReservoir() {
        if (reservoir == null) {
            createDistributedSamplingReservoir(0);
        }
        return reservoir;
    }

    private void createDistributedSamplingReservoir(int decidedLast) {
        final String appName = ServiceFactory.getConfigService().getDefaultAgentConfig().getApplicationName();
        int target = spanEventsConfig.getTargetSamplesStored();
        reservoir = new DistributedSamplingPriorityQueue<>(appName, "Span Event Service", maxSamplesStored, decidedLast, target, SPAN_EVENT_COMPARATOR);
    }

    // This is where you can add secondary sorting for Span Events
    private static final Comparator<SpanEvent> SPAN_EVENT_COMPARATOR = new Comparator<SpanEvent>() {
        @Override
        public int compare(SpanEvent left, SpanEvent right) {
            return ComparisonChain.start()
                    .compare(right.getPriority(), left.getPriority()) // Take highest priority first
                    .result();
        }
    };

}
