package com.newrelic.agent.service.analytics;

import com.newrelic.agent.Harvestable;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.config.SpanEventsConfig;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsEngine;
import com.newrelic.agent.stats.StatsWork;
import com.newrelic.agent.stats.StatsWorks;
import com.newrelic.agent.transport.DataSenderImpl;

import java.util.Map;
import java.util.concurrent.TimeUnit;

public class SpanEventHarvestableImpl implements Harvestable {

    private final String appName;
    private final SpanEventsServiceImpl spanEventService;

    private long lastHarvest;

    public SpanEventHarvestableImpl(SpanEventsServiceImpl spanEventsService, String appName) {
        this.appName = appName;
        this.lastHarvest = System.nanoTime();
        this.spanEventService = spanEventsService;
    }

    @Override
    public String getEndpointMethodName() {
        return DataSenderImpl.SPAN_EVENT_DATA_METHOD;
    }

    @Override
    public void harvest() {
        recordIntervalMetric();
        spanEventService.harvestEvents(appName);
    }

    @Override
    public void configure(Map<String, Object> configuration) {
        Number reportPeriod = (Number) configuration.get(Harvestable.REPORT_PERIOD_IN_SECONDS_KEY_NAME);
        ServiceFactory.getStatsService().doStatsWork(
                StatsWorks.getRecordMetricWork(MetricNames.SUPPORTABILITY_ERROR_SERVICE_REPORT_PERIOD_IN_SECONDS,
                        reportPeriod.longValue()));

        Number maxSamplesCollector = (Number) configuration.get(SpanEventsConfig.MAX_SAMPLES_STORED);
        int newMaxSamplesStored = maxSamplesCollector != null ?
                Math.min(maxSamplesCollector.intValue(), SpanEventsConfig.DEFAULT_MAX_SPAN_EVENTS_PER_HARVEST) :
                SpanEventsConfig.DEFAULT_MAX_SPAN_EVENTS_PER_HARVEST;

        int currentMaxSamplesStored = spanEventService.getMaxSamplesStored();
        if (newMaxSamplesStored != currentMaxSamplesStored) {
            spanEventService.harvestEvents(appName);
            spanEventService.setMaxSamplesStored(newMaxSamplesStored);
        }
    }

    @Override
    public String getAppName() {
        return appName;
    }

    private void recordIntervalMetric() {
        long startTimeInNanos = System.nanoTime();
        final long harvestIntervalInNanos = startTimeInNanos - lastHarvest;
        lastHarvest = startTimeInNanos;

        ServiceFactory.getStatsService().doStatsWork(new StatsWork() {
            @Override
            public void doWork(StatsEngine statsEngine) {
                if (harvestIntervalInNanos > 0) {
                    statsEngine.getResponseTimeStats(
                            MetricNames.SUPPORTABILITY_SPAN_SERVICE_EVENT_HARVEST_INTERVAL).recordResponseTime(harvestIntervalInNanos, TimeUnit.NANOSECONDS);
                }
            }

            @Override
            public String getAppName() {
                return appName;
            }
        });
    }

}
