package com.newrelic.agent.service.analytics;

import com.newrelic.agent.Agent;
import com.newrelic.agent.HarvestListener;
import com.newrelic.agent.MetricNames;
import com.newrelic.agent.service.AbstractService;
import com.newrelic.agent.service.Service;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsEngine;
import com.newrelic.agent.transport.HttpError;

import java.text.MessageFormat;
import java.util.ArrayDeque;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;

/**
 * A generic event service for storing and sending internal-only custom insights events. Events must extend
 * {@link BaseInternalCustomEvent} to use this service and share the same reservoir.
 */
public class InternalCustomEventService extends AbstractService implements Service, HarvestListener {

    private static final int MAX_EVENTS_PER_HARVEST = 501; // 500 row events and 1 stats event

    private final ConcurrentHashMap<String, DistributedSamplingPriorityQueue<BaseInternalCustomEvent>> reservoirForApp = new ConcurrentHashMap<>();
    private final ArrayDeque<DistributedSamplingPriorityQueue<BaseInternalCustomEvent>> pendingEvents = new ArrayDeque<>();

    public InternalCustomEventService(String name) {
        super(name);
    }

    public InternalCustomEventService() {
        this(InternalCustomEventService.class.getSimpleName());
    }

    @Override
    public void beforeHarvest(String appName, StatsEngine statsEngine) {
        DistributedSamplingPriorityQueue<BaseInternalCustomEvent> current = reservoirForApp.put(appName,
                new DistributedSamplingPriorityQueue<BaseInternalCustomEvent>(appName, "Custom Events Service", MAX_EVENTS_PER_HARVEST));
        if (current != null && current.size() > 0) {
            if (pendingEvents.size() < TransactionEventsService.MAX_UNSENT_SYNTHETICS_HOLDERS) {
                pendingEvents.add(current);
            } else {
                Agent.LOG.fine(MessageFormat.format("{0} events were discarded.", pendingEvents.size()));
                statsEngine.getStats(MetricNames.SUPPORTABILITY_INTERNAL_CUSTOM_EVENTS_TOTAL_EVENTS_DISCARDED)
                        .incrementCallCount(pendingEvents.size());
            }
        }

        /*
         * If things are working normally, pendingEvents now contains either 0 or 1 element. If there is one,
         * we simply send it. But if we are catching up after an outage, we send a few of them, and then defer to the
         * next call here and so on. This prevents having all the Agents in the world send e.g. 25 * 200 events when
         * recovering from a long outage. Worse case we'll catch up by 4 buffers per minute, so about 6 minutes.
         */
        final int maxToSend = 5;
        for (int nSent = 0; nSent < maxToSend; nSent++) {
            DistributedSamplingPriorityQueue<BaseInternalCustomEvent> toSend = pendingEvents.poll();
            if (toSend == null) {
                break;
            }

            try {
                ServiceFactory.getRPMService(appName)
                        .sendInternalCustomEvents(MAX_EVENTS_PER_HARVEST, toSend.getNumberOfTries(), toSend.asList());

                statsEngine.getStats(MetricNames.SUPPORTABILITY_INTERNAL_CUSTOM_EVENTS_TOTAL_EVENTS_SENT)
                        .incrementCallCount(toSend.size());

                statsEngine.getStats(MetricNames.SUPPORTABILITY_INTERNAL_CUSTOM_EVENTS_TOTAL_EVENTS_SEEN)
                        .incrementCallCount(toSend.getNumberOfTries());
            } catch (HttpError e) {
                if (!e.discardHarvestData()) {
                    Agent.LOG.log(Level.FINE, "Unable to send events. Unsent events will be included in the next harvest.", e);
                    // Save unsent data by merging it with toSend data using reservoir algorithm
                    pendingEvents.add(toSend);
                    break;
                } else {
                    Agent.LOG.log(Level.FINE, "Unable to send events. Unsent events will be dropped.", e);
                    break;
                }
            } catch (Exception e) {
                Agent.LOG.log(Level.FINE, "Unable to send events. Unsent events will be dropped.", e);
                break;
            }
        }
    }

    @Override
    public void afterHarvest(String appName) {
    }

    @Override
    public boolean isEnabled() {
        return true;
    }

    @Override
    protected void doStart() throws Exception {
        ServiceFactory.getHarvestService().addHarvestListener(this);
    }

    @Override
    protected void doStop() throws Exception {
        ServiceFactory.getHarvestService().removeHarvestListener(this);
        reservoirForApp.clear();
    }

    public void addInternalCustomEvent(BaseInternalCustomEvent event) {
        DistributedSamplingPriorityQueue<BaseInternalCustomEvent> reservoir = reservoirForApp.get(event.getAppName());
        while (reservoir == null) {
            // I don't think this loop can actually execute more than once, but it's prudent to assume it can.
            reservoirForApp.putIfAbsent(event.getAppName(),
                    new DistributedSamplingPriorityQueue<BaseInternalCustomEvent>(event.getAppName(), "Custom Events Service", MAX_EVENTS_PER_HARVEST));
            reservoir = reservoirForApp.get(event.getAppName());
        }
        reservoir.add(event);
    }
}
