package com.newrelic.agent.profile.v2;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;

import org.json.simple.JSONArray;
import org.json.simple.JSONStreamAware;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * 
 */
public abstract class BaseTree<S extends ProfileSegment> implements JSONStreamAware {

    private final ConcurrentMap<ProfiledMethod, S> rootSegments;

    protected final IProfile profile;

    protected BaseTree(IProfile profile) {
        this.profile = profile;
        this.rootSegments = Maps.newConcurrentMap();
    }

    protected final S add(ProfiledMethod method, S parent, boolean runnable) {
        S result = add(method, parent);
        result.incrementCallCount(runnable);

        return result;
    }

    private S add(ProfiledMethod method, S parent) {
        S result;
        if (parent == null) {
            result = rootSegments.get(method);
            if (result == null) {
                result = createProfiledMethod(method);
                S previousValue = rootSegments.putIfAbsent(method, result);
                if (null != previousValue) {
                    return previousValue;
                }
            }
        } else {
            result = parent.addChild(method);
        }
        return result;
    }

    protected abstract S createProfiledMethod(ProfiledMethod method);

    /**
     * Return number of calls to given method from all invocations
     * 
     * @param stackElement
     * @return
     *
    public int getCallCount(StackTraceElement stackElement) {
        ProfiledMethod method = ProfiledMethod.newProfiledMethod(profile, stackElement);
        if (method == null) {
            return 0;
        }

        int count = 0;
        for (ProfileSegment segment : rootSegments.values()) {
            count += segment.getCallCount(method);
        }
        return count;
    }

    /**
     * Returns the number of distinct method invocation nodes in the tree.
     * 
     * @return
     */
    public final int getCallSiteCount() {
        int count = 0;
        for (S segment : rootSegments.values()) {
            count += segment.getCallSiteCount();
        }
        return count;
    }

    /**
     * Returns the segment that matches the distinct method invocation
     * 
     * @param profiledMethod the method invocation to retrieve
     * @return a segment representing the provided method
     */
    public final S getSegment(ProfiledMethod profiledMethod) {
        return rootSegments.get(profiledMethod);
    }

    public final Collection<S> getRootSegments() {
        return rootSegments.values();
    }

    public final int getRootCount() {
        return getRootSegments().size();
    }

    public final int getMethodCount() {
        Set<ProfiledMethod> methodNames = new HashSet<ProfiledMethod>();
        for (S segment : rootSegments.values()) {
            methodNames.addAll(segment.getMethods());
        }
        return methodNames.size();
    }

    @Override
    public void writeJSONString(Writer out) throws IOException {
        Collection<S> rootSegments = getRootSegments();
        ArrayList<Object> list = Lists.newArrayListWithCapacity(rootSegments.size() + 1);
        list.add(getExtraData());
        list.addAll(rootSegments);
        JSONArray.writeJSONString(list, out);
    }

    protected abstract Map<String, Object> getExtraData();
}
