package com.newrelic.agent.profile;

import com.google.common.annotations.VisibleForTesting;
import com.newrelic.agent.Agent;
import com.newrelic.agent.IgnoreSilentlyException;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsEngine;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;

/**
 * Collect stack traces for xray session profiles.
 *
 * This class is thread-safe but the profiles are not.
 */
public class XrayProfilingTask implements ProfilingTask {

    private final List<ProfilerParameters> profilesToAdd = new CopyOnWriteArrayList<>();
    private final List<ProfilerParameters> profilesToRemove = new CopyOnWriteArrayList<>();
    private final List<IProfile> profiles = new ArrayList<>();
    private final AtomicBoolean sendProfiles = new AtomicBoolean(false);
    private final ProfileSampler profileSampler = new ProfileSampler();

    public XrayProfilingTask() {
    }

    @Override
    public void addProfile(ProfilerParameters parameters) {
        profilesToAdd.add(parameters);
    }

    @Override
    public void removeProfile(ProfilerParameters parameters) {
        profilesToRemove.add(parameters);
    }

    @Override
    public void beforeHarvest(String appName, StatsEngine statsEngine) {
    }

    @Override
    public void afterHarvest(String appName) {
        sendProfiles.set(true);
    }

    @Override
    public void run() {
        try {
            sampleStackTraces();
        } catch (Throwable t) {
            String msg = MessageFormat.format("Error sampling stack traces: {0}", t);
            if (Agent.LOG.isLoggable(Level.FINEST)) {
                Agent.LOG.log(Level.FINEST, msg, t);
            } else {
                Agent.LOG.finer(msg);
            }
        }
    }

    private void sampleStackTraces() {
        if (sendProfiles.getAndSet(false)) {
            sendProfiles();
        } else {
            removeProfiles();
        }
        addProfiles();
        profileSampler.sampleStackTraces(profiles);
    }

    private void removeProfiles() {
        for (ProfilerParameters parameters : profilesToRemove) {
            IProfile profile = getProfile(parameters);
            if (profile != null) {
                profiles.remove(profile);
                Agent.LOG.info(MessageFormat.format("Stopped xray session profiling: {0}", parameters.getKeyTransaction()));
            }
            profilesToRemove.remove(parameters);
        }
    }

    private void addProfiles() {
        for (ProfilerParameters parameters : profilesToAdd) {
            IProfile profile = getProfile(parameters);
            if (profile == null) {
                profile = createProfile(parameters);
                profile.start();
                profiles.add(profile);
                Agent.LOG.info(MessageFormat.format("Started xray session profiling: {0}", parameters.getKeyTransaction()));
            }
            profilesToAdd.remove(parameters);
        }
    }

    @VisibleForTesting
    List<IProfile> getProfiles() {
        return new CopyOnWriteArrayList<>(profiles);
    }

    private IProfile getProfile(ProfilerParameters parameters) {
        for (IProfile profile : profiles) {
            if (profile.getProfilerParameters().equals(parameters)) {
                return profile;
            }
        }
        return null;
    }

    private IProfile createProfile(ProfilerParameters parameters) {
        return new KeyTransactionProfile(new Profile(parameters));
    }

    private void sendProfiles() {
        ListIterator<IProfile> it = profiles.listIterator();
        while (it.hasNext()) {
            IProfile profile = it.next();
            int otherCallSiteCount = profile.getProfileTree(ThreadType.BasicThreadType.OTHER).getCallSiteCount();
            if (otherCallSiteCount > 0) {
                it.remove();
                profile.end();
                IProfile nProfile = createProfile(profile.getProfilerParameters());
                nProfile.start();
                it.add(nProfile);
                sendProfile(profile);
                ProfilerParameters parameters = profile.getProfilerParameters();
                if (profilesToRemove.contains(parameters)) {
                    profiles.remove(profile);
                    Agent.LOG.info(MessageFormat.format("Stopped xray session profiling: {0}", parameters.getKeyTransaction()));
                }
            }
        }
    }

    private void sendProfile(IProfile profile) {
        try {
            if (Agent.LOG.isLoggable(Level.FINE)) {
                String msg = MessageFormat.format("Sending Xray profile: {0}", profile.getProfilerParameters().getXraySessionId());
                Agent.LOG.fine(msg);
            }
            String appName = profile.getProfilerParameters().getAppName();
            List<Long> ids = ServiceFactory.getRPMService(appName).sendProfileData(Arrays.<ProfileData>asList(profile));
            if (Agent.LOG.isLoggable(Level.FINE)) {
                Agent.LOG.fine(MessageFormat.format("Xray profile id: {0}", ids));
            }
        } catch (IgnoreSilentlyException e) {
        } catch (Exception e) {
            // HttpError/LicenseException handled here
            String msg = MessageFormat.format("Unable to send profile data: {0}", e);
            if (Agent.LOG.isLoggable(Level.FINEST)) {
                Agent.LOG.log(Level.FINEST, msg, e);
            } else {
                Agent.LOG.fine(msg);
            }
        }
    }

}
