package com.newrelic.agent.profile;

import java.lang.management.ManagementFactory;
import java.text.MessageFormat;

import com.newrelic.agent.Agent;

/**
 * Calculate the sample period of a profiling task based on its CPU usage.
 * 
 * This class is not thread-safe: use only in the Harvest Service thread, except that the run() method is called in the
 * profiling task thread.
 * 
 */
public class XrayCpuTimeController extends AbstractController implements ProfilingTaskController {

    private long threadId;
    private long startCpuTimeInNanos;
    private long startTimeInNanos;
    private volatile long currentThreadId = Thread.currentThread().getId();

    public XrayCpuTimeController(ProfilingTask profilingTask) {
        super(profilingTask);
    }

    @Override
    public void run() {
        currentThreadId = Thread.currentThread().getId();
        super.run();
    }

    @Override
    protected int doCalculateSamplePeriodInMillis() {
        int samplePeriod = getSamplePeriodInMillis();
        long nThreadId = getCurrentThreadId();
        long oThreadId = threadId;
        threadId = nThreadId;
        long endCpuTimeInNanos;
        try {
            endCpuTimeInNanos = getThreadCpuTimeInNanos();
        } catch (Throwable t) {
            Agent.LOG.fine(MessageFormat.format("Error getting thread cpu time: {0}", t));
            return samplePeriod;
        }
        long endTimeInNanos = getTimeInNanos();
        if (oThreadId == nThreadId) {
            samplePeriod = calculateSamplePeriod(endTimeInNanos - startTimeInNanos, endCpuTimeInNanos
                    - startCpuTimeInNanos);
        }
        startCpuTimeInNanos = endCpuTimeInNanos;
        startTimeInNanos = endTimeInNanos;
        return samplePeriod;
    }

    private int calculateSamplePeriod(long timeInNanos, long cpuTimeInNanos) {
        if (cpuTimeInNanos == 0 || timeInNanos == 0) {
            return getSamplePeriodInMillis();
        }
        // utilization of total CPU capacity
        float cpuUtilization = ((float) cpuTimeInNanos) / (timeInNanos * getProcessorCount());
        // sample period to attain CPU utilization target
        return (int) ((cpuUtilization * getSamplePeriodInMillis()) / TARGET_UTILIZATION);
    }

    protected long getThreadCpuTimeInNanos() {
        return ManagementFactory.getThreadMXBean().getThreadCpuTime(threadId);
    }

    protected long getCurrentThreadId() {
        return currentThreadId;
    }

    protected long getTimeInNanos() {
        return System.nanoTime();
    }

}
