package com.newrelic.agent.profile;

import java.util.concurrent.atomic.AtomicLong;

/**
 * Calculate the sampling period of a profiling task based on the time spent in the run method.
 * 
 * This class is not thread-safe: use only in the Harvest Service thread, except that the run method is called in
 * another thread.
 * 
 */
public class XrayClockTimeController extends AbstractController {

    private long startTimeInNanos;
    private final AtomicLong runTime = new AtomicLong();

    public XrayClockTimeController(ProfilingTask profilingTask) {
        super(profilingTask);
    }

    @Override
    public void run() {
        long startTime = System.nanoTime();
        super.run();
        runTime.addAndGet(System.nanoTime() - startTime);
    }

    @Override
    protected int doCalculateSamplePeriodInMillis() {
        long runTimeInNanos = getAndResetRunTimeInNanos();
        long endTimeInNanos = getTimeInNanos();
        int samplePeriod = getSamplePeriodInMillis();
        if (startTimeInNanos > 0) {
            long timeInNanos = endTimeInNanos - startTimeInNanos;
            samplePeriod = calculateSamplePeriodInMillis(timeInNanos, runTimeInNanos);
        }
        startTimeInNanos = endTimeInNanos;
        return samplePeriod;
    }

    private int calculateSamplePeriodInMillis(long timeInNanos, long runTimeInNanos) {
        if (runTimeInNanos == 0 || timeInNanos == 0) {
            return getSamplePeriodInMillis();
        }
        // Utilization of total capacity
        float runUtilization = ((float) runTimeInNanos) / (timeInNanos * getProcessorCount());
        // Sample period to attain utilization target
        return (int) ((runUtilization * getSamplePeriodInMillis()) / TARGET_UTILIZATION);
    }

    protected long getTimeInNanos() {
        return System.nanoTime();
    }

    protected long getAndResetRunTimeInNanos() {
        return runTime.getAndSet(0);
    }

}
