package com.newrelic.agent.logging;

import java.io.IOException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.text.MessageFormat;
import java.util.Map;
import java.util.logging.Level;

import com.google.common.collect.Maps;

import com.newrelic.agent.Agent;

import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.ConsoleAppender;
import ch.qos.logback.core.Context;
import ch.qos.logback.core.FileAppender;
import ch.qos.logback.core.encoder.Encoder;
import ch.qos.logback.core.rolling.FixedWindowRollingPolicy;
import ch.qos.logback.core.rolling.RollingFileAppender;
import ch.qos.logback.core.rolling.RollingPolicy;
import ch.qos.logback.core.rolling.SizeAndTimeBasedRollingPolicy;
import ch.qos.logback.core.rolling.SizeBasedTriggeringPolicy;
import ch.qos.logback.core.rolling.TimeBasedRollingPolicy;
import ch.qos.logback.core.util.FileSize;

class LogbackLogger implements IAgentLogger {

    /**
     * The default prudent value.
     */
    private static final boolean PRUDENT_VALUE = false;
    /**
     * The minimum number of files.
     */
    private static final int MIN_FILE_COUNT = 1;
    /**
     * The name of the console appender.
     */
    private static final String CONSOLE_APPENDER_NAME = "Console";
    /**
     * The name of the file appender.
     */
    private static final String FILE_APPENDER_NAME = "File";
    /**
     * The default is to append to the file.
     */
    private static final boolean APPEND_TO_FILE = true;
    /**
     * The pattern to use for log messages.
     */
    private static final String CONVERSION_PATTERN = "%d{\"MMM d, yyyy HH:mm:ss ZZZZ\"} [%pid %i] %logger %ml: %m%n";
    /**
     * Where the console log should go.
     */
    private static final String SYSTEM_OUT = "System.out";

    /**
     * The logger for this LogbackLogger.
     */
    private final Logger logger;
    private final Map<String, IAgentLogger> childLoggers = Maps.newConcurrentMap();

    /**
     * Creates this LogbackLogger.
     *
     * @param name Name of this logger.
     */
    private LogbackLogger(final String name, boolean isAgentRoot) {
        super();
        logger = (Logger) LoggerFactory.getLogger(name);

        if (isAgentRoot) {
            logger.setAdditive(false);
            FineFilter.getFineFilter().start();
        }
    }

    @Override
    public void severe(String pMessage) {
        logger.error(pMessage);
    }

    @Override
    public void error(String pMessage) {
        logger.error(pMessage);
    }

    @Override
    public void warning(String pMessage) {
        logger.warn(pMessage);
    }

    @Override
    public void info(String pMessage) {
        logger.info(pMessage);
    }

    @Override
    public void config(String pMessage) {
        logger.info(pMessage);
    }

    @Override
    public void fine(String pMessage) {
        logger.debug(LogbackMarkers.FINE_MARKER, pMessage);
    }

    @Override
    public void finer(String pMessage) {
        logger.debug(LogbackMarkers.FINER_MARKER, pMessage);
    }

    @Override
    public void finest(String pMessage) {
        logger.trace(LogbackMarkers.FINEST_MARKER, pMessage);
    }

    @Override
    public void debug(String pMessage) {
        logger.debug(pMessage);
    }

    @Override
    public void trace(String pMessage) {
        logger.trace(pMessage);
    }

    @Override
    public boolean isFineEnabled() {
        return logger.isDebugEnabled() && FineFilter.getFineFilter().isEnabledFor(java.util.logging.Level.FINE);
    }

    @Override
    public boolean isFinerEnabled() {
        return logger.isDebugEnabled() && FineFilter.getFineFilter().isEnabledFor(java.util.logging.Level.FINER);
    }

    @Override
    public boolean isFinestEnabled() {
        return logger.isTraceEnabled();
    }

    @Override
    public boolean isDebugEnabled() {
        return logger.isDebugEnabled();
    }

    @Override
    public boolean isTraceEnabled() {
        return logger.isTraceEnabled();
    }

    @Override
    public boolean isLoggable(java.util.logging.Level pLevel) {
        LogbackLevel level = LogbackLevel.getLevel(pLevel);
        return ((level == null) ? false
                : (logger.isEnabledFor(level.getLogbackLevel()) && FineFilter.getFineFilter().isEnabledFor(pLevel)));
    }

    @Override
    public void log(java.util.logging.Level pLevel, final String pMessage, final Throwable pThrowable) {
        if (isLoggable(pLevel)) {
            final LogbackLevel level = LogbackLevel.getLevel(pLevel);
            AccessController.doPrivileged(new PrivilegedAction<Void>() {
                @Override
                public Void run() {
                    logger.log(level.getMarker(), Logger.FQCN, level.getLogbackLevel().toLocationAwareLoggerInteger(
                            level.getLogbackLevel()), pMessage, null, pThrowable);

                    return null;
                }
            });
        }
    }

    @Override
    public void log(java.util.logging.Level pLevel, String pMessage) {
        LogbackLevel level = LogbackLevel.getLevel(pLevel);
        logger.log(level.getMarker(), Logger.FQCN, level.getLogbackLevel().toLocationAwareLoggerInteger(
                level.getLogbackLevel()), pMessage, null, null);
    }

    @Override
    public void log(Level pLevel, String pMessage, Object[] pArgs, Throwable pThorwable) {
        LogbackLevel level = LogbackLevel.getLevel(pLevel);
        logger.log(level.getMarker(), Logger.FQCN, level.getLogbackLevel().toLocationAwareLoggerInteger(
                level.getLogbackLevel()), pMessage, pArgs, pThorwable);
    }

    @Override
    public IAgentLogger getChildLogger(Class<?> pClazz) {
        return getChildLogger(pClazz.getName());
    }

    @Override
    public IAgentLogger getChildLogger(String pFullName) {
        IAgentLogger logger = LogbackLogger.create(pFullName, false);
        childLoggers.put(pFullName, logger);
        return logger;
    }

    /**
     * Sets the level.
     *
     * @param level The level to set on the logger.
     */
    public void setLevel(String level) {
        LogbackLevel newLevel = LogbackLevel.getLevel(level, LogbackLevel.INFO);
        logger.setLevel(newLevel.getLogbackLevel());
        FineFilter.getFineFilter().setLevel(newLevel.getJavaLevel());

    }

    /**
     * Gets the level. Finest will be returned as trace.
     *
     * @return The current level of the logger.
     */
    public String getLevel() {
        if (logger.getLevel() == ch.qos.logback.classic.Level.DEBUG) {
            return FineFilter.getFineFilter().getLevel().toString();
        }
        return logger.getLevel().toString();
    }

    /**
     * Removes the console appender.
     */
    public void removeConsoleAppender() {
        logger.detachAppender(CONSOLE_APPENDER_NAME);
    }

    /**
     * Creates a console appender if none exists.
     */
    public void addConsoleAppender() {
        if (logger.getAppender(CONSOLE_APPENDER_NAME) != null) {
            return;
        }
        ConsoleAppender<ILoggingEvent> consoleAppender = new ConsoleAppender<ILoggingEvent>();
        consoleAppender.setName(CONSOLE_APPENDER_NAME);
        consoleAppender.setTarget(SYSTEM_OUT);
        consoleAppender.setEncoder(getEncoder(logger.getLoggerContext()));
        consoleAppender.setContext(logger.getLoggerContext());
        consoleAppender.addFilter(FineFilter.getFineFilter());
        consoleAppender.start();
        logger.addAppender(consoleAppender);

    }

    /**
     * Adds a file appender.
     *
     * @param fileName      Name of the appender.
     * @param logLimitBytes Log limit
     * @param fileCount     The number of files.
     * @throws IOException Thrown is a problem creating the file appender.
     */

    public void addFileAppender(String fileName, long logLimitBytes, int fileCount, boolean isDaily) throws IOException {
        if (logger.getAppender(FILE_APPENDER_NAME) != null) {
            return;
        }

        FileSize fileSize = new FileSize(logLimitBytes);
        FileAppender<ILoggingEvent> fileAppender = createFileAppender(fileCount, fileSize, fileName, isDaily);
        fileAppender.addFilter(FineFilter.getFineFilter());
        fileAppender.start();
        logger.addAppender(fileAppender);
    }

    /**
     * Initialize the correct type of rolling policy for daily log rolling.
     * @param fileSize file size limit for each file
     * @param fileName prefix for log file names
     * @return either a {@link TimeBasedRollingPolicy} or {@link SizeAndTimeBasedRollingPolicy}, depending on what was requested,
     * that will need additional settings configured
     */
    private TimeBasedRollingPolicy<ILoggingEvent> initializeDailyRollingPolicy(FileSize fileSize, int fileCount, String fileName) {
        if (fileSize.getSize() > 0) {
            SizeAndTimeBasedRollingPolicy<ILoggingEvent> sizeAndTimeBasedRollingPolicy = new SizeAndTimeBasedRollingPolicy<ILoggingEvent>();
            sizeAndTimeBasedRollingPolicy.setMaxFileSize(fileSize);
            sizeAndTimeBasedRollingPolicy.setFileNamePattern(fileName + ".%d{yyyy-MM-dd}.%i");
            sizeAndTimeBasedRollingPolicy.setTotalSizeCap(new FileSize(fileSize.getSize() * fileCount));
            return sizeAndTimeBasedRollingPolicy;
        }
        TimeBasedRollingPolicy<ILoggingEvent> timeBasedRollingPolicy = new TimeBasedRollingPolicy<ILoggingEvent>();
        timeBasedRollingPolicy.setFileNamePattern(fileName + ".%d{yyyy-MM-dd}");
        return timeBasedRollingPolicy;
    }

    /**
     * Build the full-fledged daily rolling policy.
     * @param parentFileAppender the parent appender
     * @param fileCount maximum number of log files
     * @param fileSize maximum size of a given log file
     * @param fileName prefix for log file names
     * @return a fully-initialized {@link RollingPolicy}
     */
    private RollingPolicy buildDailyRollingPolicy(FileAppender<ILoggingEvent> parentFileAppender, int fileCount, FileSize fileSize, String fileName) {
        TimeBasedRollingPolicy<ILoggingEvent> timePolicy = initializeDailyRollingPolicy(fileSize, fileCount, fileName);
        timePolicy.setContext(logger.getLoggerContext());
        timePolicy.setMaxHistory(fileCount);
        timePolicy.setParent(parentFileAppender);
        return timePolicy;
    }

    /**
     * Build a fixed window rolling policy, used when daily logs are not enabled. The size of each file is controlled
     * via a {@link ch.qos.logback.core.rolling.TriggeringPolicy}.
     * @param parentFileAppender the parent appender
     * @param fileCount maximum number of log files
     * @param fileName prefix for log file names
     * @return a full-initialized {@link RollingPolicy}
     */
    private RollingPolicy buildFixedWindowRollingPolicy(FileAppender<ILoggingEvent> parentFileAppender, int fileCount, String fileName) {
        FixedWindowRollingPolicy rollingPolicy = new FixedWindowRollingPolicy();
        rollingPolicy.setContext(logger.getLoggerContext());
        rollingPolicy.setParent(parentFileAppender);
        rollingPolicy.setMinIndex(MIN_FILE_COUNT);
        if (fileCount == 1) {
            // if the agent is configured to preserve exactly 1 log, then setting the min and max to the same value
            // ensures that logback will correctly rollover the log, keeping none of the rollovers
            rollingPolicy.setMaxIndex(1);
        } else {
            rollingPolicy.setMaxIndex(fileCount - 1);
        }
        rollingPolicy.setFileNamePattern(fileName + ".%i");
        return rollingPolicy;
    }

    /**
     * Initialize a fresh {@link RollingFileAppender} that is correctly configured
     * @param fileName file name for this appender
     * @return a {@link RollingFileAppender} that will need a {@link RollingPolicy} and
     * {@link ch.qos.logback.core.rolling.TriggeringPolicy} associated with it.
     */
    private RollingFileAppender<ILoggingEvent> initializeRollingFileAppender(String fileName) {
        RollingFileAppender<ILoggingEvent> fileAppender = new RollingFileAppender<ILoggingEvent>();
        // this needs to be set here or else it will cause a warning
        fileAppender.setContext(logger.getLoggerContext());
        fileAppender.setEncoder(getEncoder(logger.getLoggerContext()));
        fileAppender.setName(FILE_APPENDER_NAME);
        fileAppender.setFile(fileName);
        fileAppender.setAppend(APPEND_TO_FILE);
        fileAppender.setPrudent(PRUDENT_VALUE);
        return fileAppender;
    }

    /**
     * Create a full initialized FileAppender with a {@link RollingPolicy} and {@link ch.qos.logback.core.rolling.TriggeringPolicy}
     * set based on the configuration.
     * @param fileCount maximum number of log files
     * @param fileSize maximum size of a given log file
     * @param fileName prefix for log file names
     * @param isDaily if the logs are to be rolled over daily
     * @return file appender to log to
     */
    private FileAppender<ILoggingEvent> createFileAppender(int fileCount, FileSize fileSize, String fileName, boolean isDaily) {
        if (fileCount <= 1) {
            FileAppender<ILoggingEvent> fileAppender = new FileAppender<ILoggingEvent>();
            fileAppender.setName(FILE_APPENDER_NAME);
            fileAppender.setFile(fileName);
            fileAppender.setAppend(APPEND_TO_FILE);
            fileAppender.setPrudent(PRUDENT_VALUE);
            fileAppender.setContext(logger.getLoggerContext());
            fileAppender.setEncoder(getEncoder(logger.getLoggerContext()));
            return fileAppender;
        } else {
            RollingFileAppender<ILoggingEvent> rollingFileAppender = initializeRollingFileAppender(fileName);

            RollingPolicy rollingPolicy = null;
            if (isDaily) {
                rollingPolicy = buildDailyRollingPolicy(rollingFileAppender, fileCount, fileSize, fileName);
            } else {
                rollingPolicy = buildFixedWindowRollingPolicy(rollingFileAppender, fileCount, fileName);
                if (fileSize.getSize() > 0) {
                    SizeBasedTriggeringPolicy<ILoggingEvent> triggeringPolicy = new SizeBasedTriggeringPolicy<ILoggingEvent>();
                    triggeringPolicy.setMaxFileSize(fileSize);
                    triggeringPolicy.start();
                    rollingFileAppender.setTriggeringPolicy(triggeringPolicy);
                }
            }

            rollingFileAppender.setRollingPolicy(rollingPolicy);
            rollingPolicy.start();
            return rollingFileAppender;
        }
    }

    private Encoder<ILoggingEvent> getEncoder(Context context) {
        CustomPatternLogbackEncoder encoder = new CustomPatternLogbackEncoder(CONVERSION_PATTERN);
        encoder.setContext(context);
        encoder.start();
        return encoder;
    }

    /**
     * Creates a logback logger.
     *
     * @param name        The name of the logger.
     * @param isAgentRoot True means this is the root logger for the agent.
     * @return The logger created.
     */
    public static LogbackLogger create(String name, boolean isAgentRoot) {
        return new LogbackLogger(name, isAgentRoot);
    }

    @Override
    public void log(Level level, String pattern, Object[] msg) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, msg));
        }
    }

    @Override
    public void log(Level level, String pattern, Object part1) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1));
        }
    }

    @Override
    public void log(Level level, String pattern, Object part1, Object part2) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2));
        }
    }

    @Override
    public void log(Level level, String pattern, Object part1, Object part2, Object part3) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3));
        }
    }

    @Override
    public void log(Level level, String pattern, Object part1, Object part2, Object part3, Object part4) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3, part4));
        }
    }

    @Override
    public void log(Level level, String pattern, Object part1, Object part2, Object part3, Object part4, Object part5) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3, part4, part5));
        }
    }

    @Override
    public void log(Level level, String pattern, Object part1, Object part2, Object part3, Object part4, Object part5,
            Object part6) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3, part4, part5, part6));
        }
    }

    @Override
    public void log(Level level, String pattern, Object part1, Object part2, Object part3, Object part4, Object part5,
            Object part6, Object part7) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3, part4, part5, part6, part7));
        }
    }

    @Override
    public void log(Level level, String pattern, Object part1, Object part2, Object part3, Object part4, Object part5,
                    Object part6, Object part7, Object... otherParts) {
        if (isLoggable(level)) {
            Object[] parts = merge(otherParts, part1, part2, part3, part4, part5, part6, part7);
            log(level, getMessage(pattern, parts));
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern, Object[] msg) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, msg), t);
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern), t);
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern, Object part1) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1), t);
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern, Object part1, Object part2) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2), t);
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern, Object part1, Object part2, Object part3) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3), t);
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern, Object part1, Object part2, Object part3, Object part4) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3, part4), t);
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern, Object part1, Object part2, Object part3, Object part4,
            Object part5) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3, part4, part5), t);
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern, Object part1, Object part2, Object part3, Object part4,
            Object part5, Object part6) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3, part4, part5, part6), t);
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern, Object part1, Object part2, Object part3, Object part4,
            Object part5, Object part6, Object part7) {
        if (isLoggable(level)) {
            log(level, getMessage(pattern, part1, part2, part3, part4, part5, part6, part7), t);
        }
    }

    @Override
    public void log(Level level, Throwable t, String pattern, Object part1, Object part2, Object part3, Object part4,
                    Object part5, Object part6, Object part7, Object... otherParts) {
        if (isLoggable(level)) {
            Object[] parts = merge(otherParts, part1, part2, part3, part4, part5, part6, part7);
            log(level, getMessage(pattern, parts), t);
        }
    }

    private String getMessage(String pattern, Object... parts) {
        return parts == null ? pattern : MessageFormat.format(pattern, formatValues(parts));
    }

    private Object[] formatValues(Object[] parts) {
        Object[] strings = new Object[parts.length];
        for (int i = 0; i < parts.length; i++) {
            strings[i] = formatValue(parts[i]);
        }
        return strings;
    }

    private Object formatValue(Object obj) {
        if (obj instanceof Class) {
            return ((Class<?>) obj).getName();
        } else if (obj instanceof Throwable) {
            return obj.toString();
        } else {
            return obj;
        }
    }

    private Object[] merge(Object[] otherParts, Object... firstParameters) {
        int otherPartsLength = otherParts != null ? otherParts.length : 0;
        Object[] mergedArray = new Object[firstParameters.length + otherPartsLength];

        System.arraycopy(firstParameters, 0, mergedArray, 0, firstParameters.length);
        if (otherPartsLength > 0) {
            System.arraycopy(otherParts, 0, mergedArray, firstParameters.length, otherPartsLength);
        }

        return mergedArray;
    }

    @Override
    public void logToChild(String childName, Level level, String pattern, Object part1, Object part2, Object part3,
            Object part4) {
        if (isLoggable(level)) {
            IAgentLogger logger = childLoggers.get(childName);
            if (logger == null) {
                logger = Agent.LOG;
            }
            logger.log(level, pattern, part1, part2, part3, part4);
        }

    }

}
