package com.newrelic.agent.logging;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Marker;

import ch.qos.logback.classic.Level;

enum LogbackLevel {

    /** The off level. */
    OFF("off", Level.OFF, java.util.logging.Level.OFF, null),
    /** The level all. */
    ALL("all", Level.ALL, java.util.logging.Level.ALL, null),
    /** The fatal level. */
    FATAL("fatal", Level.ERROR, java.util.logging.Level.SEVERE, null),
    /** The level severe. */
    SEVERE("severe", Level.ERROR, java.util.logging.Level.SEVERE, null),
    /** The error level. */
    ERROR("error", Level.ERROR, java.util.logging.Level.SEVERE, null),
    /** The warn level. */
    WARN("warn", Level.WARN, java.util.logging.Level.WARNING, null),
    /** The warning level. */
    WARNING("warning", Level.WARN, java.util.logging.Level.WARNING, null),
    /** The info level. */
    INFO("info", Level.INFO, java.util.logging.Level.INFO, null),
    /** The config level. */
    CONFIG("config", Level.INFO, java.util.logging.Level.CONFIG, null),
    /** The fine level. */
    FINE("fine", Level.DEBUG, java.util.logging.Level.FINE, LogbackMarkers.FINE_MARKER),
    /** The finer level. */
    FINER("finer", Level.DEBUG, java.util.logging.Level.FINER, LogbackMarkers.FINER_MARKER),
    /** The finest level. */
    FINEST("finest", Level.TRACE, java.util.logging.Level.FINEST, LogbackMarkers.FINEST_MARKER),
    /** The debug level. */
    DEBUG("debug", Level.DEBUG, java.util.logging.Level.FINE, null),
    /** The trace level. */
    TRACE("trace", Level.TRACE, java.util.logging.Level.FINEST, null);

    /**
     * Name of the level.
     */
    private final String name;
    /** The logback level: ERROR, WARN, INFO, DEBUG, or TRACE. */
    private final Level logbackLevel;
    /** The java util level. */
    private final java.util.logging.Level javaLevel;
    /** The marker for the level. */
    private final Marker marker;

    /**
     * Key is the string value of the level. Value is the associated logback level.
     */
    private static final Map<String, LogbackLevel> CONVERSION = new HashMap<String, LogbackLevel>();
    /**
     * Key is the java.util logging level. Value is the associated logback level.
     */
    private static final Map<java.util.logging.Level, LogbackLevel> JAVA_TO_LOGBACK = new HashMap<java.util.logging.Level, LogbackLevel>();

    static {
        LogbackLevel[] levels = LogbackLevel.values();
        for (LogbackLevel level : levels) {
            CONVERSION.put(level.name, level);
        }

        JAVA_TO_LOGBACK.put(java.util.logging.Level.ALL, ALL);
        JAVA_TO_LOGBACK.put(java.util.logging.Level.FINER, FINER);
        JAVA_TO_LOGBACK.put(java.util.logging.Level.FINEST, FINEST);
        JAVA_TO_LOGBACK.put(java.util.logging.Level.FINE, FINE);
        JAVA_TO_LOGBACK.put(java.util.logging.Level.WARNING, WARNING);
        JAVA_TO_LOGBACK.put(java.util.logging.Level.SEVERE, SEVERE);
        JAVA_TO_LOGBACK.put(java.util.logging.Level.CONFIG, CONFIG);
        JAVA_TO_LOGBACK.put(java.util.logging.Level.INFO, INFO);
        JAVA_TO_LOGBACK.put(java.util.logging.Level.OFF, OFF);
    }

    /**
     * 
     * Creates this LogbackLevel.
     * 
     * @param pName Name of the level.
     * @param pLogbackLevel Associated logback level.
     * @param pJavaLevel Associated java.util level.
     * @param pMarker Associated marker.
     */
    LogbackLevel(String pName, Level pLogbackLevel, java.util.logging.Level pJavaLevel, Marker pMarker) {
        name = pName;
        logbackLevel = pLogbackLevel;
        javaLevel = pJavaLevel;
        marker = pMarker;
    }

    /**
     * Gets the field marker.
     * 
     * @return the marker
     */
    public Marker getMarker() {
        return marker;
    }

    /**
     * Gets the field logbackLevel.
     * 
     * @return the logbackLevel
     */
    public Level getLogbackLevel() {
        return logbackLevel;
    }

    /**
     * Gets the field javaLevel.
     * 
     * @return the javaLevel
     */
    public java.util.logging.Level getJavaLevel() {
        return javaLevel;
    }

    /**
     * Returns the logback level associated with the input string.
     * 
     * @param pName The name of the level.
     * @param pDefault The level.
     * @return The associated logback level.
     */
    public static LogbackLevel getLevel(String pName, LogbackLevel pDefault) {
        LogbackLevel level = CONVERSION.get(pName);
        return ((level == null) ? pDefault : level);
    }

    /**
     * Returns the logback level associated with the input level.
     * 
     * @param pName The name of the level.
     * @return The associated logback level.
     */
    public static LogbackLevel getLevel(java.util.logging.Level pName) {
        return JAVA_TO_LOGBACK.get(pName);
    }
}
