package com.newrelic.agent.logging;

import java.util.logging.Level;

import org.slf4j.Marker;

import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.filter.Filter;
import ch.qos.logback.core.spi.FilterReply;

class FineFilter extends Filter<ILoggingEvent> {

    private static FineFilter instance;

    /** The current java level. */
    private volatile Level javaLevel;

    /** The marker to accept if it matches. */
    private final Marker markerToMatch = LogbackMarkers.FINE_MARKER;

    /** The marker to fail if it matches. */
    private final Marker markerToFail = LogbackMarkers.FINER_MARKER;

    public static FineFilter getFineFilter() {
        if (instance == null) {
            instance = new FineFilter();
        }
        return instance;
    }

    /**
     * 
     * Creates this FineFilter.
     * 
     */
    private FineFilter() {
        super();
        javaLevel = Level.INFO;
    }

    /**
     * Determines if the input event should be logged.
     * 
     * @param pEvent The current event to be evaluated.
     */
    @Override
    public FilterReply decide(ILoggingEvent pEvent) {
        if (!isStarted()) {
            return FilterReply.NEUTRAL;
        }

        if (Level.FINE.equals(javaLevel)) {
            Marker marker = pEvent.getMarker();
            if (marker == null) {
                return FilterReply.NEUTRAL;
            } else if (marker.contains(markerToMatch)) {
                return FilterReply.ACCEPT;
            } else if (marker.contains(markerToFail)) {
                return FilterReply.DENY;
            }
        }

        return FilterReply.NEUTRAL;
    }

    /**
     * True if the filter is enabled for the level.
     * 
     * @param pLevel The current level.
     * @return True if the filter is enabled for the level, else false.
     */
    public boolean isEnabledFor(Level pLevel) {
        return javaLevel.intValue() <= pLevel.intValue();
    }

    /**
     * Sets the level.
     * 
     * @param level The new level.
     */
    public void setLevel(Level level) {
        javaLevel = level;
    }

    /**
     * Gets the current level.
     * 
     * @return The current level for this filter.
     */
    public Level getLevel() {
        return javaLevel;
    }

    /**
     * Starts this filter.
     */
    @Override
    public void start() {
        if (javaLevel != null) {
            super.start();
        }
    }

}
