package com.newrelic.agent.instrumentation.pointcuts.container.glassfish;

import java.text.MessageFormat;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;

import com.newrelic.agent.Agent;
import com.newrelic.agent.Transaction;
import com.newrelic.agent.instrumentation.PointCutClassTransformer;
import com.newrelic.agent.instrumentation.PointCutConfiguration;
import com.newrelic.agent.instrumentation.TracerFactoryPointCut;
import com.newrelic.agent.instrumentation.classmatchers.ClassMatcher;
import com.newrelic.agent.instrumentation.classmatchers.ExactClassMatcher;
import com.newrelic.agent.instrumentation.methodmatchers.MethodMatcher;
import com.newrelic.agent.instrumentation.pointcuts.PointCut;
import com.newrelic.agent.jmx.values.GlassfishJmxValues;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.tracers.ClassMethodSignature;
import com.newrelic.agent.tracers.MethodExitTracer;
import com.newrelic.agent.tracers.Tracer;

@PointCut
public class Glassfish4StartUpPointCut extends TracerFactoryPointCut {

    private final AtomicBoolean addedJmx = new AtomicBoolean(false);

    public Glassfish4StartUpPointCut(PointCutClassTransformer classTransformer) {
        super(new PointCutConfiguration(Glassfish4StartUpPointCut.class.getName(),
                Glassfish3StartUpPointCut.GLASSFISH_INSTRUMENTATION_GROUP_NAME, true), createClassMatcher(),
                createMethodMatcher());
    }

    private static ClassMatcher createClassMatcher() {
        return new ExactClassMatcher("com/sun/appserv/server/util/Version");
    }

    private static MethodMatcher createMethodMatcher() {
        return createExactMethodMatcher("getMajorVersion", "()Ljava/lang/String;");
    }

    @Override
    public boolean isDispatcher() {
        return true;
    }

    @Override
    public Tracer doGetTracer(Transaction transaction, ClassMethodSignature sig, Object object, Object[] args) {
        if (!addedJmx.get()) {
            return new MethodExitTracer(sig, transaction) {

                @Override
                protected void doFinish(int opcode, Object returnValue) {
                    try {
                        if (returnValue instanceof String) {
                            String majorVersion = ((String) returnValue).trim();
                            if (majorVersion.length() > 0) {
                                addJMX(majorVersion);
                            }

                        }
                    } catch (Exception e) {
                        if (Agent.LOG.isFinestEnabled()) {
                            Agent.LOG.log(Level.FINER, "Glassfish Jmx error", e);
                        }
                    }
                }

            };
        } else {
            return null;
        }
    }

    private void addJMX(String majorVersion) {
        // GlassFish 2 and 3 are added with different point cuts
        if (!("1".equals(majorVersion) || "2".equals(majorVersion) || "3".equals(majorVersion))) {
            ServiceFactory.getJmxService().addJmxFrameworkValues(new GlassfishJmxValues());
            addedJmx.set(true);
            if (Agent.LOG.isFinerEnabled()) {
                Agent.LOG.log(Level.FINER, MessageFormat.format("Added JMX for Glassfish {0}", majorVersion));
            }
        }
    }

}
