package com.newrelic.agent.install;

import com.newrelic.agent.util.EditableFile;

public class JBossSelfInstaller extends SelfInstaller {

    // Define constants and regular expressions

    // JBoss 4.x, 5.x, 6.x
    private final String scriptPath = "/bin/run";

    // Script snippets to look for, or add
    private final String agentAlreadySet = "(.*)JAVA_OPTS=(.*)\\-javaagent:(.*)newrelic.jar";
    private final String windowsLocator = "set JBOSS_CLASSPATH=%RUN_CLASSPATH%";

    private String rootDir;

    /**
     * Adds text like this to run.conf:
     * 
     * # ---- New Relic switch automatically added to start command on Aug 13 2010, 20:06:55
     * JAVA_OPTS="$JAVA_OPTS -javaagent:/full/path/to/newrelic.jar"
     * 
     * ...or text like this to run.bat:
     * 
     * :: ---- New Relic switch automatically added to start command on Aug 13 2010, 20:06:55 set
     * JAVA_OPTS=-javaagent:"C:/full/path/to/newrelic.jar" %JAVA_OPTS%
     * 
     */
    @Override
    public boolean backupAndEditStartScript(String appServerRootDir) {

        rootDir = appServerRootDir;
        return backupAndEdit(appServerRootDir + getStartScript());

    }

    private boolean backupAndEdit(String fullPathToScript) {
        try {
            EditableFile file = new EditableFile(fullPathToScript);

            String fullSwitch = getCommentForAgentSwitch(file.comment) + lineSep + getAgentSettings();

            // Install the javaagent switch, if it is not already set.
            if (!file.contains(getAgentAlreadySetExpr())) {
                backup(file);
                if (osIsWindows) {
                    file.insertAfterLocator(getLocator(), lineSep + fullSwitch, true);
                } else {
                    file.append(fullSwitch + lineSep);
                }
                System.out.println("* Added agent switch to start script " + file.getLocation() + lineSep);

            } else {
                System.out.println("* Did not edit start script " + file.getLocation() + " because:");
                System.out.println("The agent switch is already set" + lineSep);
            }

            return true;

        } catch (Exception e) {
            System.out.println(e.getMessage());
            return false;
        }
    }

    @Override
    public String getStartScript() {
        String path = scriptPath;
        if (osIsWindows) {
            path = path.replaceAll("/", "\\\\");
        }
        return path + (osIsWindows ? DOTBAT : DOTCONF);
    }

    @Override
    public String getAlternateStartScript() {
        return getStartScript();
    }

    @Override
    public String getLocator() {
        return windowsLocator;
    }

    @Override
    public String getAlternateLocator() {
        return getLocator();
    }

    // TODO Batch scripts need parens escaped with hats
    @Override
    public String getAgentSettings() {
        String unixSwitch = "JAVA_OPTS=\"$JAVA_OPTS -javaagent:" + rootDir + "/newrelic/newrelic.jar\"" + lineSep;
        String windowsSwitch = "set JAVA_OPTS=-javaagent:\"" + rootDir.replaceAll("\\\\", "/")
                + "/newrelic/newrelic.jar\" %JAVA_OPTS%";

        return osIsWindows ? windowsSwitch : unixSwitch;
    }

    @Override
    public String getAgentAlreadySetExpr() {
        return agentAlreadySet;
    }

}
