package com.newrelic.agent.install;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.Date;

import com.newrelic.agent.Agent;
import com.newrelic.agent.config.ConfigFileHelper;
import com.newrelic.agent.util.Strings;
import com.newrelic.weave.utils.Streams;

public class ConfigInstaller {
    private final static String REPLACE_WITH_YOUR_LICENSE_KEY = "replace_with_your_license_key";
    private final static String LICENSE_KEY_CONFIG_PARAM = "<%= license_key %>";
    private final static String GENERATE_FOR_USER_CONFIG_PARAM = "<%= generated_for_user %>";

    public static boolean isConfigInstalled(File configDir) {
        if (configDir != null && configDir.exists() && configDir.isDirectory()) {
            return new File(configDir, ConfigFileHelper.NEW_RELIC_YAML_FILE).exists();
        }

        return false;
    }

    public static String configPath(File configDir) {
        return new File(configDir, ConfigFileHelper.NEW_RELIC_YAML_FILE).getAbsolutePath();
    }

    public static void install(String licenseKey, File configDir) throws Exception {
        generateConfig(licenseKey == null ? REPLACE_WITH_YOUR_LICENSE_KEY : licenseKey, configDir);
    }

    private static void generateConfig(String licenseKey, File configDir) throws Exception {
        InputStream inStream = ConfigInstaller.class.getClassLoader().getResourceAsStream("newrelic.yml");
        if (inStream != null) {
            String generatedFrom = getGeneratedFromString();
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            try {
                Streams.copy(inStream, output);

                String yaml = output.toString().replace(GENERATE_FOR_USER_CONFIG_PARAM, generatedFrom).replace(
                        LICENSE_KEY_CONFIG_PARAM, licenseKey);

                FileOutputStream fileOut = new FileOutputStream(configPath(configDir));
                try {
                    Streams.copy(new ByteArrayInputStream(yaml.getBytes()), fileOut, yaml.getBytes().length);
                } finally {
                    fileOut.close();
                }
            } finally {
                inStream.close();
            }
        } else {
            throw new IOException("Unable to open newrelic.yml template");
        }
    }

    private static String getGeneratedFromString() {
        return MessageFormat.format("Generated on {0}, from version {1}", new Date(), Agent.getVersion());
    }

    public static boolean isLicenseKeyEmpty(String license) {
        return license == null || license.equals(REPLACE_WITH_YOUR_LICENSE_KEY)
                || license.equals(LICENSE_KEY_CONFIG_PARAM) || Strings.isEmpty(license)
                || Strings.isEmpty(license.trim());
    }
}
