package com.newrelic.agent.extension.jaxb;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.List;
import java.util.Map;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.annotation.XmlValue;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

public class UnmarshallerFactory {

    static final Map<Class<?>, Unmarshaller<?>> cachedUnmarshallers = Maps.newConcurrentMap();

    /**
     * Returns an unmarshaller for a JAXB class. The unmarshaller will be cached for subsequent calls.
     */
    @SuppressWarnings("unchecked")
    public static <T extends Object> Unmarshaller<T> create(Class<T> clazz) throws UnmarshalException {

        Unmarshaller<?> cachedUnmarshaller = cachedUnmarshallers.get(clazz);
        if (cachedUnmarshaller != null) {
            return (Unmarshaller<T>) cachedUnmarshaller;
        }

        final Unmarshaller<T> unmarshaller = create(clazz,
                Maps.<Class<?>, Unmarshaller<?>> newHashMap(Unmarshaller.getDefaultUnmarshallers()));

        Unmarshaller<T> newUnmarshaller = new Unmarshaller<T>(clazz) {

            @Override
            public T unmarshall(Node node) throws UnmarshalException {
                // the top level marshaller will get a Document. We need to start with the root document element of that
                // document
                return unmarshaller.unmarshall(((Document) node).getDocumentElement());
            }
        };
        cachedUnmarshallers.put(clazz, newUnmarshaller);
        return newUnmarshaller;
    }

    /**
     * Creates a class unmarshaller using the {@link XmlType} and {@link XmlAttribute} annotations of the class to
     * understand how to unmarshal a document into it.
     */
    private static <T extends Object> Unmarshaller<T> create(final Class<T> clazz,
            final Map<Class<?>, Unmarshaller<?>> unmarshallers) throws UnmarshalException {

        try {
            final Setter attributesSetter = getAttributesSetter(clazz, unmarshallers);
            final Setter childrenSetter = getChildSetter(clazz, unmarshallers);

            Unmarshaller<T> classUnmarshaller = new Unmarshaller<T>(clazz) {

                @Override
                public T unmarshall(Node node) throws UnmarshalException {
                    try {
                        T newInstance = clazz.newInstance();

                        attributesSetter.set(newInstance, node);

                        childrenSetter.set(newInstance, node);

                        return newInstance;
                    } catch (InstantiationException e) {
                        throw new UnmarshalException(e);
                    } catch (IllegalAccessException e) {
                        throw new UnmarshalException(e);
                    }
                }
            };
            unmarshallers.put(clazz, classUnmarshaller);
            return classUnmarshaller;
        } catch (InstantiationException e) {
            throw new UnmarshalException(e);
        } catch (IllegalAccessException e) {
            throw new UnmarshalException(e);
        }
    }

    /**
     * Returns a setter that can process the child xml elements of a class. The fields that store child xml elements are
     * discovered using the {@link XmlType#propOrder()} elements of the {@link XmlType} annotation on the given class.
     */
    private static Setter getChildSetter(final Class<?> clazz, final Map<Class<?>, Unmarshaller<?>> unmarshallers)
            throws InstantiationException, IllegalAccessException, UnmarshalException {

        final Map<String, Setter> childSetters = Maps.newHashMap();
        XmlType type = clazz.getAnnotation(XmlType.class);
        if (type != null) {
            for (String t : type.propOrder()) {
                if (!t.isEmpty()) {
                    try {
                        Field field = clazz.getDeclaredField(t);
                        field.setAccessible(true);

                        Class<?> fieldType = field.getType();
                        final Unmarshaller<?> unmarshaller = getUnmarshaller(unmarshallers, fieldType, field);
                        XmlValue xmlValue = field.getAnnotation(XmlValue.class);
                        if (xmlValue != null) {
                            if (type.propOrder().length > 1) {
                                throw new UnmarshalException(
                                        clazz.getName()
                                                + " has an @XmlValue field so only one child type was expected, but multiple were found: "
                                                + type.propOrder());
                            }
                            return new ChildSetter(t, unmarshaller, field);
                        } else {
                            childSetters.put(t, new ChildSetter(t, unmarshaller, field));
                        }
                    } catch (Exception e) {
                        throw new UnmarshalException(e);
                    }
                }
            }
        }

        return new Setter() {

            @Override
            public void set(Object obj, Node node) throws IllegalArgumentException, IllegalAccessException,
                    InstantiationException, UnmarshalException {
                NodeList childNodes = node.getChildNodes();

                for (int i = 0; i < childNodes.getLength(); i++) {
                    Node item = childNodes.item(i);
                    if (item.getNodeType() != Node.COMMENT_NODE) {
                        String nodeName = item.getNodeName();
                        String prefix = item.getPrefix();
                        // strip off the prefix if necessary
                        if (prefix != null && !prefix.isEmpty()) {
                            nodeName = nodeName.substring(prefix.length() + 1);
                        }
                        Setter setter = childSetters.get(nodeName);
                        if (setter != null) {
                            setter.set(obj, item);
                        } else {
                            throw new UnmarshalException("No setter for node name " + nodeName + " on "
                                    + clazz.getName());
                        }
                    }
                }
            }

        };
    }

    /**
     * Returns a setter that processes all of the attributes of an xml node and populates the class instance. Attribute
     * fields are marked with the {@link XmlAttribute} annotation.
     */
    private static Setter getAttributesSetter(Class<?> clazz, Map<Class<?>, Unmarshaller<?>> unmarshallers)
            throws InstantiationException, IllegalAccessException, UnmarshalException {
        final List<Setter> attributeSetters = Lists.newArrayList();

        // attribute fields are marked with the @XmlAttribute annotation.
        // make a pass through to process those
        for (Field field : clazz.getDeclaredFields()) {
            XmlAttribute attribute = field.getAnnotation(XmlAttribute.class);
            if (attribute != null) {
                field.setAccessible(true);
                Class<?> declaringClass = field.getType();
                Unmarshaller<?> unmarshaller = getUnmarshaller(unmarshallers, declaringClass, field);
                attributeSetters.add(new AttributeSetter(attribute.name(), unmarshaller, field));
            }
        }
        return new Setter() {

            @Override
            public void set(Object obj, Node node) throws IllegalArgumentException, IllegalAccessException,
                    InstantiationException, UnmarshalException {
                if (node.getAttributes() != null) {
                    for (Setter setter : attributeSetters) {
                        setter.set(obj, node);
                    }
                }

            }

        };
    }

    /**
     * Gets a marshaller for the given class from the map of marshallers, creating it if it doesn't exist.
     */
    private static Unmarshaller<?> getUnmarshaller(Map<Class<?>, Unmarshaller<?>> unmarshallers, Class<?> clazz,
            Field field) throws InstantiationException, IllegalAccessException, UnmarshalException {

        // if this is a list, we need to create an unmarshaller for the generic type of the list
        if (clazz.isAssignableFrom(List.class)) {
            ParameterizedType parameterizedType = (ParameterizedType) field.getGenericType();
            clazz = (Class<?>) parameterizedType.getActualTypeArguments()[0];
        }

        Unmarshaller<?> unmarshaller = unmarshallers.get(clazz);
        if (unmarshaller == null) {
            unmarshaller = create(clazz, unmarshallers);
            unmarshallers.put(clazz, unmarshaller);
        }
        return unmarshaller;
    }

    private interface Setter {
        void set(Object obj, Node node) throws IllegalArgumentException, IllegalAccessException,
                InstantiationException, UnmarshalException;
    }

    /**
     * A setter that handles setting a field that stores a child xml node.
     * 
     * @author sdaubin
     * 
     */
    private static class ChildSetter implements Setter {

        private final Unmarshaller<?> unmarshaller;
        private final Field field;

        public ChildSetter(String t, Unmarshaller<?> unmarshaller, Field field) {
            this.unmarshaller = unmarshaller;
            this.field = field;
        }

        @Override
        public void set(Object obj, Node node) throws IllegalArgumentException, IllegalAccessException,
                InstantiationException, UnmarshalException {
            Object value = unmarshaller.unmarshall(node);

            if (value != null) {
                // handle Lists
                if (field.getType().isAssignableFrom(List.class)) {
                    @SuppressWarnings("unchecked")
                    List<Object> list = (List<Object>) field.get(obj);
                    if (list == null) {
                        list = Lists.newArrayList();
                        field.set(obj, list);
                    }
                    list.add(value);
                } else {
                    field.set(obj, value);
                }
            }
        }

        @Override
        public String toString() {
            return "ChildSetter [field=" + field + "]";
        }

    }

    /**
     * A setter that handles setting an attribute field.
     * 
     * @author sdaubin
     * 
     */
    private static class AttributeSetter implements Setter {

        private final String name;
        private final Unmarshaller<?> unmarshaller;
        private final Field field;

        public AttributeSetter(String name, Unmarshaller<?> unmarshaller, Field field) {
            super();
            this.name = name;
            this.unmarshaller = unmarshaller;
            this.field = field;
        }

        @Override
        public void set(Object obj, Node node) throws IllegalArgumentException, IllegalAccessException,
                InstantiationException, UnmarshalException {

            Node namedItem = node.getAttributes().getNamedItem(name);
            if (namedItem != null) {
                Object value = unmarshaller.unmarshall(namedItem);
                if (value != null) {
                    field.set(obj, value);
                }
            }
        }

        @Override
        public String toString() {
            return "AttributeSetter [name=" + name + "]";
        }

    }
}
