package com.newrelic.agent.database;

import java.sql.ResultSetMetaData;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.newrelic.agent.Agent;
import com.newrelic.agent.bridge.datastore.DatabaseVendor;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.util.Strings;

public class DefaultDatabaseStatementParser implements DatabaseStatementParser {

    private final static int PATTERN_SWITCHES = Pattern.CASE_INSENSITIVE | Pattern.DOTALL;
    private final static Pattern COMMENT_PATTERN = Pattern.compile("/\\*.*?\\*/", Pattern.DOTALL);
    private final static Pattern NR_HINT_PATTERN = Pattern.compile(
            "\\s*/\\*\\s*nrhint\\s*:\\s*([^\\*]*)\\s*\\*/\\s*([^\\s]*).*", Pattern.DOTALL);
    private final static Pattern VALID_METRIC_NAME_MATCHER = Pattern.compile("[a-zA-Z0-9.$_@]+");
    private final static Pattern FROM_MATCHER = Pattern.compile("\\s+from\\s+", PATTERN_SWITCHES);
    private final static Pattern SELECT_PATTERN = Pattern.compile("^\\s*select.*?\\sfrom[\\s\\[]+([^\\]\\s,)(;]*).*",
            PATTERN_SWITCHES);
    private final static Pattern EXEC_VAR_PATTERN = Pattern.compile(
            ".*(?:exec|execute)\\s+[^\\s(,]*.*?=(?:\\s|)([^\\s]*)", PATTERN_SWITCHES);

    private final Set<String> knownOperations;
    private final List<StatementFactory> statementFactories;
    private final boolean reportSqlParserErrors;
    private final StatementFactory selectStatementFactory = new DefaultStatementFactory(SELECT_OPERATION,
            SELECT_PATTERN, true);

    public DefaultDatabaseStatementParser(AgentConfig agentConfig) {
        super();

        this.reportSqlParserErrors = agentConfig.isReportSqlParserErrors();

        // the ordering of these factories is important
        statementFactories = Arrays.asList(
                new InnerSelectStatementFactory(),
                new DefaultStatementFactory("show", Pattern.compile("^\\s*show\\s+(.*)$", PATTERN_SWITCHES), false) {
                    @Override
                    protected boolean isValidModelName(String name) {
                        return true;
                    }
                },
                // @formatter:off
                new DefaultStatementFactory(INSERT_OPERATION, Pattern.compile("^\\s*insert(?:\\s+ignore)?(?:\\s+into)?\\s+([^\\s(,;]*).*", PATTERN_SWITCHES), true),
                new DefaultStatementFactory("update", Pattern.compile("^\\s*update\\s+([^\\s,;]*).*", PATTERN_SWITCHES), true),
                new DefaultStatementFactory("delete", Pattern.compile("^\\s*delete\\s*?.*?\\s+from\\s+([^\\s,(;]*).*", PATTERN_SWITCHES), true),
                new DDLStatementFactory("create", Pattern.compile("^\\s*create\\s+procedure.*", PATTERN_SWITCHES), "Procedure"),
                new SelectVariableStatementFactory(),
                new DDLStatementFactory("drop", Pattern.compile("^\\s*drop\\s+procedure.*", PATTERN_SWITCHES), "Procedure"),
                new DDLStatementFactory("create", Pattern.compile("^\\s*create\\s+table.*", PATTERN_SWITCHES), "Table"),
                new DDLStatementFactory("drop", Pattern.compile("^\\s*drop\\s+table.*", PATTERN_SWITCHES), "Table"),
                new DefaultStatementFactory("alter", Pattern.compile("^\\s*alter\\s+([^\\s]*).*", PATTERN_SWITCHES), false),
                new DefaultStatementFactory("call", Pattern.compile(".*call\\s+([^\\s(,]*).*", PATTERN_SWITCHES), true),
                new DefaultStatementFactory("exec", Pattern.compile(".*(?:exec|execute)\\s+(?!as\\s+)([^\\s(,=;]*+);?\\s*+(?:[^=]|$).*", PATTERN_SWITCHES), true, EXEC_VAR_PATTERN),
                new DefaultStatementFactory("set", Pattern.compile("^\\s*set\\s+(.*)\\s*(as|=).*", PATTERN_SWITCHES), false));
                // @formatter:on

        knownOperations = new HashSet<String>();
        for (StatementFactory factory : statementFactories) {
            knownOperations.add(factory.getOperation());
        }
    }

    @Override
    public ParsedDatabaseStatement getParsedDatabaseStatement(
            DatabaseVendor databaseVendor, String statement,
            ResultSetMetaData metaData) {
        Matcher hintMatcher = NR_HINT_PATTERN.matcher(statement);
        if (hintMatcher.matches()) {
            String model = hintMatcher.group(1).trim().toLowerCase();
            String operation = hintMatcher.group(2).toLowerCase();
            if (!knownOperations.contains(operation)) {
                operation = "unknown";
            }

            return new ParsedDatabaseStatement(model, operation, true);
        }
        if (metaData != null) {
            try {
                int columnCount = metaData.getColumnCount();
                if (columnCount > 0) {
                    String tableName = metaData.getTableName(1);
                    if (!Strings.isEmpty(tableName)) {
                        return new ParsedDatabaseStatement(tableName.toLowerCase(), SELECT_OPERATION, true);
                    }
                }
            } catch (Exception e) {
            }
        }
        return parseStatement(statement);
    }

    ParsedDatabaseStatement parseStatement(String statement) {
        try {
            statement = COMMENT_PATTERN.matcher(statement).replaceAll("");
            for (StatementFactory factory : statementFactories) {
                ParsedDatabaseStatement parsedStatement = factory.parseStatement(statement);
                if (parsedStatement != null) {
                    return parsedStatement;
                }
            }
            Agent.LOG.log(Level.FINE, "Returning UNPARSEABLE_STATEMENT for statement: {0}", statement);
            return UNPARSEABLE_STATEMENT;
        } catch (Throwable t) {
            Agent.LOG.fine(MessageFormat.format("Unable to parse sql \"{0}\" - {1}", statement, t.toString()));
            Agent.LOG.log(Level.FINER, "SQL parsing error", t);
            Agent.LOG.log(Level.FINE, t, "Returning UNPARSEABLE_STATEMENT for statement: {0}", statement);
            return UNPARSEABLE_STATEMENT;
        }
    }

    private interface StatementFactory {
        String getOperation();
        ParsedDatabaseStatement parseStatement(String statement);
    }

    class DefaultStatementFactory implements StatementFactory {
        private final Pattern pattern;
        private final DefaultStatementFactory backupPattern;
        protected final String key;
        private final boolean generateMetric;

        public DefaultStatementFactory(String key, Pattern pattern, boolean generateMetric) {
            this.key = key;
            this.pattern = pattern;
            this.generateMetric = generateMetric;
            this.backupPattern = null;
        }

        public DefaultStatementFactory(String key, Pattern pattern, boolean generateMetric, Pattern backupPattern) {
            this.key = key;
            this.pattern = pattern;
            this.generateMetric = generateMetric;
            this.backupPattern = new DefaultStatementFactory(key, backupPattern, generateMetric);
        }

        protected boolean isMetricGenerator() {
            return generateMetric;
        }

        @Override
        public ParsedDatabaseStatement parseStatement(String statement) {
            // Optimization to prevent running complex regex when we don't need to
            if (!Strings.containsIgnoreCase(statement, key)) {
                return null;
            }

            Matcher matcher = pattern.matcher(statement);
            if (matcher.find()) {
                String model = matcher.groupCount() > 0 ? matcher.group(1).trim() : "unknown";
                if (model.length() == 0) {
                    Agent.LOG.log(Level.FINE, MessageFormat.format(
                            "Parsed an empty model name for {0} statement : {1}", key, statement));
                    return null;
                }
                model = Strings.unquoteDatabaseName(model);
                // remove brackets from metric name because they are reserved for units suffix
                model = Strings.removeBrackets(model);
                // if we aren't generating a metric, don't bother to validate the model name
                if (generateMetric && !isValidModelName(model)) {
                    if (reportSqlParserErrors) {
                        Agent.LOG.log(Level.FINE, MessageFormat.format(
                                "Parsed an invalid model name {0} for {1} statement : {2}", model, key, statement));
                    }

                    model = "ParseError";
                }
                return createParsedDatabaseStatement(model);
            }
            if (backupPattern != null) {
                return backupPattern.parseStatement(statement);
            }
            return null;
        }

        protected boolean isValidModelName(String name) {
            return isValidName(name);
        }

        ParsedDatabaseStatement createParsedDatabaseStatement(String model) {
            return new ParsedDatabaseStatement(model.toLowerCase(), key, generateMetric);
        }

        @Override
        public String getOperation() {
            return key;
        }
    }

    static boolean isValidName(String string) {
        return VALID_METRIC_NAME_MATCHER.matcher(string).matches();
    }

    private class SelectVariableStatementFactory implements StatementFactory {
        private final ParsedDatabaseStatement innerSelectStatement = new ParsedDatabaseStatement("INNER_SELECT",
                SELECT_OPERATION, false);
        private final ParsedDatabaseStatement statement = new ParsedDatabaseStatement("VARIABLE", SELECT_OPERATION,
                false);
        // REVIEW not sure about this matcher
        private final Pattern pattern = Pattern.compile(".*select\\s+([^\\s,]*).*", PATTERN_SWITCHES);

        @Override
        public ParsedDatabaseStatement parseStatement(String statement) {
            Matcher matcher = pattern.matcher(statement);
            if (matcher.matches()) {
                if (FROM_MATCHER.matcher(statement).find()) {
                    return innerSelectStatement;
                } else {
                    return this.statement;
                }
            } else {
                return null;
            }
        }

        @Override
        public String getOperation() {
            return "select";
        }
    }

    private class InnerSelectStatementFactory implements StatementFactory {
        private final Pattern innerSelectPattern = Pattern.compile("^\\s*SELECT.*?\\sFROM\\s*\\(\\s*(SELECT.*)",
                PATTERN_SWITCHES);

        @Override
        public ParsedDatabaseStatement parseStatement(String statement) {
            String sql = statement;
            String res = null;
            while (true) {
                String res2 = findMatch(sql);
                if (res2 == null) {
                    break;
                }
                res = res2;
                sql = res2;
            }

            if (res != null) {
                return selectStatementFactory.parseStatement(res);
            }
            return selectStatementFactory.parseStatement(statement);
        }

        private String findMatch(String statement) {
            Matcher matcher = innerSelectPattern.matcher(statement);
            if (matcher.matches()) {
                return matcher.group(1);
            }
            return null;
        }

        @Override
        public String getOperation() {
            return "select";
        }
    }

    private class DDLStatementFactory extends DefaultStatementFactory {
        private final String type;

        public DDLStatementFactory(String key, Pattern pattern, String type) {
            super(key, pattern, false);
            this.type = type;
        }

        @Override
        ParsedDatabaseStatement createParsedDatabaseStatement(String model) {
            return new ParsedDatabaseStatement(type, key, isMetricGenerator());
        }
    }

}
