package com.newrelic.agent.config;

import static com.newrelic.agent.Agent.LOG;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.objectweb.asm.Type;

import com.google.common.collect.Sets;
import com.newrelic.agent.Agent;
import com.newrelic.agent.instrumentation.annotationmatchers.AnnotationMatcher;
import com.newrelic.agent.instrumentation.annotationmatchers.AnnotationNames;
import com.newrelic.agent.instrumentation.annotationmatchers.ClassNameAnnotationMatcher;
import com.newrelic.agent.instrumentation.annotationmatchers.NoMatchAnnotationMatcher;
import com.newrelic.agent.instrumentation.annotationmatchers.OrAnnotationMatcher;
import com.newrelic.weave.weavepackage.WeavePackageConfig;

final class ClassTransformerConfigImpl extends BaseConfig implements ClassTransformerConfig {

    public static final String ENABLED = "enabled";
    public static final String EXCLUDES = "excludes";
    public static final String INCLUDES = "includes";
    public static final String CLASSLOADER_BLACKLIST = "classloader_blacklist";
    public static final String CLASSLOADER_DELEGATION_EXCLUDES = "classloader_delegation_excludes";
    public static final String CLASSLOADER_EXCLUDES = "classloader_excludes";
    public static final String MAX_PREVALIDATED_CLASSLOADERS = "max_prevalidated_classloaders";
    public static final String PREVALIDATE_WEAVE_PACKAGES = "prevalidate_weave_packages";
    public static final String PREMATCH_WEAVE_METHODS = "prematch_weave_methods";

    public static final String DEFAULT_INSTRUMENTATION = "instrumentation_default";
    public static final String BUILTIN_EXTENSIONS = "builtin_extensions";

    public static final String COMPUTE_FRAMES = "compute_frames";
    public static final String SHUTDOWN_DELAY = "shutdown_delay";
    public static final String GRANT_PACKAGE_ACCESS = "grant_package_access";

    public static final boolean DEFAULT_COMPUTE_FRAMES = true;
    public static final boolean DEFAULT_ENABLED = true;
    public static final boolean DEFAULT_DISABLED = false;
    public static final int DEFAULT_SHUTDOWN_DELAY = -1;
    public static final boolean DEFAULT_GRANT_PACKAGE_ACCESS = false;
    public static final int DEFAULT_MAX_PREVALIDATED_CLASSLOADERS = 10;
    public static final boolean DEFAULT_PREVALIDATE_WEAVE_PACKAGES = true;
    public static final boolean DEFAULT_PREMATCH_WEAVE_METHODS = true;

    private static final String SYSTEM_PROPERTY_ROOT = "newrelic.config.class_transformer.";

    static final String NEW_RELIC_TRACE_TYPE_DESC = "Lcom/newrelic/api/agent/Trace;";
    static final String DEPRECATED_NEW_RELIC_TRACE_TYPE_DESC = "Lcom/newrelic/agent/Trace;";

    private final boolean isEnabled;
    private final boolean custom_tracing;
    private final Set<String> excludes;
    private final Set<String> includes;
    private final Set<String> classloaderBlacklist;
    private final Set<String> classloaderDelegationExcludes;
    private final boolean computeFrames;
    private final boolean isDefaultInstrumentationEnabled;
    private final long shutdownDelayInNanos;
    private final boolean grantPackageAccess;
    private final int maxPreValidatedClassLoaders;
    private final boolean preValidateWeavePackages;
    private final boolean preMatchWeaveMethods;

    private final AnnotationMatcher ignoreTransactionAnnotationMatcher;
    private final AnnotationMatcher ignoreApdexAnnotationMatcher;
    private final AnnotationMatcher traceAnnotationMatcher;
    private final boolean defaultMethodTracingEnabled;
    private final boolean isBuiltinExtensionEnabled;
    private final boolean litemode;

    public ClassTransformerConfigImpl(Map<String, Object> props, boolean customTracingEnabled, boolean litemode) {
        super(props, SYSTEM_PROPERTY_ROOT);
        this.custom_tracing = customTracingEnabled;
        this.litemode = litemode;
        isEnabled = getProperty(ENABLED, DEFAULT_ENABLED);
        isDefaultInstrumentationEnabled = getDefaultInstrumentationEnabled();
        isBuiltinExtensionEnabled = getBuiltinExensionEnabled();
        excludes = Collections.unmodifiableSet(new HashSet<String>(getUniqueStrings(EXCLUDES)));
        includes = Collections.unmodifiableSet(new HashSet<String>(getUniqueStrings(INCLUDES)));
        classloaderBlacklist = initializeClassloaderBlacklist();
        classloaderDelegationExcludes = initializeClassloaderDelegationExcludes();
        computeFrames = getProperty(COMPUTE_FRAMES, DEFAULT_COMPUTE_FRAMES);
        shutdownDelayInNanos = initShutdownDelay();
        grantPackageAccess = getProperty(GRANT_PACKAGE_ACCESS, DEFAULT_GRANT_PACKAGE_ACCESS);
        maxPreValidatedClassLoaders = getProperty(MAX_PREVALIDATED_CLASSLOADERS, DEFAULT_MAX_PREVALIDATED_CLASSLOADERS);
        preValidateWeavePackages = getProperty(PREVALIDATE_WEAVE_PACKAGES, DEFAULT_PREVALIDATE_WEAVE_PACKAGES);
        preMatchWeaveMethods = getProperty(PREMATCH_WEAVE_METHODS, DEFAULT_PREMATCH_WEAVE_METHODS);
        defaultMethodTracingEnabled = getProperty("default_method_tracing_enabled", true);

        this.traceAnnotationMatcher = customTracingEnabled ? initializeTraceAnnotationMatcher(props)
                : new NoMatchAnnotationMatcher();
        this.ignoreTransactionAnnotationMatcher = new ClassNameAnnotationMatcher(
                AnnotationNames.NEW_RELIC_IGNORE_TRANSACTION, false);

        this.ignoreApdexAnnotationMatcher = new ClassNameAnnotationMatcher(AnnotationNames.NEW_RELIC_IGNORE_APDEX,
                false);

    }
    public ClassTransformerConfigImpl(Map<String, Object> props, boolean customTracingEnabled) {
        this(props, customTracingEnabled, false);
    }

    private boolean getBuiltinExensionEnabled() {
        Boolean builtinExtensionEnabled = getInstrumentationConfig(BUILTIN_EXTENSIONS).getProperty(ENABLED);
        if (!isDefaultInstrumentationEnabled && (builtinExtensionEnabled == null || !builtinExtensionEnabled)) {
            return DEFAULT_DISABLED;
        } else {
            return DEFAULT_ENABLED;
        }
    }

    private boolean getDefaultInstrumentationEnabled() {
        Boolean defaultInstrumentationEnabled = getInstrumentationConfig(DEFAULT_INSTRUMENTATION).getProperty(ENABLED);
        if (litemode) {
            return false;
        }
        if (defaultInstrumentationEnabled == null || defaultInstrumentationEnabled) {
            return DEFAULT_ENABLED;
        } else {
            Agent.LOG.info("Instrumentation is disabled by default");
            return DEFAULT_DISABLED;
        }
    }

    private Set<String> initializeClassloaderBlacklist() {
        // We relased an undocumented property called classloader_blacklist, and renamed it to classloader_excludes
        // before making it public in 3.30.0.
        Set<String> classloadersToExclude = new HashSet<String>(getUniqueStrings(CLASSLOADER_BLACKLIST));
        classloadersToExclude.addAll(getUniqueStrings(CLASSLOADER_EXCLUDES));
        return  Collections.unmodifiableSet(classloadersToExclude);
    }

    private Set<String> initializeClassloaderDelegationExcludes() {
        Set<String> classloadersToExclude = new HashSet<String>(getUniqueStrings(CLASSLOADER_DELEGATION_EXCLUDES));
        return Collections.unmodifiableSet(classloadersToExclude);
    }

    private AnnotationMatcher initializeTraceAnnotationMatcher(Map<?, ?> props) {
        List<AnnotationMatcher> matchers = new ArrayList<AnnotationMatcher>();
        matchers.add(new ClassNameAnnotationMatcher(Type.getType(DEPRECATED_NEW_RELIC_TRACE_TYPE_DESC).getDescriptor()));
        matchers.add(new ClassNameAnnotationMatcher(Type.getType(NEW_RELIC_TRACE_TYPE_DESC).getDescriptor()));
        
        final Collection<String> traceAnnotationClassNames = getUniqueStrings("trace_annotation_class_name");
        if (traceAnnotationClassNames.isEmpty()) {
            matchers.add(new ClassNameAnnotationMatcher("NewRelicTrace", false));
        } else {
            final Set<String> internalizedNames = Sets.newHashSet();
            for (String name : traceAnnotationClassNames) {
                Agent.LOG.fine("Adding " + name + " as a Trace annotation");
                internalizedNames.add(internalizeName(name));
            }
            matchers.add(new AnnotationMatcher() {

                @Override
                public boolean matches(String annotationDesc) {
                    return internalizedNames.contains(annotationDesc);
                }

            });
        }
        return OrAnnotationMatcher.getOrMatcher(matchers.toArray(new AnnotationMatcher[0]));
    }

    static String internalizeName(String name) {
        return 'L' + name.trim().replace('.', '/') + ';';
    }

    private long initShutdownDelay() {
        int shutdownDelayInSeconds = getIntProperty(SHUTDOWN_DELAY, DEFAULT_SHUTDOWN_DELAY);
        if (shutdownDelayInSeconds > 0) {
            return TimeUnit.NANOSECONDS.convert(shutdownDelayInSeconds, TimeUnit.SECONDS);
        }
        return -1L;
    }

    @Override
    public boolean isEnabled() {
        return isEnabled;
    }

    @Override
    public boolean isCustomTracingEnabled() {
        return custom_tracing;
    }

    @Override
    public Set<String> getIncludes() {
        return includes;
    }

    @Override
    public Set<String> getClassloaderBlacklist() {
        return classloaderBlacklist;
    }

    @Override
    public Set<String> getClassloaderDelegationExcludes() {
        return classloaderDelegationExcludes;
    }

    @Override
    public boolean isDefaultInstrumentationEnabled() {
        return isDefaultInstrumentationEnabled;
    }

    @Override
    public boolean isBuiltinExtensionEnabled() {
        return isBuiltinExtensionEnabled;
    }

    @Override
    public Set<String> getExcludes() {
        return excludes;
    }

    @Override
    public boolean computeFrames() {
        return computeFrames;
    }

    @Override
    public boolean isGrantPackageAccess() {
        return grantPackageAccess;
    }

    @Override
    public long getShutdownDelayInNanos() {
        return shutdownDelayInNanos;
    }

    @Override
    public final AnnotationMatcher getIgnoreTransactionAnnotationMatcher() {
        return ignoreTransactionAnnotationMatcher;
    }

    @Override
    public final AnnotationMatcher getIgnoreApdexAnnotationMatcher() {
        return ignoreApdexAnnotationMatcher;
    }

    @Override
    public AnnotationMatcher getTraceAnnotationMatcher() {
        return traceAnnotationMatcher;
    }

    @Override
    public int getMaxPreValidatedClassLoaders() {
        return maxPreValidatedClassLoaders;
    }

    @Override
    public boolean preValidateWeavePackages() {
        return preValidateWeavePackages;
    }

    public boolean preMatchWeaveMethods() {
        return preMatchWeaveMethods;
    }

    public static final String JDBC_STATEMENTS_PROPERTY = "jdbc_statements";

    @Override
    public Collection<String> getJdbcStatements() {
        String jdbcStatementsProp = getProperty(JDBC_STATEMENTS_PROPERTY);
        if (jdbcStatementsProp == null) {
            return Collections.emptyList();
        }
        // FIXME support yaml lists in addition to comma separated
        return Arrays.asList(jdbcStatementsProp.split(",[\\s]*"));
    }

    static ClassTransformerConfig createClassTransformerConfig(Map<String, Object> settings, boolean custom_tracing, boolean litemode) {
        if (settings == null) {
            settings = Collections.emptyMap();
        }

        return new ClassTransformerConfigImpl(settings, custom_tracing, litemode);
    }

    @Override
    @SuppressWarnings("unchecked")
    public Config getInstrumentationConfig(String implementationTitle) {
        Map<String, Object> config = Collections.emptyMap();
        if (implementationTitle != null) {
            Object pointCutConfig = getProperty(implementationTitle);
            if (pointCutConfig instanceof Map) {
                config = (Map<String, Object>) pointCutConfig; // SuppressWarnings applied here
            }
        }
        return new BaseConfig(config, SYSTEM_PROPERTY_ROOT + implementationTitle + ".");
    }

    @Override
    public boolean isWeavePackageEnabled(WeavePackageConfig weavePackageConfig) {

        // user can override with traditional instrumentation module name
        String moduleName = weavePackageConfig.getName();
        Config instrumentationConfig = getInstrumentationConfig(moduleName);
        Boolean moduleNameEnabled = null;

        if (instrumentationConfig != null) {
            moduleNameEnabled = instrumentationConfig.getProperty("enabled");
        }

        // ..or, an alias for backwards compatibility with legacy pointcut configurations, if available
        String aliasName = weavePackageConfig.getAlias();
        Boolean aliasEnabled = null;
        if (aliasName != null) {
            Config aliasConfig = getInstrumentationConfig(aliasName);
            if (aliasConfig != null) {
                LOG.log(Level.INFO, "Using deprecated configuration setting {0} for instrumentation {1}", aliasName,
                        moduleName);
                aliasEnabled = aliasConfig.getProperty("enabled");
            }
        }

        LOG.log(Level.FINEST, " ### Considering instrumentation: {0}({1}) enabled?{2}({3})",
                moduleName, aliasName, moduleNameEnabled, aliasEnabled);

        if (moduleNameEnabled == null && aliasEnabled == null && !isDefaultInstrumentationEnabled) {
            LOG.log(Level.FINEST, " Instrumentation is disabled by default. Skipping: {0} because it is not explicitly enabled.",
                moduleName);
            return false; // no configuration and instrumentation_disabled is true
        } else if(moduleNameEnabled == null && aliasEnabled == null && isDefaultInstrumentationEnabled) {
            return weavePackageConfig.isEnabled(); // no configuration, return default from weave package
        } else if(moduleNameEnabled == null) {
            return aliasEnabled; // only alias was configured
        } else if(aliasEnabled == null) {
            return moduleNameEnabled; // only module name was configured
        } else {
            return aliasEnabled && moduleNameEnabled; // both were configured
        }
    }

    @Override
    public boolean isDefaultMethodTracingEnabled() {
        return defaultMethodTracingEnabled;
    }
}
