package com.newrelic.agent.config;

import com.newrelic.agent.Agent;
import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

public class BaseConfig implements Config {

    public static final String COMMA_SEPARATOR = ",";
    public static final String SEMI_COLON_SEPARATOR = ";";

    private final Map<String, Object> props;
    protected final String systemPropertyPrefix;

    /**
     * Construct a BaseConfig for the map passed as an argument. Values in the map are not subject to override by either
     * system properties or environment variables.
     *
     * @param props the collection from which the values of keys are taken.
     */
    public BaseConfig(Map<String, Object> props) {
        this(props, null);
    }

    /**
     * Construct a BaseConfig.
     *
     * @param props the collection from which the values of keys are taken, subject to override behaviors.
     * @param systemPropertyPrefix If specified as null, values from the collection will not be subject to override by
     *        values from the system properties or environment. If non-null, the value serves as the prefix for matching
     *        overrides by the system properties and environment, with the environment having priority. Passing an empty
     *        string would expose the keys in the collection to override by arbitrary keys the user may have present in
     *        the the environment or system properties, so is not allowed.
     * @throws IllegalArgumentException - 0-length string given as systemPropertyPrefix
     */
    public BaseConfig(Map<String, Object> props, String systemPropertyPrefix) {
        if (systemPropertyPrefix != null && systemPropertyPrefix.length() == 0) {
            throw new IllegalArgumentException("prefix must be null or non-empty");
        }

        this.props = props == null ? Collections.<String, Object>emptyMap() : Collections.unmodifiableMap(props);
        this.systemPropertyPrefix = systemPropertyPrefix;
    }

    public Map<String, Object> getProperties() {
        return props;
    }

    protected Map<String, Object> createMap() {
        return new HashMap<String, Object>();
    }

    @SuppressWarnings("unchecked")
    protected Map<String, Object> nestedProps(String key) {
        Object value = getProperties().get(key);
        if (value == null) {
            return null;
        }
        if (value instanceof ServerProp) {
            value = ((ServerProp) value).getValue();
        }
        if (Map.class.isInstance(value)) {
            return (Map<String, Object>) value;
        }
        String msg = MessageFormat.format(
                "Agent configuration expected nested configuration values for \"{0}\", got \"{1}\"", key, value);
        Agent.LOG.warning(msg);
        return null;
    }

    protected Object getPropertyFromSystemProperties(String name, Object defaultVal) {
        if (systemPropertyPrefix == null) {
            return null;
        }

        String key = getSystemPropertyKey(name);
        String result = SystemPropertyFactory.getSystemPropertyProvider().getSystemProperty(key);
        return parseValue(result);
    }

    protected String getSystemPropertyKey(String key) {
        if (key == null) {
            throw new IllegalArgumentException("key");
        }
        return systemPropertyPrefix + key;
    }

    protected Object getPropertyFromSystemEnvironment(String name, Object defaultVal) {
        if (systemPropertyPrefix == null) {
            return null;
        }

        String key = getSystemPropertyKey(name);
        String result = SystemPropertyFactory.getSystemPropertyProvider().getEnvironmentVariable(key);
        return parseValue(result);
    }

    public static Object parseValue(String val) {
        if (val == null) {
            return val;
        }
        try {
            return new JSONParser().parse(val);
        } catch (ParseException e) {
            return val.toString();
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> T getProperty(String key, T defaultVal) {
        Object propVal = getProperties().get(key);
        if (propVal instanceof ServerProp) {
            propVal = ((ServerProp) propVal).getValue();
            return castValue(key, propVal, defaultVal);
        }
        Object override = getPropertyFromSystemEnvironment(key, defaultVal);
        if (override != null) {
            return castValue(key, override, defaultVal);
        }
        override = getPropertyFromSystemProperties(key, defaultVal);
        if (override != null) {
            return castValue(key, override, defaultVal);
        }
        return castValue(key, propVal, defaultVal);
    }

    @SuppressWarnings("unchecked")
    protected <T> T castValue(String key, Object value, T defaultVal) {
        try {
            if (defaultVal instanceof Integer && value instanceof Long) {
                return (T) Integer.valueOf(((Long) value).intValue());
            }

            T val = (T) value;
            if (val == null) {
                return defaultVal;
            } else if (val instanceof String) {
                return (T) ((String) val).trim();

            } else {
                return val;
            }
        } catch (ClassCastException e) {
            Agent.LOG.log(Level.FINE, e, "Unable to parse {0}", key);
            return defaultVal;
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> T getProperty(String key) {
        return (T) getProperty(key, null);
    }

    protected Set<Integer> getIntegerSet(String key, Set<Integer> defaultVal) {
        Object val = getProperty(key);
        if (val instanceof String) {
            return Collections.unmodifiableSet(getIntegerSetFromString((String) val));
        }
        if (val instanceof Collection<?>) {
            return Collections.unmodifiableSet(getIntegerSetFromCollection((Collection<?>) val));
        }
        if (val instanceof Integer) {
            return Collections.unmodifiableSet(getIntegerSetFromCollection(Arrays.asList((Integer) val)));
        }
        return defaultVal;
    }

    protected Set<Map<String, Object>> getMapSet(String key) {
        Object val = getProperty(key);
        if (val instanceof Collection<?>) {
            return Collections.unmodifiableSet(getMapSetFromCollection((Collection<?>) val));
        }
        return Collections.emptySet();
    }

    @SuppressWarnings("unchecked")
    protected Set<Map<String, Object>> getMapSetFromCollection(Collection<?> values) {
        Set<Map<String, Object>> result = new HashSet<Map<String, Object>>(values.size());
        for (Object value : values) {
            result.add((Map<String, Object>) value);
        }
        return result;
    }

    protected String getFirstString(String key, String separator) {
        Object val = getProperty(key);
        if (val instanceof String) {
            String[] values = ((String) val).split(separator);
            if (values.length == 0) {
                return null;
            }
            String res = values[0].trim();
            if (res.length() == 0) {
                return null;
            }
            return res;
        }
        if (val instanceof Collection<?>) {
            Collection<?> values = (Collection<?>) val;
            for (Object value : values) {
                String res = (String) value;
                res = res.trim();
                if (res.length() != 0) {
                    return res;
                }
                return null;
            }
        }
        return null;
    }

    /**
     * Returns a collection of strings for the given key using comma as a separator character.
     *
     * @see BaseConfig#getUniqueStrings(String, String)
     */
    protected Collection<String> getUniqueStrings(String key) {
        return getUniqueStrings(key, COMMA_SEPARATOR);
    }

    /**
     * Returns a collection of strings for the given key.  The property value can be a collection or
     * a String list that uses a separator character.
     */
    protected Collection<String> getUniqueStrings(String key, String separator) {
        Object val = getProperty(key);
        if (val instanceof String) {
            return Collections.unmodifiableList(getUniqueStringsFromString((String) val, separator));
        }
        if (val instanceof Collection<?>) {
            return Collections.unmodifiableList(getUniqueStringsFromCollection((Collection<?>) val));
        }
        return Collections.emptySet();
    }

    public static List<String> getUniqueStringsFromCollection(Collection<?> values, String prefix) {
        List<String> result = new ArrayList<String>(values.size());
        boolean noPrefix = (prefix == null || prefix.isEmpty());
        for (Object value : values) {
            String val = null;
            if (value instanceof Integer) {
                val = String.valueOf(value);
            } else if (value instanceof Long) {
                val = String.valueOf(value);
            } else {
                val = (String) value;
            }
            val = val.trim();
            if (val.length() != 0 && !result.contains(val)) {
                if (noPrefix) {
                    result.add(val);
                } else {
                    result.add(prefix + val);
                }
            }
        }
        return result;
    }

    public static List<String> getUniqueStringsFromCollection(Collection<?> values) {
        return getUniqueStringsFromCollection(values, null);
    }

    public static List<String> getUniqueStringsFromString(String valuesString, String separator, String prefix) {
        String[] valuesArray = valuesString.split(separator);
        List<String> result = new ArrayList<String>(valuesArray.length);
        boolean noPrefix = (prefix == null || prefix.isEmpty());
        for (String value : valuesArray) {
            value = value.trim();
            if (value.length() != 0 && !result.contains(value)) {
                if (noPrefix) {
                    result.add(value);
                } else {
                    result.add(prefix + value);
                }
            }
        }
        return result;
    }

    /**
     * The ordering of many of the configuration lists is important. Don't use hashsets to get unique sets because it
     * will destroy the ordering.
     *
     * @param valuesString
     * @param separator
     * @return
     */
    public static List<String> getUniqueStringsFromString(String valuesString, String separator) {
        return getUniqueStringsFromString(valuesString, separator, null);
    }

    protected int getIntProperty(String key, int defaultVal) {
        Number val = getProperty(key);
        if (val == null) {
            return defaultVal;
        }
        return val.intValue();
    }

    protected String getStringPropertyOrNull(String key) {
        Object val = getProperty(key);
        if (val == null) {
            String msg = MessageFormat.format("Value for \"{0}\" is null", key);
            Agent.LOG.fine(msg);
            return null;
        }
        return val.toString();
    }

    protected double getDoubleProperty(String key, double defaultVal) {
        Number val = getProperty(key);
        if (val == null) {
            return defaultVal;
        }
        return val.doubleValue();
    }

    private Set<Integer> getIntegerSetFromCollection(Collection<?> values) {
        Set<Integer> result = new HashSet<Integer>(values.size());
        for (Object value : values) {
            int val = ((Number) value).intValue();
            result.add(val);
        }
        return result;
    }

    private Set<Integer> getIntegerSetFromString(String valuesString) {
        String[] valuesArray = valuesString.split(COMMA_SEPARATOR);
        Set<Integer> result = new HashSet<Integer>(valuesArray.length);
        for (String value : valuesArray) {
            value = value.trim();
            if (value.length() != 0) {
                result.add(Integer.parseInt(value));
            }
        }
        return result;
    }

}
